/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.util.UUID;

import javax.management.*;

import org.apache.commons.logging.Log;
import org.json.JSONObject;

import com.openexchange.management.ManagementException;
import com.openexchange.management.ManagementService;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.configuration.*;
import com.openexchange.usm.mapping.ObjectIdMappingService;
import com.openexchange.usm.ox_event.OXEventListener;
import com.openexchange.usm.ox_event.OXEventManager;
import com.openexchange.usm.ox_json.OXJSONAccess;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.sync.IncrementalContentSyncer;
import com.openexchange.usm.session.sync.SlowContentSyncer;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.uuid.OXObjectID;
import com.openexchange.usm.uuid.UUIDMappingService;
import com.openexchange.usm.uuid.exceptions.OXObjectAlreadyMappedException;
import com.openexchange.usm.uuid.exceptions.UUIDAlreadyMappedException;

public class SessionManagerImpl implements SessionManager, OXEventListener {
	// Stores interval after which an automatic access check is performed.
	// LATER we might want to make this constant configurable
	private final static long ACCESS_CHECK_INTERVAL = 300000L;

	private OXJSONAccess _oxAjaxAccess;

	private Log _journal;

	private ContentTypeManager _contentTypeManager;

	private FolderContentType _folderContentType;

	private SessionStorage _sessionStorage;

	private OXEventManager _oxEventManager;

	private ConfigurationManager _configurationManager;

	private DatabaseAccess _databaseAccess;

	private ObjectIdMappingService _mappingService;

	private UUIDMappingService _uuidService;

	private ManagementService _managementService;

	private ObjectName _jmxObjectName;

	private String _usmInterface;

	// A value < 0 indicates percent, >= 0 an absolute value
	private int _waitForChangesEmailPullDelay;

	private int _waitForChangesEmailMinPullDelay = -1;

	private int _maxSyncStatesInDB = ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT;

	private boolean _keepLastSentSyncStateInDB = ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_DEFAULT;

	private SlowContentSyncer _slowSyncer;

	private IncrementalContentSyncer _incrementalSyncer;

	public void activate(Log log, OXJSONAccess ajaxAccess, ContentTypeManager contentTypeManager,
			FolderContentType folderContentType, OXEventManager oxEventManager,
			ConfigurationManager configurationManager, DatabaseAccess dbAccess, ObjectIdMappingService mappingService,
			UUIDMappingService uuidService) {
		_journal = log;
		_oxAjaxAccess = ajaxAccess;
		_contentTypeManager = contentTypeManager;
		_folderContentType = folderContentType;
		_oxEventManager = oxEventManager;
		_configurationManager = configurationManager;
		_databaseAccess = dbAccess;
		_mappingService = mappingService;
		_uuidService = uuidService;
		int retryCount;
		try {
			retryCount = _configurationManager.getProperty(
					ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_PROPERTY,
					ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT, false);
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR,
					"Configuration for Sync retries is invalid", e);
		}
		_slowSyncer = new SlowContentSyncer(this, retryCount);
		_incrementalSyncer = new IncrementalContentSyncer(this, retryCount);
		_waitForChangesEmailPullDelay = readEmailPullDelayProperty();
		_waitForChangesEmailMinPullDelay = readEmailMinPullDelayProperty();
		try {
			_maxSyncStatesInDB = _configurationManager.getProperty(
					ConfigurationProperties.SYNC_MAX_STATES_IN_DB_PROPERTY,
					ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT, false);
			if (_maxSyncStatesInDB < 2)
				throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_5,
						"Configuration for max_states_in_db must be greater or equal to 2");
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_6,
					"Configuration for max_states_in_db is invalid", e);
		}
		try {
			_keepLastSentSyncStateInDB = _configurationManager.getProperty(
					ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_PROPERTY,
					ConfigurationProperties.SYNC_KEEP_LAST_SENT_STATE_IN_DB_DEFAULT, false);
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_7,
					"Configuration for keep_last_state_in_db is invalid", e);
		}

		try {
			_usmInterface = _configurationManager.getProperty(ConfigurationProperties.USM_INTERFACE_CONFIG_PROPERTY,
					ConfigurationProperties.USM_INTERFACE_CONFIG_DEFAULT, true);
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_8,
					"Configuration for usm interface module is invalid", e);
		}

		_sessionStorage = new SessionStorage();
		_oxEventManager.addEventListener(this);

		_journal.info("USM SessionManager activated");
		_journal.info("Open-Xchange USM Version " + USMVersion.VERSION + " (Build " + USMVersion.BUILD_NUMBER
				+ ") successfully started");
	}

	public UUID getContextUUID(int cid) throws DatabaseAccessException, USMSQLException {
		return _uuidService.getContextUUID(cid);
	}

	public void deactivate() {
		_oxEventManager.removeEventListener(this);
		_sessionStorage.shutdown(_oxAjaxAccess);
		_configurationManager = null;
		_oxEventManager = null;
		_sessionStorage = null;
		_contentTypeManager = null;
		_oxAjaxAccess = null;
		_databaseAccess = null;
		_mappingService = null;
		_waitForChangesEmailPullDelay = 0;
		_waitForChangesEmailMinPullDelay = -1;
		_slowSyncer = null;
		_incrementalSyncer = null;
		_journal.info("USM SessionManager deactivated");
		_journal = null;
	}

	public Session getSession(String user, String password, String protocol, String device) throws USMException {
		return getSession(user, password, protocol, device, null);
	}

	public Session getSession(String user, String password, String protocol, String device,
			SessionInitializer initializer) throws USMException {
		SessionID id = new SessionID(user, protocol, device);
		SessionImpl session = _sessionStorage.getSession(id);
		if (session != null) {
			if (!session.getPassword().equals(password)) {
				session.setPassword(password);
				_oxAjaxAccess.login(session);
			}
			try {
				recheckUSMAccess(initializer, session);
				return session;
			} catch (USMAccessDeniedException e) {
				_sessionStorage.removeSession(session);
				throw e;
			}
		}
		session = new SessionImpl(this, user, password, protocol, device);
		_oxAjaxAccess.login(session);
		JSONObject configuration = _oxAjaxAccess.getConfiguration(session);
		checkUSMAccess(session, configuration);
		session.setLastAccessCheck(System.currentTimeMillis());
		session.initialize(initializer, configuration);
		_sessionStorage.storeSession(id, session);
		return session;
	}

	private void recheckUSMAccess(SessionInitializer initializer, SessionImpl session) throws USMException {
		long now = System.currentTimeMillis();
		if (session.getLastAccessCheck() < now - ACCESS_CHECK_INTERVAL) {
			checkUSMAccess(session, null);
			initializer.checkAccess(session);
			session.setLastAccessCheck(now);
		}
	}

	private void checkUSMAccess(SessionImpl session, JSONObject configuration) throws USMAccessDeniedException,
			AuthenticationFailedException, OXCommunicationException {
		if (_usmInterface == null || _usmInterface.length() == 0) {
			JSONObject usmModule = getUSMModule(session, configuration, "com.openexchange.usm");
			if (usmModule != null && usmModule.optBoolean("module"))
				return;
		} else {
			JSONObject oxInterfaces = getUSMModule(session, configuration, "interfaces");
			if (oxInterfaces != null && oxInterfaces.optBoolean(_usmInterface))
				return;
		}
		_oxAjaxAccess.logout(session);
		throw new USMAccessDeniedException(USMSessionManagementErrorCodes.USM_ACCESS_DISABLED,
				"USM access disabled for user");
	}

	private JSONObject getUSMModule(SessionImpl session, JSONObject configuration, String module)
			throws AuthenticationFailedException {
		if (configuration != null)
			return JSONToolkit.getJSONObject(configuration, "modules", module);
		try {
			return _oxAjaxAccess.getConfiguration(session, "modules", module);
		} catch (OXCommunicationException e) {
			return null;
		}
	}

	public OXJSONAccess getOXAjaxAccess() {
		return _oxAjaxAccess;
	}

	public Log getJournal() {
		return _journal;
	}

	public ContentTypeManager getContentTypeManager() {
		return _contentTypeManager;
	}

	public FolderContentType getFolderContentType() {
		return _folderContentType;
	}

	public DatabaseAccess getDatabaseAccess() {
		return _databaseAccess;
	}

	public ObjectIdMappingService getMappingService() {
		return _mappingService;
	}

	public int getMaxSyncStatesInDB() {
		return _maxSyncStatesInDB;
	}

	public boolean isKeepLastSentSyncStateInDB() {
		return _keepLastSentSyncStateInDB;
	}

	public void removeSessionFromStorage(SessionImpl session) {
		_sessionStorage.removeSession(session);
		_oxAjaxAccess.logout(session);
	}

	// --- Interface OXEventListener ---

	public void folderChanged(int contextId, int userId, String folderId, long timestamp) {
		SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
		if (sessions != null) {
			if (timestamp == 0)
				timestamp = System.currentTimeMillis();
			for (SessionID session : sessions) {
				SessionImpl s = _sessionStorage.getSession(session);
				if (s != null)
					s.foldersChanged(timestamp);
			}
		}
	}

	public void folderContentChanged(int contextId, int userId, String folderId, long timestamp) {
		SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
		if (sessions != null) {
			if (timestamp == 0)
				timestamp = System.currentTimeMillis();
			for (SessionID session : sessions) {
				SessionImpl s = _sessionStorage.getSession(session);
				if (s != null) {
					if (folderId == null)
						s.folderContentChanged(timestamp);
					else
						s.folderContentChanged(timestamp, folderId);
				}
			}
		}
	}

	public boolean isEmailPullEnabled() {
		return _waitForChangesEmailMinPullDelay >= 0;
	}

	public long computePullTime(long start, long end) {
		long result = start + _waitForChangesEmailMinPullDelay;
		// If disabled or minimum wait time greater than interval, return -1 to indicate no email pull
		if (_waitForChangesEmailMinPullDelay < 0 || result > end)
			return -1L;
		// Compute pulltime as percentage of interval or absolute delay from start
		long pullTime = (_waitForChangesEmailPullDelay < 0) ? (start - (end - start) * _waitForChangesEmailPullDelay
				/ 100L) : (start + _waitForChangesEmailPullDelay);
		// return at least minimum wait time, at most normal end time
		return Math.min(Math.max(result, pullTime), end);
	}

	// internal methods

	private int readEmailMinPullDelayProperty() {
		try {
			return _configurationManager.getProperty(ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY,
					ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_DEFAULT, true);
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_4,
					"Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY, e);
		}
	}

	private int readEmailPullDelayProperty() {
		try {
			String pullValue = _configurationManager.getProperty(
					ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY,
					ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_DEFAULT, true);
			if (pullValue.endsWith("%"))
				return -Integer.parseInt(pullValue.substring(0, pullValue.length() - 1));
			return Integer.parseInt(pullValue);
		} catch (NumberFormatException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_2,
					"Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY, e);
		} catch (USMInvalidConfigurationException e) {
			throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_3,
					"Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY, e);
		}
	}

	public void storeUUID(DataObject o) throws DatabaseAccessException, USMSQLException, UUIDAlreadyMappedException,
			OXObjectAlreadyMappedException {
		int contentTypeCode = o.getContentType().getCode();
		if (contentTypeCode != DefaultContentTypes.MAIL_CODE) {
			try {
				int objectId = Integer.parseInt(o.getID());
				_uuidService.storeMapping(o.getUUID(), new OXObjectID(o.getSession().getContextId(), contentTypeCode,
						objectId));
			} catch (NumberFormatException ignored) {
				// For all cases in which we have no integer OX object id (e.g. mail folders), only store UUIDs locally in sync states
			}
		}
	}

	public void storeUUID(int cid, ContentType contentType, int objectId, UUID uuid) throws DatabaseAccessException,
			USMSQLException, UUIDAlreadyMappedException, OXObjectAlreadyMappedException {
		OXObjectID oxObjectId = new OXObjectID(cid, contentType.getCode(), objectId);
		_uuidService.storeMapping(uuid, oxObjectId);
	}

	//	public void storeOrRemapUUID(int cid, ContentType contentType, int objectId, UUID uuid)
	//			throws DatabaseAccessException, USMSQLException, UUIDAlreadyMappedException, OXObjectAlreadyMappedException {
	//		OXObjectID oxObjectId = new OXObjectID(cid, contentType.getCode(), objectId);
	//		_uuidService.storeMapping(uuid, oxObjectId); //TODO: implement method in the uuid service
	//	}

	public DataObjectSet insertStoredUUID(DataObject o, DataObjectSet oldData, DataObjectSet newestCachedData)
			throws DatabaseAccessException, USMSQLException {
		if (oldData != null) {
			DataObject oldObject = oldData.get(o.getID());
			if (oldObject != null) {
				o.setUUID(oldObject.getUUID());
				return newestCachedData;
			}
		}
		int contentTypeCode = o.getContentType().getCode();
		switch (contentTypeCode) {
			case DefaultContentTypes.FOLDER_CODE:
				if (newestCachedData == null)
					newestCachedData = new DataObjectSet(o.getSession().getCachedFolders());
				DataObject cachedFolder = newestCachedData.get(o.getID());
				if (cachedFolder != null)
					o.setUUID(cachedFolder.getUUID());
				break;
			case DefaultContentTypes.MAIL_CODE:
				if (newestCachedData == null)
					newestCachedData = new DataObjectSet(o.getSession().getCachedFolderElements(o.getParentFolderID(),
							o.getContentType()));
				DataObject cachedMail = newestCachedData.get(o.getID());
				if (cachedMail != null)
					o.setUUID(cachedMail.getUUID());
				break;
			default:
				try {
					int objectId = Integer.parseInt(o.getID());
					o.setUUID(_uuidService.getUUID(o.getSession().getContextId(), contentTypeCode, objectId));
				} catch (NumberFormatException ignored) {
				}
				break;
		}
		return newestCachedData;
	}

	public UUID getUUID(Session session, int contentTypeCode, int objectId) throws DatabaseAccessException,
			USMSQLException {
		return _uuidService.getUUID(session.getContextId(), contentTypeCode, objectId);
	}

	public int getMappedObject(Session session, int contentTypeCode, UUID uuid) throws DatabaseAccessException,
			USMSQLException {
		OXObjectID oxObjectId = _uuidService.getMappedObject(session.getContextId(), uuid);
		if (oxObjectId != null && oxObjectId.getContentType() == contentTypeCode)
			return oxObjectId.getObjectId();
		else
			return 0;
	}

	public void objectDeleted(int contextID, int objectID, int contentType) {
		OXObjectID oxObjectId = new OXObjectID(contextID, contentType, objectID);
		try {
			_uuidService.removeMapping(contextID, oxObjectId);
		} catch (DatabaseAccessException e) {
			_journal.error("Database could not be accessed to delete UUID mapping for " + oxObjectId);
		} catch (USMSQLException e) {
			_journal.error("SQL error occurred while deleting UUID mapping for " + oxObjectId);
		}
	}

	public synchronized void setManagementService(ManagementService managementService) {
		if (_managementService != null) {
			if (_jmxObjectName != null) {
				try {
					_managementService.unregisterMBean(_jmxObjectName);
				} catch (final ManagementException e) {
					_journal.error(e.getMessage(), e);
				} finally {
					_jmxObjectName = null;
					_managementService = null;
				}
			}
		}
		_managementService = managementService;
		if (managementService != null) {
			try {
				_jmxObjectName = getObjectName(USMSessionInformation.class.getName(),
						USMSessionInformationMBean.CACHE_DOMAIN);
				_managementService.registerMBean(_jmxObjectName, new USMSessionInformation(this));
			} catch (final MalformedObjectNameException e) {
				_journal.error(e.getMessage(), e);
			} catch (final ManagementException e) {
				_journal.error(e.getMessage(), e);
			} catch (NotCompliantMBeanException e) {
				_journal.error(e.getMessage(), e);
			}
		}
	}

	/**
	 * Creates an appropriate instance of {@link ObjectName} from specified class name and domain name.
	 * 
	 * @param className The class name to use as object name
	 * @param domain The domain name
	 * @return An appropriate instance of {@link ObjectName}
	 * @throws MalformedObjectNameException If instantiation of {@link ObjectName} fails
	 */
	private static ObjectName getObjectName(final String className, final String domain)
			throws MalformedObjectNameException {
		final int pos = className.lastIndexOf('.');
		return new ObjectName(domain, "name", pos == -1 ? className : className.substring(pos + 1));
	}

	public SessionStorage getSessionStorage() {
		return _sessionStorage;
	}

	public void setWaitForChangesEmailPullDelay(int waitForChangesEmailPullDelay) {
		_waitForChangesEmailPullDelay = waitForChangesEmailPullDelay;
	}

	public int getWaitForChangesEmailPullDelay() {
		return _waitForChangesEmailPullDelay;
	}

	public void setWaitForChangesEmailMinPullDelay(int waitForChangesEmailMinPullDelay) {
		_waitForChangesEmailMinPullDelay = waitForChangesEmailMinPullDelay;
	}

	public int getWaitForChangesEmailMinPullDelay() {
		return _waitForChangesEmailMinPullDelay;
	}

	public void setKeepLastSentSyncStateInDB(boolean set) {
		_keepLastSentSyncStateInDB = set;
	}

	public void setMaxSyncStatesInDB(int states) {
		_maxSyncStatesInDB = states;
	}

	public SlowContentSyncer getSlowSyncer() {
		return _slowSyncer;
	}

	public IncrementalContentSyncer getIncrementalSyncer() {
		return _incrementalSyncer;
	}
}
