/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.mapping.FolderIdMapping;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.FolderImpl;
import com.openexchange.usm.session.sync.*;

public class SessionImpl implements Session {

	private static final boolean MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION = false;

	private static final DataObject[] EMPTY_DATA_OBJECT_ARRAY = new DataObject[0];

	private final static String FOLDER_HIERARCHY_ID = "";

	private final static int FOLDER_ID_LENGTH_LIMIT = 39;

	private final PersistentSessionData _persistentData;

	private final SessionManagerImpl _sessionManager;

	private final String _user;

	private final String _protocol;

	private final String _device;

	private final CopyOnWriteArrayList<SessionChangeListener> _changeListeners = new CopyOnWriteArrayList<SessionChangeListener>();

	private final Map<String, Long> _syncedFolderContent = new ConcurrentHashMap<String, Long>();

	private final Map<String, Object> _customProperties = new ConcurrentHashMap<String, Object>();

	private final Object _waitObject = new Object(); // Used to wait on folder changes and to notify about folder changes

	private int _contextID;

	private int _uniqueSessionID;

	private int _userIdentifier;

	private String _password;

	private DataObjectFilter _syncServerObjectFilter;

	private DataObjectStorage _dataObjectCache;

	private TimeZone _userTimeZone;

	private String[] _rootFolders = new String[0];

	private long _startDate = 0L;

	private long _endDate = Long.MAX_VALUE;

	private int _mailLimit = 0;

	private long _lastFolderSync = 0L;

	private long _lastFolderChange = 0L;

	private FolderIdMapping _folderIdMapping = null;

	private DataObjectFilter _lastEmailFilter = null;

	private Map<String, Folder> _dummyFolderMap = new ConcurrentHashMap<String, Folder>();

	private long _lastAccessCheck = 0L;

	public SessionImpl(SessionManagerImpl sessionManager, String user, String password, String protocol, String device) {
		_sessionManager = sessionManager;
		_user = user;
		_password = password;
		_protocol = protocol;
		_device = device;
		_persistentData = new PersistentSessionData(this);
		_dataObjectCache = new DataObjectStorage(this);
	}

	public void initialize(SessionInitializer initializer, JSONObject configuration) throws USMException {
		try {
			_contextID = configuration.getInt("context_id");
		} catch (JSONException e) {
			_sessionManager.getJournal().error(this + " No context_id found in initialization", e);
		}
		try {
			_userIdentifier = configuration.getInt("identifier");
		} catch (JSONException e) {
			_sessionManager.getJournal().error(this + " No identifier found in initialization", e);
		}
		try {
			String timeZone = configuration.getString("timezone");
			_userTimeZone = TimeZone.getTimeZone(timeZone);
			if (!timeZone.equals(_userTimeZone.getID()))
				_sessionManager.getJournal().error(
						this + " Can't resolve TimeZone " + timeZone + ", using " + _userTimeZone.getID());
		} catch (JSONException e) {
			_sessionManager.getJournal().error(this + " No identifier found in initialization", e);
		}
		_persistentData.initialize(initializer, configuration);
	}

	public String getUser() {
		return _user;
	}

	public String getPassword() {
		return _password;
	}

	public void setPassword(String password) {
		_password = password;
	}

	public String getDevice() {
		return _device;
	}

	public String getProtocol() {
		return _protocol;
	}

	public int getContextId() {
		return _contextID;
	}

	public int getSessionId() {
		return _uniqueSessionID;
	}

	public void setUniqueSessionID(int uniqueSessionID) {
		_uniqueSessionID = uniqueSessionID;
	}

	public int getUserIdentifier() {
		return _userIdentifier;
	}

	public String getPersistentField(String name) {
		return _persistentData.getField(name);
	}

	public void setPersistentField(String name, String value) throws USMSQLException, DatabaseAccessException {
		_persistentData.setField(name, value);
	}

	public Map<String, String> getPersistentFields() {
		return _persistentData.getPersistentFields();
	}

	public ConflictResolution getConflictResolution() {
		return _persistentData.getConflictResolution();
	}

	private ConflictResolution getConflictResolution(ConflictResolution conflictResolution) {
		return (conflictResolution == null) ? getConflictResolution() : conflictResolution;
	}

	public void setConflictResolution(ConflictResolution resolution) throws DatabaseAccessException, USMSQLException {
		_persistentData.setConflictResolution(resolution);
	}

	public void setContentTypeFilter(ContentType... usedContentTypes) throws USMSQLException, DatabaseAccessException {
		_persistentData.setContentTypeFilter(usedContentTypes);
	}

	public void setFieldFilter(ContentType contentType, String... fieldsOfInterest) throws USMSQLException,
			DatabaseAccessException {
		setFieldFilter(contentType, generateFieldBitSet(contentType, fieldsOfInterest));
	}

	public void setFieldFilter(ContentType contentType, BitSet fieldsOfInterest) throws DatabaseAccessException,
			USMSQLException {
		_persistentData.setFieldFilter(contentType, fieldsOfInterest);
	}

	public void setRootFolders(String... folderIds) {
		_rootFolders = folderIds;
	}

	public SessionManagerImpl getSessionManager() {
		return _sessionManager;
	}

	public EncapsulatedConnection getWritableDBConnection() throws DatabaseAccessException {
		return _sessionManager.getDatabaseAccess().getWritable(_contextID);
	}

	public EncapsulatedConnection getReadOnlyDBConnection() throws DatabaseAccessException {
		return _sessionManager.getDatabaseAccess().getReadOnly(_contextID);
	}

	public BitSet getFieldFilter(ContentType contentType) {
		return _persistentData.getFieldFilter(contentType);
	}

	@Override
	public String toString() {
		return String.valueOf(_contextID) + ':' + String.valueOf(_userIdentifier) + ':' + _user + ':' + _protocol + ':'
				+ _device;
	}

	public String getDescription() {
		return "CID: " + _contextID + ", ID: " + _userIdentifier + ", User: " + _user + ", Protocol: " + _protocol
				+ ", Device: " + _device;
	}

	public SyncResult syncChangesWithServer(String folderId, long timestamp, int resultLimit, DataObjectFilter filter,
			boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException {
		Folder folder = findFolder(folderId);
		ContentType contentType = folder.getElementsContentType();
		if (contentType == null)
			return SyncResult.EMPTY_RESULT;
		if (_sessionManager.isEmailPullEnabled() && contentType.getID().equals(DefaultContentTypes.MAIL_ID))
			_lastEmailFilter = filter;
		try {
			_syncedFolderContent.put(folderId, 0L);
			return _sessionManager.getIncrementalSyncer().syncChangesWithServer(
					createFolderElementsStorage(folder, filter, storeResult, timestamp),
					getConflictResolution(conflictResolution), _sessionManager.getJournal(),
					getCachedFolderElements(folderId, contentType, timestamp), elements, timestamp, resultLimit);
		} catch (USMException e) {
			_syncedFolderContent.remove(folderId);
			throw e;
		}
	}

	public SyncResult syncWithServer(String folderId, int limit, DataObjectFilter filter, boolean storeResult,
			ConflictResolution conflictResolution, DataObject... elements) throws USMException {
		Folder folder = findFolder(folderId);
		if (folder.getElementsContentType() == null)
			return SyncResult.EMPTY_RESULT;
		if (_sessionManager.isEmailPullEnabled()
				&& DefaultContentTypes.MAIL_ID.equals(folder.getElementsContentTypeID()))
			_lastEmailFilter = filter;
		try {
			_syncedFolderContent.put(folderId, 0L);
			return _sessionManager.getSlowSyncer().syncWithServer(
					createFolderElementsStorage(folder, filter, storeResult, 0L),
					getConflictResolution(conflictResolution), _sessionManager.getJournal(), limit, elements);
		} catch (USMException e) {
			_syncedFolderContent.remove(folderId);
			throw e;
		}
	}

	private FolderElementsStorage createFolderElementsStorage(Folder folder, DataObjectFilter filter,
			boolean storeResult, long oldTimestamp) throws USMException {
		ContentType contentType = folder.getElementsContentType();
		return new FolderElementsStorage(this, folder, getValidFilter(contentType), (filter != null) ? filter
				: _syncServerObjectFilter, storeResult, oldTimestamp);
	}

	public BitSet getValidFilter(ContentType elementsContentType) {
		if (elementsContentType == null)
			throw new IllegalArgumentException("No ContentType set");
		BitSet filter = _persistentData.getFieldFilter(elementsContentType);
		if (filter == null)
			throw new IllegalArgumentException("ContentType " + elementsContentType.getID()
					+ " excluded from synchronization");
		return filter;
	}

	public SyncResult syncFolderChangesWithServer(long timestamp, int resultLimit, DataObjectFilter filter,
			boolean storeResult, ConflictResolution conflictResolution, Folder... folderChanges) throws USMException {
		SyncResult result = _sessionManager.getIncrementalSyncer().syncChangesWithServer(
				createFolderHierachyStorage(filter, storeResult, timestamp), getConflictResolution(conflictResolution),
				_sessionManager.getJournal(), getCachedFolders(timestamp), folderChanges, timestamp, resultLimit);
		return updateSynchronizationStateOfFolders(result);
	}

	public SyncResult syncFoldersWithServer(int limit, DataObjectFilter filter, boolean storeResult,
			ConflictResolution conflictResolution, Folder... clientData) throws USMException {
		SyncResult result = _sessionManager.getSlowSyncer().syncWithServer(
				createFolderHierachyStorage(filter, storeResult, 0L), getConflictResolution(conflictResolution),
				_sessionManager.getJournal(), limit, clientData);
		return updateSynchronizationStateOfFolders(result);
	}

	private SyncResult updateSynchronizationStateOfFolders(SyncResult result) {
		if (MARK_CHANGED_FOLDERS_AS_NEEDS_SYNCHRONIZATION) {
			for (DataObject object : result.getChanges()) {
				Folder f = (Folder) object;
				_syncedFolderContent.remove(f.getID());
			}
		}
		for (DataObject object : result.getNewState()) {
			Folder f = (Folder) object;
			if (f.getElementsContentType() == null)
				_syncedFolderContent.put(f.getID(), f.getTimestamp());
		}
		return result;
	}

	private FolderHierarchyStorage createFolderHierachyStorage(DataObjectFilter filter, boolean storeResult,
			long oldTimestamp) {
		return new FolderHierarchyStorage(this, filter == null ? _syncServerObjectFilter : filter, storeResult,
				oldTimestamp, _rootFolders);
	}

	public Folder[] getCachedFolders() throws DatabaseAccessException, USMSQLException {
		return getCachedFolders(_dataObjectCache.get(FOLDER_HIERARCHY_ID), 0L);
	}

	public Folder[] getCachedFolders(long timestamp) throws DatabaseAccessException, USMSQLException {
		return getCachedFolders(_dataObjectCache.get(FOLDER_HIERARCHY_ID, timestamp), timestamp);
	}

	private Folder[] getCachedFolders(Serializable[][] values, long timestamp) {
		if (values == null)
			return null;
		FolderContentType folderContentType = _sessionManager.getFolderContentType();
		Folder[] result = new Folder[values.length];
		for (int i = 0; i < result.length; i++) {
			Folder f = folderContentType.newDataObject(this);
			try {
				f.deserialize(timestamp, values[i]);
			} catch (DeserializationFailedException e) {
				logDeserializationError(FOLDER_HIERARCHY_ID, timestamp, e);
				return null;
			}
			result[i] = completeStructure(f);
		}
		return result;
	}

	public Folder getCachedFolder(String folderID, long timestamp) throws DatabaseAccessException, USMSQLException {
		return getCachedFolder(_dataObjectCache.get(FOLDER_HIERARCHY_ID, timestamp), folderID, timestamp);
	}

	public Folder getCachedFolder(String folderID) throws DatabaseAccessException, USMSQLException {
		return getCachedFolder(_dataObjectCache.get(FOLDER_HIERARCHY_ID), folderID, 0L);
	}

	private Folder getCachedFolder(Serializable[][] values, String folderID, long timestamp) {
		if (values != null) {
			Folder result = _sessionManager.getFolderContentType().newDataObject(this);
			for (int i = 0; i < values.length; i++) {
				try {
					result.deserialize(timestamp, values[i]);
				} catch (DeserializationFailedException e) {
					logDeserializationError(folderID, timestamp, e);
					return null;
				}
				if (folderID.equals(result.getID()))
					return completeStructure(result);
			}
		}
		return null;
	}

	public void logDeserializationError(String folderID, long timestamp, DeserializationFailedException e) {
		_sessionManager.getJournal().error(
				this + " Ignoring invalid serialized cache data for " + folderID + '(' + timestamp + "): "
						+ e.getMessage());
	}

	public DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp)
			throws DatabaseAccessException, USMSQLException {
		return getCachedFolderElements(folderId, contentType, timestamp, _dataObjectCache.get(
				getShortFolderID(folderId), timestamp));
	}

	public DataObject[] getCachedFolderElements(String folderId, ContentType contentType)
			throws DatabaseAccessException, USMSQLException {
		return getCachedFolderElements(folderId, contentType, 0L, _dataObjectCache.get(getShortFolderID(folderId)));
	}

	private DataObject[] getCachedFolderElements(String folderId, ContentType contentType, long timestamp,
			Serializable[][] values) {
		if (contentType == null)
			return EMPTY_DATA_OBJECT_ARRAY;
		if (values == null)
			return null;
		DataObject[] result = new DataObject[values.length];
		for (int i = 0; i < result.length; i++) {
			DataObject o = contentType.newDataObject(this);
			try {
				o.deserialize(timestamp, values[i]);
			} catch (DeserializationFailedException e) {
				logDeserializationError(folderId, timestamp, e);
				return null;
			}
			result[i] = o;
		}
		return result;
	}

	public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID, long timestamp)
			throws DatabaseAccessException, USMSQLException {
		return getCachedFolderElement(folderId, contentType, objectID, _dataObjectCache.get(getShortFolderID(folderId),
				timestamp), timestamp);
	}

	public DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID)
			throws DatabaseAccessException, USMSQLException {
		return getCachedFolderElement(folderId, contentType, objectID,
				_dataObjectCache.get(getShortFolderID(folderId)), 0L);
	}

	private DataObject getCachedFolderElement(String folderId, ContentType contentType, String objectID,
			Serializable[][] values, long timestamp) {
		if (values != null) {
			DataObject result = contentType.newDataObject(this);
			for (int i = 0; i < values.length; i++) {
				try {
					result.deserialize(timestamp, values[i]);
				} catch (DeserializationFailedException e) {
					logDeserializationError(folderId, timestamp, e);
					return null;
				}
				if (objectID.equals(result.getID()))
					return result;
			}
		}
		return null;
	}

	public long storeCachedFolders(long oldTimestamp, long timestamp, DataObjectSet objects, boolean updatesPending)
			throws DatabaseAccessException, USMSQLException {
		long result = internalStoreCachedDataObjects(FOLDER_HIERARCHY_ID, oldTimestamp, timestamp, objects);
		if (!updatesPending)
			_lastFolderSync = result;
		int size = objects.size();
		String[] cacheIDs = new String[size + 1 + _dummyFolderMap.size()];
		String[] ids = new String[objects.size()];
		int i = 0;
		for (DataObject o : objects) {
			cacheIDs[i] = getShortFolderID(o.getID());
			ids[i++] = o.getID();
		}
		cacheIDs[i++] = FOLDER_HIERARCHY_ID;
		for (String folderID : _dummyFolderMap.keySet()) {
			cacheIDs[i++] = getShortFolderID(folderID);
		}
		_dataObjectCache.retain(cacheIDs);
		if (_folderIdMapping != null)
			_folderIdMapping.retain(ids);
		return result;
	}

	public long storeCachedDataObjects(String folderID, long oldTimestamp, long timestamp, DataObjectSet objects,
			boolean updatesPending) throws DatabaseAccessException, USMSQLException {
		long result = internalStoreCachedDataObjects(getShortFolderID(folderID), oldTimestamp, timestamp, objects);
		if (updatesPending)
			_syncedFolderContent.remove(folderID);
		else if (_syncedFolderContent.containsKey(folderID))
			_syncedFolderContent.put(folderID, result);
		return result;
	}

	private long internalStoreCachedDataObjects(String folderID, long oldTimestamp, long timestamp,
			DataObjectSet objects) throws DatabaseAccessException, USMSQLException {
		if (!_sessionManager.isKeepLastSentSyncStateInDB())
			oldTimestamp = 0L;
		long resultingTimestamp = _dataObjectCache.put(folderID, timestamp, serializeObjects(objects), oldTimestamp);
		if (resultingTimestamp != timestamp) {
			for (DataObject o : objects) {
				o.setTimestamp(resultingTimestamp);
				o.commitChanges();
			}
		}
		return resultingTimestamp;
	}

	public DataObject readDataObject(String folderID, String objectID, String... fields) throws USMException {
		Folder folder = findFolder(folderID);
		return readDataObject(folder, objectID, generateFieldBitSet(folder.getElementsContentType(), fields));
	}

	public DataObject readDataObject(String folderID, String objectID, BitSet fields) throws USMException {
		return readDataObject(findFolder(folderID), objectID, fields);
	}

	private Serializable[][] serializeObjects(DataObjectSet objects) {
		Serializable[][] data = new Serializable[objects.size()][];
		int i = 0;
		for (DataObject o : objects)
			data[i++] = o.serialize();
		return data;
	}

	private Folder completeStructure(Folder f) {
		f.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(f.getElementsContentTypeID()));
		return f;
	}

	private DataObject readDataObject(Folder f, String objectID, BitSet fields) throws USMException {
		ContentType contentType = f.getElementsContentType();
		DataObject object = contentType.newDataObject(this);
		object.setID(objectID);
		object.setParentFolder(f);
		contentType.getTransferHandler().readDataObject(object, fields);
		return object;
	}

	public Map<DataObject, USMException> updateDataObjects(DataObject... changedObjects) throws USMException {
		Map<DataObject, USMException> resultMap = new HashMap<DataObject, USMException>();

		// First, split between folder changes and element changes within folders
		List<DataObject> elementChanges = new ArrayList<DataObject>();
		List<DataObject> folderChanges = new ArrayList<DataObject>();
		for (DataObject o : changedObjects) {
			o.setFailed(true);
			if (o.getContentType().getID().equals(DefaultContentTypes.FOLDER_ID))
				folderChanges.add(o);
			else
				elementChanges.add(o);
		}
		if (!folderChanges.isEmpty()) {
			// perform all folder changes, update timestamp if possible
			long timestamp = folderChanges.get(0).getTimestamp();
			DataObjectSet newFolderHierarchy = new DataObjectSet();
			long newTimestamp = checkTimestampAndUpdate(newFolderHierarchy, folderChanges, timestamp,
					getCachedFolders(timestamp), "All Folders must have the same timestamp", resultMap);
			if (newTimestamp != 0) {
				long modifiedTimestamp = storeCachedFolders(timestamp, newTimestamp, newFolderHierarchy, true);
				for (DataObject folder : folderChanges)
					folder.setTimestamp(modifiedTimestamp);
			}
		}
		// group elements by folder, operate on each folder separately
		while (!elementChanges.isEmpty()) {
			DataObject first = elementChanges.remove(elementChanges.size() - 1);
			long timestamp = first.getTimestamp();
			List<DataObject> currentFolderChanges = new ArrayList<DataObject>();
			currentFolderChanges.add(first);
			String folderId = first.getParentFolderID();
			for (int i = elementChanges.size() - 1; i >= 0; i--) {
				DataObject o = elementChanges.get(i);
				if (o.getParentFolderID().equals(folderId))
					currentFolderChanges.add(elementChanges.remove(i));
			}
			DataObjectSet newElements = new DataObjectSet();
			long newTimestamp = checkTimestampAndUpdate(newElements, currentFolderChanges, timestamp,
					getCachedFolderElements(folderId, first.getContentType(), timestamp),
					"All elements in a Folder must have the same timestamp", resultMap);
			if (newTimestamp != 0) {
				long modifiedTimestamp = storeCachedDataObjects(folderId, timestamp, newTimestamp, newElements, true);
				for (DataObject object : currentFolderChanges)
					object.setTimestamp(modifiedTimestamp);
			}
		}
		return resultMap;
	}

	private long checkTimestampAndUpdate(DataObjectSet resultingServerObjects, List<DataObject> requests,
			long timestamp, DataObject[] oldState, String errorMessage, Map<DataObject, USMException> resultMap)
			throws DatabaseAccessException, USMSQLException {
		for (DataObject o : requests) {
			if (o.getTimestamp() != timestamp) {
				// If at least one element has a timestamp mismatch, fail on all elements in the folder
				for (DataObject o2 : requests) {
					resultMap.put(o2, new TimestampMismatchException(
							USMSessionManagementErrorCodes.TIMESTAMP_MISMATCH_ON_DATAOBJECT_UPDATE, errorMessage));
				}
				return 0;
			}
		}
		ContentType type = requests.get(0).getContentType();
		ContentTypeTransferHandler handler = type.getTransferHandler();
		if (oldState == null) {
			for (DataObject o : requests) {
				callServerUpdatesAccordingToTimstamp(resultMap, handler, o);
			}
			for (DataObject o : requests)
				o.setFailed(false);
			return 0;
		}
		resultingServerObjects.clear();
		List<DataObject> creations = new ArrayList<DataObject>();
		List<DataObject> changes = new ArrayList<DataObject>();
		List<DataObject> deletions = new ArrayList<DataObject>();
		for (DataObject o : oldState)
			resultingServerObjects.add(o.createCopy(true));
		for (DataObject o : requests) {
			fillMapsAccordingToChangedState(resultMap, creations, changes, deletions, o);
		}

		ContentSyncerSupport.executeServerUpdates(_sessionManager, resultingServerObjects, creations, changes,
				deletions, resultMap);

		for (DataObject o : requests)
			o.setFailed(false);
		return ContentSyncerSupport.updateTimestampAndCommitChanges(resultingServerObjects);
	}

	private void fillMapsAccordingToChangedState(Map<DataObject, USMException> resultMap, List<DataObject> creations,
			List<DataObject> changes, List<DataObject> deletions, DataObject o) {
		switch (o.getChangeState()) {
			case CREATED:
				creations.add(o);
				return;
			case DELETED:
				deletions.add(o);
				return;
			case MODIFIED:
				changes.add(o);
				return;
			case UNMODIFIED:
				return;
		}
		resultMap.put(o, new InternalUSMException(USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER1,
				"Illegal ChangeState " + o.getChangeState()));
	}

	private void callServerUpdatesAccordingToTimstamp(Map<DataObject, USMException> resultMap,
			ContentTypeTransferHandler handler, DataObject o) {
		switch (o.getChangeState()) {
			case CREATED:
				try {
					handler.writeNewDataObject(o);
					o.setTimestamp(0);
				} catch (USMException e) {
					resultMap.put(o, e);
				}
				return;
			case DELETED:
				try {
					handler.writeDeletedDataObject(o);
					o.setTimestamp(0);
				} catch (USMException e) {
					resultMap.put(o, e);
				}
				return;
			case MODIFIED:
				try {
					handler.writeUpdatedDataObject(o, o.getTimestamp());
					o.setTimestamp(0);
				} catch (USMException e) {
					resultMap.put(o, e);
				}
				return;
			case UNMODIFIED:
				return;
		}
		resultMap.put(o, new InternalUSMException(USMSessionManagementErrorCodes.ILLEGAL_CHANGE_STATE_NUMBER2,
				"Illegal ChangeState " + o.getChangeState()));
	}

	public BitSet generateFieldBitSet(ContentType contentType, String... fieldsOfInterest) {
		ContentTypeField[] fields = contentType.getFields();
		BitSet fieldSet = new BitSet();
		for (String fieldName : fieldsOfInterest) {
			generateFieldsOfInterestBitSet(contentType, fields, fieldSet, fieldName);
		}
		return fieldSet;
	}

	private void generateFieldsOfInterestBitSet(ContentType contentType, ContentTypeField[] fields, BitSet fieldSet,
			String fieldName) {
		for (int i = 0; i < fields.length; i++) {
			if (fields[i].getFieldName().equals(fieldName)) {
				fieldSet.set(i);
				return;
			}
		}
		throw new IllegalArgumentException("Field " + fieldName + " not provided by ContentType " + contentType);
	}

	public Folder findFolder(String folderID) throws USMException {
		Folder folder = _dummyFolderMap.get(folderID);
		if (folder != null)
			return folder;
		folder = getCachedFolder(folderID);
		if (folder != null)
			return folder;
		FolderContentType type = _sessionManager.getFolderContentType();
		Folder result = type.newDataObject(this);
		result.setID(folderID);
		try {
			type.getTransferHandler().readDataObject(result, getFieldFilter(type));
		} catch (USMException e) {
			throw new FolderNotFoundException(USMSessionManagementErrorCodes.FOLDER_NOT_FOUND, "Folder not found: "
					+ folderID, e);
		}
		result.setElementsContentType(_sessionManager.getContentTypeManager().getContentType(
				result.getElementsContentTypeID()));
		return result;
	}

	public void addChangeListener(SessionChangeListener listener) {
		_changeListeners.add(listener);
	}

	public void removeChangeListener(SessionChangeListener listener) {
		_changeListeners.remove(listener);
	}

	public void foldersChanged(long timestamp) {
		if (_changeListeners.size() > 0) {
			ObjectChanges changes = new ObjectChangesImpl(true);
			for (SessionChangeListener listener : _changeListeners) {
				try {
					listener.changesOccurred(changes);
				} catch (Exception e) {
					_sessionManager.getJournal()
							.error(
									this + " Exception while notifying listener " + listener
											+ " of folder structure change", e);
				}
			}
		}
		_lastFolderChange = timestamp;
		synchronized (_waitObject) {
			_waitObject.notifyAll();
		}
	}

	public void folderContentChanged(long timestamp) {
		try {
			folderContentChanged(timestamp, getAllFolderIDs(getCachedFolders()));
		} catch (DatabaseAccessException e) {
			_sessionManager.getJournal().error(this + " Database access error during OX event notification", e);
		} catch (USMSQLException e) {
			_sessionManager.getJournal().error(this + " SQL error during OX event notification", e);
		}
	}

	public void folderContentChanged(long timestamp, String... folderIds) {
		if (_changeListeners.size() > 0) {
			ObjectChanges changes = new ObjectChangesImpl(false, folderIds);
			for (SessionChangeListener listener : _changeListeners) {
				try {
					listener.changesOccurred(changes);
				} catch (Exception e) {
					_sessionManager.getJournal().error(
							this + " Exception while notifying listener " + listener + " of content change in folders "
									+ Arrays.toString(folderIds), e);
				}
			}
		}
		for (String id : folderIds)
			_syncedFolderContent.remove(id);
		synchronized (_waitObject) {
			_waitObject.notifyAll();
		}
	}

	public ObjectChanges waitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout)
			throws USMException, InterruptedException {
		Log journal = _sessionManager.getJournal();
		if (journal.isDebugEnabled())
			journal.debug(this + " waitforChanges(" + watchFolderChanges + ","
					+ Arrays.toString(contentChangeParentIDsToWatch) + "," + timeout + "), lastFolderChange="
					+ _lastFolderChange + ", lastFolderSync=" + _lastFolderSync);
		ObjectChanges result = waitForChanges(_sessionManager.isEmailPullEnabled(), watchFolderChanges,
				contentChangeParentIDsToWatch, timeout);
		if (journal.isDebugEnabled())
			journal.debug(this + " waitforChanges Result=" + result);
		return result;
	}

	private ObjectChanges waitForChanges(boolean emailPullEnabled, boolean watchFolderChanges,
			String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException {
		Log journal = _sessionManager.getJournal();
		ObjectChanges result;
		Folder[] folders = null;
		if (contentChangeParentIDsToWatch == null) {
			folders = getCachedFolders();
			contentChangeParentIDsToWatch = getAllFolderIDs(folders);
		}
		synchronized (_waitObject) {
			result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
			if (result != null)
				return result;
		}
		emailPullEnabled &= contentChangeParentIDsToWatch.length > 0;
		long start = System.currentTimeMillis();
		long end = start + timeout;
		if (emailPullEnabled) {
			if (folders == null)
				folders = getCachedFolders();
			List<Folder> emailFolders = new ArrayList<Folder>();
			if (folders != null) {
				DataObjectSet folderSet = new DataObjectSet(folders);
				for (String folderId : contentChangeParentIDsToWatch) {
					Folder f = (Folder) folderSet.get(folderId);
					if (f != null && DefaultContentTypes.MAIL_ID.equals(f.getElementsContentTypeID()))
						emailFolders.add(f);
				}
			}
			if (!emailFolders.isEmpty()) {
				long pullTime = _sessionManager.computePullTime(start, end);
				if (pullTime > 0) {
					if (journal.isDebugEnabled())
						journal.debug(this + " Waiting " + (pullTime - start) + "ms until pull timeout");
					result = doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, pullTime);
					if (result != null)
						return result;
					if (journal.isDebugEnabled())
						_sessionManager.getJournal().debug(this + " Performing email pull");
					result = performEmailPull(emailFolders);
					if (result != null)
						return result;
				}
				if (journal.isDebugEnabled())
					journal.debug(this + " Waiting " + (end - System.currentTimeMillis()) + "ms after pull timeout");
			}
		}
		return doWaitForChanges(watchFolderChanges, contentChangeParentIDsToWatch, end);
	}

	private ObjectChanges performEmailPull(List<Folder> emailFolders) throws USMException {
		List<String> changedFolders = new ArrayList<String>();
		for (Folder f : emailFolders) {
			String folderId = f.getID();
			Long timestamp = _syncedFolderContent.get(folderId);
			if (timestamp == null || timestamp == 0L)
				changedFolders.add(folderId);
			else {
				SyncResult syncResult = syncChangesWithServer(folderId, timestamp, NO_LIMIT, _lastEmailFilter, false,
						null);
				if (syncResult.getChanges().length > 0)
					changedFolders.add(folderId);
			}
		}
		return ObjectChangesImpl.create(false, changedFolders);
	}

	private ObjectChanges doWaitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long end)
			throws DatabaseAccessException, USMSQLException, InterruptedException {
		ObjectChanges result = null;
		for (long waitTime = end - System.currentTimeMillis(); waitTime > 0; waitTime = end
				- System.currentTimeMillis()) {
			synchronized (_waitObject) {
				result = extractFolderChanges(watchFolderChanges, contentChangeParentIDsToWatch);
				if (result != null)
					return result;
				_waitObject.wait(waitTime);
			}
		}
		return result;
	}

	private ObjectChanges extractFolderChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch)
			throws DatabaseAccessException, USMSQLException {
		new ArrayList<String>();
		boolean folderStructureChanged = watchFolderChanges && _lastFolderChange >= _lastFolderSync;
		List<String> changedFolders = new ArrayList<String>();
		for (String folderId : contentChangeParentIDsToWatch) {
			if (!_syncedFolderContent.containsKey(folderId))
				changedFolders.add(folderId);
		}
		return ObjectChangesImpl.create(folderStructureChanged, changedFolders);
	}

	private String[] getAllFolderIDs(Folder[] folders) {
		if (folders == null)
			return new String[0];
		String[] folderIds = new String[folders.length];
		for (int i = 0; i < folders.length; i++)
			folderIds[i] = folders[i].getID();
		return folderIds;
	}

	public Object getCustomProperty(String key) {
		return _customProperties.get(key);
	}

	public Object setCustomProperty(String key, Object value) {
		return (value == null) ? _customProperties.remove(key) : _customProperties.put(key, value);
	}

	public DataObjectFilter getSyncServerObjectFilter() {
		return _syncServerObjectFilter;
	}

	public void setSyncServerObjectFilter(DataObjectFilter filter) {
		_syncServerObjectFilter = filter;
	}

	public static void main(String[] args) {
		Thread[] writeThreads = new Thread[100];
		final SessionImpl mySession = new SessionImpl(null, "user", "password", "protocol", "device");
		for (int i = 0; i < writeThreads.length; i++) {
			writeThreads[i] = new Thread("Thread " + i) {

				@Override
				public void run() {
					try {
						for (;;) {
							mySession.foldersChanged(0L);
							Thread.sleep(100L);
							mySession.folderContentChanged(0L, getName());
							Thread.sleep(100L);
						}
					} catch (InterruptedException ignored) {
						// exit on interrupt
					}
				}
			};
			writeThreads[i].start();
		}
		Thread readThread = new Thread("Reader") {

			@Override
			public void run() {
				try {
					for (;;) {
						ObjectChanges result = mySession.waitForChanges(true, null, 1000L);
						System.out.println("Result: " + result);
					}
				} catch (USMException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		};
		readThread.start();
		//		try {
		//			Thread.sleep(10000L);
		//		} catch (InterruptedException e) {
		//		}
		//		for (Thread t : writeThreads)
		//			t.interrupt();
		//		readThread.interrupt();
	}

	public TimeZone getUserTimeZone() {
		return _userTimeZone;
	}

	public long getEndDate() {
		return _endDate;
	}

	public long getStartDate() {
		return _startDate;
	}

	public void setEndDate(long date) {
		_endDate = date;
	}

	public void setStartDate(long date) {
		_startDate = date;
	}

	public int getMailLimit() {
		return _mailLimit;
	}

	public void setMailLimit(int limit) {
		_mailLimit = limit;
	}

	public String getFolderID(String shortFolderID) throws USMSQLException {
		if (shortFolderID.length() > 0) {
			switch (shortFolderID.charAt(0)) {
				case 'D':
					try {
						return getFolderIdMapping().getLongID(Integer.parseInt(shortFolderID.substring(1)));
					} catch (NumberFormatException e) {
					}
					break;
				case 'F':
					return shortFolderID.substring(1);
				default:
					break;
			}
		}
		throw new IllegalArgumentException("Illegal short folder ID " + shortFolderID);
	}

	public String getShortFolderID(String folderID) throws DatabaseAccessException, USMSQLException {
		if (folderID.length() < FOLDER_ID_LENGTH_LIMIT)
			return "F" + folderID;
		return "D" + getFolderIdMapping().getShortID(folderID);
	}

	private FolderIdMapping getFolderIdMapping() {
		if (_folderIdMapping == null)
			_folderIdMapping = _sessionManager.getMappingService().getShortIDStorage(this);
		return _folderIdMapping;
	}

	@Deprecated
	public long storeSyncState(long timestamp, String folderID, DataObject[] objects) throws USMException {
		return storeSyncState(0L, timestamp, folderID, objects);
	}

	public long storeSyncState(long timestampToKeep, long timestamp, String folderID, DataObject[] objects)
			throws USMException {
		DataObjectSet set = new DataObjectSet();
		ContentType t = null;
		for (DataObject o : objects) {
			if (t == null)
				t = o.getContentType();
			else if (!o.getContentType().getID().equals(t.getID()))
				throw new IllegalArgumentException("DataObjects of different ContentTypes");
			if (!folderID.equals(o.getParentFolderID()))
				throw new IllegalArgumentException("DataObjects of different folders");
			set.add(o);
		}
		return storeCachedDataObjects(folderID, timestampToKeep, timestamp, set, false);
	}

	public void endSyncronization() throws USMSQLException, DatabaseAccessException {
		_dataObjectCache.removeAllObjectsForSession();
		_persistentData.removeAllDBFieldsForSession();
		_sessionManager.removeSessionFromStorage(this);
	}

	public void storeUUID(ContentType contentType, int objectId, UUID uuid) throws USMException {
		_sessionManager.storeUUID(_contextID, contentType, objectId, uuid);
	}

	public PersistentSessionData getPersistentSessionData() {
		return _persistentData;
	}

	public UUID getContextUUID() throws DatabaseAccessException, USMSQLException {
		return _sessionManager.getContextUUID(_contextID);
	}

	public void insertStoredUUID(DataObject o) throws DatabaseAccessException, USMSQLException {
		_sessionManager.insertStoredUUID(o, null, null);
	}

	public int getMappedObjectId(ContentType contentType, UUID uuid) throws USMException {
		return _sessionManager.getMappedObject(this, contentType.getCode(), uuid);
	}

	public UUID getUUID(ContentType contentType, int objectId) throws USMException {
		return _sessionManager.getUUID(this, contentType.getCode(), objectId);
	}

	public Folder getDummyContentTypeFolder(ContentType type) {
		Folder folder = _dummyFolderMap.get(type.getID());
		if (folder != null)
			return folder;
		folder = new FolderImpl(this, _sessionManager.getFolderContentType());
		folder.setID(type.getID());
		long timestamp = System.currentTimeMillis();
		folder.setTimestamp(timestamp);
		folder.setElementsContentType(type);
		_dummyFolderMap.put(type.getID(), folder);
		return folder;
	}

	public ContentType[] getContentTypes() throws DatabaseAccessException, USMSQLException {
		return _persistentData.getContentTypes();
	}

	public void remapCachedData(String oldObjectID, String newObjectID) throws DatabaseAccessException, USMSQLException {
		_dataObjectCache.remapStates(oldObjectID, newObjectID);
	}

	public void setLastAccessCheck(long lastAccessCheck) {
		_lastAccessCheck = lastAccessCheck;
	}

	public long getLastAccessCheck() {
		return _lastAccessCheck;
	}

	public JSONObject getOXUserConfiguration(String... path) throws AuthenticationFailedException,
			OXCommunicationException {
		return _sessionManager.getOXAjaxAccess().getConfiguration(this, path);
	}
}
