/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.impl;

import java.io.*;
import java.sql.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import org.apache.commons.logging.Log;

import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.util.Toolkit;

/**
 * DB DataObject storage. 
 * @author ldo
 *
 */
public class DataObjectStorage {
	// This is special profiling code that prints statistics of the average size of the stored folder content in the DB
	//	private static AtomicInteger _numberOfObjects = new AtomicInteger(0);
	//	private static AtomicLong _sumOfObjectSizes = new AtomicLong(0L);
	//
	//	static {
	//		Thread t = new Thread("Debug Statistic Thread") {
	//			@Override
	//			public void run() {
	//				int lastNumber = 0;
	//				SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
	//				for (;;) {
	//					try {
	//						sleep(1000L);
	//						int number = _numberOfObjects.get();
	//						if (number != lastNumber) {
	//							long size = _sumOfObjectSizes.get();
	//							System.out.printf("%s: %6d entries, avg. size %9.3f\n",
	//									format.format(new java.util.Date()), number, size / (double) number);
	//							lastNumber = number;
	//						}
	//					} catch (InterruptedException e) {
	//						break;
	//					}
	//				}
	//			}
	//		};
	//		t.setDaemon(true);
	//		t.start();
	//	}

	private static final int NUMBER_OF_SESSION_ID_FIELDS = 2;

	private static final int NUMBER_OF_OBJECT_ID_FIELDS = 3;

	private final SessionImpl _session;

	private final Map<String, CachedData> _internalCache = new ConcurrentHashMap<String, CachedData>();

	public DataObjectStorage(SessionImpl session) {
		_session = session;
	}

	public void remapStates(String oldObjectID, String newObjectID) throws DatabaseAccessException, USMSQLException {
		CachedData data = _internalCache.get(oldObjectID);
		_internalCache.remove(oldObjectID);
		if (data!= null) _internalCache.put(newObjectID, data);
		//restore the data in the DB
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			String updateStatement = "UPDATE USMDataStorage SET objectId = ? WHERE objectId = ? AND cid = ? AND usmSessionId = ?";
			con = _session.getWritableDBConnection();
			statement = con.prepareStatement(updateStatement);
			statement.setString(1, newObjectID);
			statement.setString(2, oldObjectID);
			statement.setInt(3, _session.getContextId());
			statement.setInt(4, _session.getSessionId());
			if (statement.executeUpdate() == 0) {
				Log journal = getJournal();
				if (journal.isDebugEnabled())
					journal.debug(_session + " No states were remapped for object:  " + oldObjectID);
			}
		}  catch (SQLException e) {
			throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER5, _session
					+ " Error re-storing cache data for " + oldObjectID, e); 
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	public long put(String objectID, long timestamp, Serializable[][] objects, long timestampToKeep)
			throws DatabaseAccessException, USMSQLException {
		CachedData data = getInternalCacheData(objectID);
		boolean shouldKeepTimestamp = timestampToKeep > 0L;
		long newTS = data.store(timestamp, objects, timestampToKeep);
		long oldTS = data.getTimestamp(shouldKeepTimestamp ? 1 : 0);
		Log journal = getJournal();
		if (journal.isDebugEnabled()) {
			if (shouldKeepTimestamp)
				journal.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS + ", keeping "
						+ timestampToKeep + ", removing older than " + oldTS);
			else
				journal.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS
						+ ", removing older than " + oldTS);
		}
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _session.getWritableDBConnection();
			String deleteStatement = shouldKeepTimestamp ? (DataObjectStorageSQLStatements.DELETE
					+ DataObjectStorageSQLStatements.UNIQUE_OBJECT_IDENTIFIER + DataObjectStorageSQLStatements.CACHE_DELETE_CONDITION_COMPLEX)
					: (DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_OBJECT_IDENTIFIER + DataObjectStorageSQLStatements.CACHE_DELETE_CONDITION_SIMPLE);
			statement = con.prepareStatement(deleteStatement);
			addObjectIdentifierToStatement(statement, objectID);
			statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 1, newTS);
			statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 2, oldTS);
			if (shouldKeepTimestamp)
				statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 3, timestampToKeep);
			statement.executeUpdate();
			statement.close();
			statement = con.prepareStatement(DataObjectStorageSQLStatements.FULL_INSERT);
			String element = writeToString(objects);
			addObjectIdentifierToStatement(statement, objectID);
			statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 1, newTS);
			statement.setString(NUMBER_OF_OBJECT_ID_FIELDS + 2, element);
			if (statement.executeUpdate() != 1)
				throw new SQLException("Insert did not produce any results");
		} catch (SQLException e) {
			throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER1, _session
					+ " Error storing cache data for " + objectID, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		return newTS;
	}

	public Serializable[][] get(String objectID, long timestamp) throws DatabaseAccessException, USMSQLException {
		// Check if we have object in cache already
		CachedData data = _internalCache.get(objectID);
		if (data != null) {
			Serializable[][] result = data.get(timestamp);
			if (result != null) // If we have cached information, return it
				return result;
			_internalCache.remove(objectID); // Otherwise, clear cached information so that we will re-read it from DB
		}
		data = getInternalCacheData(objectID);
		return data.get(timestamp);
	}

	/**
	 * Special method that returns the newest cache entry for internal USM methods that do not know of the current
	 * sync key.
	 * @param objectID
	 * @return
	 * @throws DatabaseAccessException
	 * @throws USMSQLException
	 */
	public Serializable[][] get(String objectID) throws DatabaseAccessException, USMSQLException {
		// Check if we have object already in cache
		CachedData data = _internalCache.get(objectID);
		if (data != null) {
			Serializable[][] result = data.get();
			if (result != null) // If we have cached information, return it
				return result;
			_internalCache.remove(objectID); // Otherwise, clear cached information so that we will re-read it from DB
		}
		data = getInternalCacheData(objectID);
		return data.get();
	}

	/**
	 * Removes all DB entries for the given user and device that have object IDs as the ones provided as parameters
	 * @param user
	 * @param device
	 * @param objectIDs
	 * @throws USMSQLException 
	 */
	public void remove(String... objectIDs) throws USMSQLException {
		Log journal = getJournal();
		if (journal.isDebugEnabled())
			journal.debug(_session + " Removing from DB cache " + Arrays.toString(objectIDs));
		for (String string : objectIDs) {
			_internalCache.remove(string);
		}
		executeUpdate("", "Error while deleting cached information for folders", completeComplexStatement(
				DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER
						+ " and ObjectID IN (", objectIDs.length), objectIDs);
	}

	/**
	 * Removes all DB entries for the given user and device that have object IDs other than the ones provided as parameters
	 * @param objectIDs
	 * @throws USMSQLException 
	 */
	public void retain(String... objectIDs) throws USMSQLException {
		Log journal = getJournal();
		if (journal.isDebugEnabled())
			journal.debug(_session + " Removing from DB cache all except " + Arrays.toString(objectIDs));
		for (Iterator<String> i = _internalCache.keySet().iterator(); i.hasNext();) {
			String key = i.next();
			removeFromInternalCacheIfNotInList(i, key, objectIDs);
		}
		executeUpdate("", "Error while deleting cached information for unused folders", completeComplexStatement(
				DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER
						+ " and ObjectID NOT IN (", objectIDs.length), objectIDs);
	}

	/**
	 * Removes all saved states for this session
	 * @throws USMSQLException 
	 */
	public void removeAllObjectsForSession() throws USMSQLException {
		Log journal = getJournal();
		if (journal.isDebugEnabled())
			journal.debug(_session + " Removing from DB cache all elements for session: " + _session.getSessionId());
		executeUpdate("", "Error while deleting cached information for one session",
				DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER);
	}

	private void addSessionIdentifierToStatement(PreparedStatement statement) throws SQLException {
		statement.setInt(1, _session.getContextId());
		statement.setInt(2, _session.getSessionId());
	}

	private void addObjectIdentifierToStatement(PreparedStatement statement, String objectID) throws SQLException {
		addSessionIdentifierToStatement(statement);
		statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, objectID);
	}

	private String completeComplexStatement(String prefix, int length) {
		StringBuilder sb = new StringBuilder(prefix);
		for (int i = 0; i < length; i++) {
			if (i > 0)
				sb.append(',');
			sb.append('?');
		}
		return sb.append(')').toString();
	}

	private void removeFromInternalCacheIfNotInList(Iterator<String> i, String key, String... objectIDs) {
		for (String s : objectIDs) {
			if (key.equals(s))
				return;
		}
		i.remove();
	}

	/**
	 * Execute a SQL statement, the first 2 parameters must be user and session and need not be specified as parameters
	 * to this method. The remaining parameters are filled from the provided values.
	 * 
	 * @param warnMessage
	 * @param errorMessage
	 * @param command
	 * @param parameters
	 * @throws USMSQLException 
	 */
	private void executeUpdate(String warnMessage, String errorMessage, String command, String... parameters)
			throws USMSQLException {
		EncapsulatedConnection con = null;
		try {
			con = _session.getWritableDBConnection();
			executeUpdate(con, warnMessage, errorMessage, command, parameters);
		} catch (DatabaseAccessException e) {
			getJournal().error(String.valueOf(_session) + ' ' + errorMessage, e);
		} finally {
			Toolkit.close(con);
		}
	}

	private void executeUpdate(EncapsulatedConnection con, String warnMessage, String errorMessage, String command,
			String... parameters) throws USMSQLException {
		PreparedStatement statement = null;
		try {
			statement = con.prepareStatement(command);
			addSessionIdentifierToStatement(statement);
			int i = NUMBER_OF_SESSION_ID_FIELDS + 1;
			for (String param : parameters)
				statement.setString(i++, param);
			if (statement.executeUpdate() == 0 && warnMessage.length() > 0)
				getJournal().warn(String.valueOf(_session) + ' ' + warnMessage);
		} catch (SQLException e) {
			getJournal().error(String.valueOf(_session) + ' ' + errorMessage, e);
			throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER3, _session + ": "
					+ errorMessage, e);
		} finally {
			Toolkit.close(statement);
		}
	}

	private CachedData getInternalCacheData(String objectID) throws DatabaseAccessException, USMSQLException {
		CachedData data = _internalCache.get(objectID);
		if (data != null)
			return data;
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		List<CachedDBEntry> dbList = new ArrayList<CachedDBEntry>();
		try {
			con = _session.getReadOnlyDBConnection();
			statement = con.prepareStatement(DataObjectStorageSQLStatements.SELECT_CACHED_DATA
					+ DataObjectStorageSQLStatements.UNIQUE_OBJECT_IDENTIFIER
					+ DataObjectStorageSQLStatements.SELECT_CACHED_DATA_ORDER_BY);
			addObjectIdentifierToStatement(statement, objectID);
			ResultSet result = statement.executeQuery();
			while (result.next()) {
				long timestamp = result.getLong(1);
				Clob element = result.getClob(2);
				dbList.add(new CachedDBEntry(timestamp, element.getSubString(1L, (int) element.length())));
			}
		} catch (SQLException e) {
			String errorMessage = _session + " Error retrieving cached data for " + objectID;
			getJournal().error(errorMessage, e);
			throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER4, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		data = new CachedData(_session.getSessionManager().getMaxSyncStatesInDB());
		for (CachedDBEntry entry : dbList) {
			try {
				data.loadFromDB(entry.getTimestamp(), readFromString(entry.getData()));
			} catch (DeserializationFailedException e) {
				_session.logDeserializationError(objectID, entry.getTimestamp(), e);
			}
		}
		_internalCache.put(objectID, data);
		return data;
	}

	private Serializable[][] readFromString(String encodedData) throws DeserializationFailedException {
		try {
			ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new ByteArrayInputStream(Toolkit
					.decodeBase64(encodedData))));
			return (Serializable[][]) in.readObject();
		} catch (ClassCastException e) {
			throw new DeserializationFailedException(
					USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_CAST_EXCEPTION,
					"Stored object is not of correct type", e);
		} catch (IOException e) {
			throw new DeserializationFailedException(USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_IO_EXCEPTION,
					"Can not retrieve object from bytes", e);
		} catch (ClassNotFoundException e) {
			throw new DeserializationFailedException(
					USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_NOT_FOUND_EXCEPTION,
					"Can not find the class of the serializable object", e);
		}
	}

	private String writeToString(Serializable[][] objects) {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			ObjectOutputStream oos = new ObjectOutputStream(new GZIPOutputStream(baos));
			oos.writeObject(objects);
			oos.close();
			String result = Toolkit.encodeBase64(baos.toByteArray());
			//			_numberOfObjects.incrementAndGet(); //TODO: remove.. just for testing
			//			_sumOfObjectSizes.addAndGet(result.length()); //TODO: remove.. just for testing
			return result;
		} catch (IOException e) {
			throw new USMIllegalStateException(USMSessionManagementErrorCodes.SERIALIZATION_ERROR,
					"Can not serialize data to byte array", e);
		}
	}

	private Log getJournal() {
		return _session.getSessionManager().getJournal();
	}
}
