/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.dataobject;

import java.util.*;

import junit.framework.TestCase;

import org.json.JSONArray;
import org.json.JSONException;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.session.*;

public class SimpleDataObjectTest extends TestCase {

	private final static SimSession SESSION = new SimSession();
	private final static SimContentType CONTENT_TYPE = new SimContentType();

	public static SimpleDataObject[] createTestDataObjectsFromStringArray(Session session, ContentType type, String text) {
		List<SimpleDataObject> list = new ArrayList<SimpleDataObject>();
		if (text.charAt(0) != '[')
			throw new IllegalArgumentException("Missing '['");
		int len = text.length();
		if (text.charAt(len - 1) != ']')
			throw new IllegalArgumentException("Missing ']'");
		text = text.substring(1, len - 1);
		int start = 0;
		for (int i = findCharacter(text, ',', start); i >= 0; i = findCharacter(text, ',', start)) {
			list.add(createTestDataObjectFromString(session, type, text.substring(start, i)));
			start = i + 1;
		}
		list.add(createTestDataObjectFromString(session, type, text.substring(start)));
		return list.toArray(new SimpleDataObject[list.size()]);
	}

	public static SimpleDataObject createTestDataObjectFromString(Session session, ContentType type, String text) {
		try {
			int i = text.indexOf('(');
			String name = text.substring(0, i++);
			if (!SimpleDataObject.class.getSimpleName().equals(name))
				throw new IllegalArgumentException("DataObject type mismatch:" + name + " is not "
						+ SimpleDataObject.class.getSimpleName());
			int j = text.indexOf(',', i);
			String sessionName = text.substring(i, j++);
			if (!sessionName.equals(SimSession.class.getSimpleName()))
				throw new IllegalArgumentException("Session mismatch:" + sessionName + " is not "
						+ SimSession.class.getSimpleName());
			i = text.indexOf(',', j);
			String contentType = text.substring(j, i++);
			if (!contentType.equals(type.getID()))
				throw new IllegalArgumentException("ContentType mismatch:" + contentType + " is not " + type.getID());
			j = text.indexOf(')', i);
			String stateName = text.substring(i, j);
			ChangeState state = ChangeState.valueOf(stateName);
			String[] timeStamps = new String[2];
			j = parseField(text, "Timestamp", timeStamps, j + 2);

			ContentTypeField[] fields = type.getFields();
			String[][] fieldValues = new String[fields.length][2];
			for (i = 0; i < fields.length; i++) {
				j = parseField(text, fields[i].getFieldName(), fieldValues[i], j);
			}
			if (j < text.length())
				throw new IllegalArgumentException("Extra text after object");
			SimpleDataObject object = new SimpleDataObject(new SimSession(), type);
			object.setTimestamp(Long.parseLong(timeStamps[0]));

			for (i = 0; i < fields.length; i++)
				object.setFieldContent(i, generateFieldContent(session, fields[i], fieldValues[i][0]));
			object.commitChanges();
			object.setTimestamp(Long.parseLong(timeStamps[1]));

			for (i = 0; i < fields.length; i++)
				object.setFieldContent(i, generateFieldContent(session, fields[i], fieldValues[i][1]));
			if (object.getChangeState() != state)
				object.setChangeState(state);
			return object;
		} catch (IllegalArgumentException iae) {
			throw iae;
		} catch (Exception e) {
			throw new IllegalStateException("Error while parsing", e);
		}
	}

	public static FolderImpl createTestFolderObjectFromString(Session session, FolderContentType type, String text)
			throws IllegalArgumentException {
		try {
			int i = text.indexOf('(');
			i++;
			//			String name = text.substring(0, i++);

			int j = text.indexOf(',', i);
			String sessionName = text.substring(i, j++);
			if (!sessionName.equals(SimSession.class.getSimpleName()))
				throw new IllegalArgumentException("Session mismatch:" + sessionName + " is not "
						+ SimSession.class.getSimpleName());
			i = text.indexOf(',', j);
			String contentType = text.substring(j, i++);
			if (!contentType.equals(type.getID()))
				throw new IllegalArgumentException("ContentType mismatch:" + contentType + " is not " + type.getID());
			j = text.indexOf(')', i);
			String stateName = text.substring(i, j);
			ChangeState state = ChangeState.valueOf(stateName);
			String[] timeStamps = new String[2];
			j = parseField(text, "Timestamp", timeStamps, j + 2);

			ContentTypeField[] fields = type.getFields();
			String[][] fieldValues = new String[fields.length][2];
			for (i = 0; i < fields.length; i++) {
				j = parseField(text, fields[i].getFieldName(), fieldValues[i], j);
			}
			if (j < text.length())
				throw new IllegalArgumentException("Extra text after object");
			FolderImpl object = new FolderImpl(new SimSession(), type);
			object.setTimestamp(Long.parseLong(timeStamps[0]));

			for (i = 0; i < fields.length; i++)
				object.setFieldContent(i, generateFieldContent(session, fields[i], fieldValues[i][0]));
			object.commitChanges();
			object.setTimestamp(Long.parseLong(timeStamps[1]));

			for (i = 0; i < fields.length; i++)
				object.setFieldContent(i, generateFieldContent(session, fields[i], fieldValues[i][1]));
			if (object.getChangeState() != state)
				object.setChangeState(state);
			return object;
		} catch (IllegalArgumentException iae) {
			throw iae;
		} catch (Exception e) {
			throw new IllegalStateException("Error while parsing", e);
		}
	}

	private static Object generateFieldContent(Session session, ContentTypeField contentTypeField, String str)
			throws JSONException {
		if (str.equals("null"))
			return null;
		if (str.equals("[]"))
			return new Object[0];
		try {
			return contentTypeField.getFieldType().extractFromJSONArray(session, new JSONArray('[' + str + ']'), 0);
		} catch (Exception e) {
			throw new IllegalArgumentException("Can't parse field " + contentTypeField + " with value " + str, e);
		}
	}

	private static int parseField(String text, String fieldName, String[] result, int j) {
		int colPos = text.indexOf(':', j);
		if (!text.substring(j, colPos).equals(fieldName))
			throw new IllegalArgumentException("Field name mismatch: " + text.substring(j, colPos) + " is not "
					+ fieldName);
		j = colPos + 1;
		int modI = text.indexOf("=>", j);
		int endI = findCharacter(text, ',', j);
		if (endI < 0)
			endI = findCharacter(text, '}', j);
		if (modI >= 0 && modI < endI) {
			result[0] = text.substring(j, modI);
			result[1] = text.substring(modI + 2, endI);
		} else {
			result[0] = text.substring(j, endI);
			result[1] = result[0];
		}
		return endI + 1;
	}

	private static int findCharacter(String text, char character, int start) {
		int len = text.length();
		for (int i = start; i < len; i++) {
			char c = text.charAt(i);
			if (c == character)
				return i;
			switch (text.charAt(i)) {
				case '(':
					i = findCharacter(text, ')', i + 1);
					break;
				case '{':
					i = findCharacter(text, '}', i + 1);
					break;
				case '[':
					i = findCharacter(text, ']', i + 1);
					break;
				default:
					break;
			}
		}
		return -1;
	}

	public void testSimpleDataObject() {
		SimpleDataObject object = createTestObject();
		assertEquals(ChangeState.CREATED, object.getChangeState());
		assertTrue(object.isModified());
		try {
			new SimpleDataObject(SESSION, null);
			fail("Constructor with null ContentType did not fail");
		} catch (IllegalArgumentException ignored) {
		}
		try {
			new SimpleDataObject(null, CONTENT_TYPE);
			fail("Constructor with null Session did not fail");
		} catch (IllegalArgumentException ignored) {
		}
	}

	public void testGetSession() {
		SimpleDataObject object = createTestObject();
		assertEquals(SESSION, object.getSession());
	}

	public void testGetContentType() {
		SimpleDataObject object = createTestObject();
		assertEquals(CONTENT_TYPE, object.getContentType());
	}

	public void testSetChangeState() {
		SimpleDataObject object = createTestObject();
		assertEquals(ChangeState.CREATED, object.getChangeState());
		checkForFailureOnChangeStateChange(object);
		object.commitChanges();
		assertEquals(ChangeState.UNMODIFIED, object.getChangeState());
		object.setChangeState(ChangeState.CREATED);
		assertEquals(ChangeState.CREATED, object.getChangeState());
		checkForFailureOnChangeStateChange(object);
		object.commitChanges();
		assertEquals(ChangeState.UNMODIFIED, object.getChangeState());
		object.setChangeState(ChangeState.DELETED);
		assertEquals(ChangeState.DELETED, object.getChangeState());
		checkForFailureOnChangeStateChange(object);
		object.commitChanges();
		assertEquals(ChangeState.UNMODIFIED, object.getChangeState());
		try {
			object.setChangeState(ChangeState.MODIFIED);
			fail("setChangeState to " + ChangeState.MODIFIED + " on " + ChangeState.UNMODIFIED
					+ " DataObject did not fail");
		} catch (IllegalArgumentException e) {
		}
	}

	private void checkForFailureOnChangeStateChange(SimpleDataObject object) {
		ChangeState originalState = object.getChangeState();
		for (ChangeState state : ChangeState.values()) {
			try {
				object.setChangeState(state);
				if (state != ChangeState.CREATED && state != ChangeState.DELETED)
					fail("setChangeState to " + state + " on " + originalState + " DataObject did not fail");
			} catch (IllegalArgumentException e) {
				if (state == ChangeState.CREATED || state == ChangeState.DELETED)
					fail("setChangeState to " + state + " on " + originalState + " DataObject failed");
			}
			object.setChangeState(originalState);
		}
	}

	public void testSetID() {
		SimpleDataObject object = createTestObject();
		assertEquals(null, object.getID());
		assertEquals(null, object.getOriginalID());
		object.setID("TestID");
		assertEquals("TestID", object.getID());
		assertEquals(null, object.getOriginalID());
		object.commitChanges();
		assertEquals("TestID", object.getID());
		assertEquals("TestID", object.getOriginalID());
		object.setID("TestID2");
		assertEquals("TestID2", object.getID());
		assertEquals("TestID", object.getOriginalID());
		object.rollbackChanges();
		assertEquals("TestID", object.getID());
		assertEquals("TestID", object.getOriginalID());
	}

	public void testSetTimestamp() {
		SimpleDataObject object = createTestObject();
		assertEquals(0L, object.getTimestamp());
		assertEquals(0L, object.getOriginalTimestamp());
		object.setTimestamp(1L);
		assertEquals(1L, object.getTimestamp());
		assertEquals(0L, object.getOriginalTimestamp());
		object.commitChanges();
		assertEquals(1L, object.getTimestamp());
		assertEquals(1L, object.getOriginalTimestamp());
		object.setTimestamp(2L);
		assertEquals(2L, object.getTimestamp());
		assertEquals(1L, object.getOriginalTimestamp());
		object.rollbackChanges();
		assertEquals(1L, object.getTimestamp());
		assertEquals(1L, object.getOriginalTimestamp());
	}

	public void testSetParentFolderID() {
		SimpleDataObject object = createTestObject();
		assertEquals(null, object.getParentFolderID());
		assertEquals(null, object.getOriginalParentFolderID());
		object.setParentFolderID("TestID");
		assertEquals("TestID", object.getParentFolderID());
		assertEquals(null, object.getOriginalParentFolderID());
		object.commitChanges();
		assertEquals("TestID", object.getParentFolderID());
		assertEquals("TestID", object.getOriginalParentFolderID());
		object.setParentFolderID("TestID2");
		assertEquals("TestID2", object.getParentFolderID());
		assertEquals("TestID", object.getOriginalParentFolderID());
		object.rollbackChanges();
		assertEquals("TestID", object.getParentFolderID());
		assertEquals("TestID", object.getOriginalParentFolderID());
	}

	public void testSetParentFolder() {
		SimpleDataObject object = createTestObject();
		FolderImpl parentFolder = new FolderImpl(SESSION, new SimFolderContentType());
		FolderImpl parentFolder2 = new FolderImpl(SESSION, new SimFolderContentType());
		parentFolder.setID("folderID");
		parentFolder2.setID("folderID2");
		assertEquals(null, object.getParentFolderID());
		assertEquals(null, object.getOriginalParentFolderID());
		object.setParentFolder(parentFolder);
		assertEquals("folderID", object.getParentFolderID());
		assertEquals(null, object.getOriginalParentFolderID());
		object.commitChanges();
		assertEquals("folderID", object.getParentFolderID());
		assertEquals("folderID", object.getOriginalParentFolderID());
		object.setParentFolder(parentFolder2);
		assertEquals("folderID2", object.getParentFolderID());
		assertEquals("folderID", object.getOriginalParentFolderID());
		object.rollbackChanges();
		assertEquals("folderID", object.getParentFolderID());
		assertEquals("folderID", object.getOriginalParentFolderID());
	}

	public void testSetFieldContentIntObject() {
		SimpleDataObject object = createTestObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		Object[] emptyValue = new Object[0];
		Object[] setValue = new Object[] { "Test-Data" };
		Object[] setValue2 = new Object[] { "Test-Data2" };
		// TODO Add checks for DataType, fail on no error for invalid setter
		for (int i = 0; i < fields.length; i++) {
			if (i == 2) {
				assertTrue("Index " + i, Arrays.equals(emptyValue, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(emptyValue, (Object[]) object.getOriginalFieldContent(i)));
				object.setFieldContent(i, setValue[0]);
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(emptyValue, (Object[]) object.getOriginalFieldContent(i)));
				object.commitChanges();
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(i)));
				object.setFieldContent(i, setValue2[0]);
				assertTrue("Index " + i, Arrays.equals(setValue2, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(i)));
				object.rollbackChanges();
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(i)));
				object.setFieldContent(i, emptyValue);
				assertTrue("Index " + i, Arrays.equals(emptyValue, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(i)));
				object.setFieldContent(i, setValue2);
				assertTrue("Index " + i, Arrays.equals(setValue2, (Object[]) object.getFieldContent(i)));
				assertTrue("Index " + i, Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(i)));
			} else {
				assertEquals("Index " + i, null, object.getFieldContent(i));
				assertEquals("Index " + i, null, object.getOriginalFieldContent(i));
				object.setFieldContent(i, TEST_DATA[i]);
				assertEquals("Index " + i, TEST_DATA[i], object.getFieldContent(i));
				assertEquals("Index " + i, null, object.getOriginalFieldContent(i));
				object.commitChanges();
				assertEquals("Index " + i, TEST_DATA[i], object.getFieldContent(i));
				assertEquals("Index " + i, TEST_DATA[i], object.getOriginalFieldContent(i));
				object.setFieldContent(i, TEST_DATA2[i]);
				assertEquals("Index " + i, TEST_DATA2[i], object.getFieldContent(i));
				assertEquals("Index " + i, TEST_DATA[i], object.getOriginalFieldContent(i));
				object.rollbackChanges();
				assertEquals("Index " + i, TEST_DATA[i], object.getFieldContent(i));
				assertEquals("Index " + i, TEST_DATA[i], object.getOriginalFieldContent(i));
			}
		}
	}

	public void testSetFieldContentStringObject() {
		SimpleDataObject object = createTestObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		Object[] emptyValue = new Object[0];
		Object[] setValue = new Object[] { "Test-Data" };
		Object[] setValue2 = new Object[] { "Test-Data2" };
		// TODO Add checks for DataType, fail on no error for invalid setter
		for (int i = 0; i < fields.length; i++) {
			String fieldName = fields[i].getFieldName();
			if (i == 2) {
				assertTrue(Arrays.equals(emptyValue, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(emptyValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.setFieldContent(i, setValue[0]);
				assertTrue(Arrays.equals(setValue, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(emptyValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.commitChanges();
				assertTrue(Arrays.equals(setValue, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.setFieldContent(i, setValue2[0]);
				assertTrue(Arrays.equals(setValue2, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.rollbackChanges();
				assertTrue(Arrays.equals(setValue, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.setFieldContent(i, emptyValue);
				assertTrue(Arrays.equals(emptyValue, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(fieldName)));
				object.setFieldContent(i, setValue2);
				assertTrue(Arrays.equals(setValue2, (Object[]) object.getFieldContent(fieldName)));
				assertTrue(Arrays.equals(setValue, (Object[]) object.getOriginalFieldContent(fieldName)));
			} else {
				assertEquals(null, object.getFieldContent(fieldName));
				assertEquals(null, object.getOriginalFieldContent(fieldName));
				object.setFieldContent(i, TEST_DATA[i]);
				assertEquals(TEST_DATA[i], object.getFieldContent(fieldName));
				assertEquals(null, object.getOriginalFieldContent(fieldName));
				object.commitChanges();
				assertEquals(TEST_DATA[i], object.getFieldContent(fieldName));
				assertEquals(TEST_DATA[i], object.getOriginalFieldContent(fieldName));
				object.setFieldContent(i, TEST_DATA2[i]);
				assertEquals(TEST_DATA2[i], object.getFieldContent(fieldName));
				assertEquals(TEST_DATA[i], object.getOriginalFieldContent(fieldName));
				object.rollbackChanges();
				assertEquals(TEST_DATA[i], object.getFieldContent(fieldName));
				assertEquals(TEST_DATA[i], object.getOriginalFieldContent(fieldName));
			}
		}
	}

	private final static Object[] TEST_DATA = {
			"Test-Data",
			"Test-Data",
			new String[] { "Test-Data" },
			Boolean.FALSE,
			1.0,
			0L,
			0L,
			0L };
	private final static Object[] TEST_DATA2 = {
			"Test-Data2",
			"Test-Data2",
			new String[] { "Test-Data2" },
			Boolean.TRUE,
			2.0,
			1L,
			1L,
			1L };

	public void testIsFieldModifiedInt() {
		SimpleDataObject object = createTestObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		for (int i = 0; i < fields.length; i++) {
			assertFalse(object.isFieldModified(i));
			object.setFieldContent(i, TEST_DATA[i]);
			assertTrue(object.isFieldModified(i));
			object.commitChanges();
			assertFalse(object.isFieldModified(i));
			object.setFieldContent(i, TEST_DATA2[i]);
			assertTrue(object.isFieldModified(i));
			object.rollbackChanges();
			assertFalse(object.isFieldModified(i));
			object.setFieldContent(i, TEST_DATA2[i]);
			assertTrue(object.isFieldModified(i));
			object.setFieldContent(i, TEST_DATA[i]);
			assertFalse(object.isFieldModified(i));
		}
	}

	public void testIsFieldModifiedString() {
		SimpleDataObject object = createTestObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		for (int i = 0; i < fields.length; i++) {
			String fieldName = fields[i].getFieldName();
			assertFalse(object.isFieldModified(fieldName));
			object.setFieldContent(i, TEST_DATA[i]);
			assertTrue(object.isFieldModified(fieldName));
			object.commitChanges();
			assertFalse(object.isFieldModified(fieldName));
			object.setFieldContent(i, TEST_DATA2[i]);
			assertTrue(object.isFieldModified(fieldName));
			object.rollbackChanges();
			assertFalse(object.isFieldModified(fieldName));
			object.setFieldContent(i, TEST_DATA2[i]);
			assertTrue(object.isFieldModified(fieldName));
			object.setFieldContent(i, TEST_DATA[i]);
			assertFalse(object.isFieldModified(fieldName));
		}
	}

	public void testIsModified() {
		SimpleDataObject object = createTestObject();
		assertTrue(object.isModified());
		object.commitChanges();
		assertFalse(object.isModified());
		object.setChangeState(ChangeState.CREATED);
		assertTrue(object.isModified());
		object.commitChanges();
		assertFalse(object.isModified());
		object.setChangeState(ChangeState.DELETED);
		assertTrue(object.isModified());
		object.commitChanges();
		assertFalse(object.isModified());
		object.setTimestamp(1L);
		assertTrue(object.isModified());
		object.setTimestamp(0L);
		assertFalse(object.isModified());
		object.setFieldContent(1, "Test-Data");
		assertTrue(object.isModified());
		object.setFieldContent(1, null);
		assertFalse(object.isModified());
	}

	public void testRollbackChanges() {
		SimpleDataObject object = createTestObject();
		assertEquals(0L, object.getOriginalTimestamp());
		assertEquals(0L, object.getTimestamp());
		object.setTimestamp(1L);
		object.rollbackChanges();
		assertEquals(0L, object.getTimestamp());
	}

	public void testCommitChanges() {
		SimpleDataObject object = createTestObject();
		assertEquals(0L, object.getOriginalTimestamp());
		assertEquals(0L, object.getTimestamp());
		object.setTimestamp(1L);
		object.commitChanges();
		assertEquals(1L, object.getTimestamp());
		assertEquals(1L, object.getOriginalTimestamp());
	}

	public void testToString() {
		SimpleDataObject object = createTestObject();
		assertEquals(
				"SimpleDataObject(SimSession,SimContentType,CREATED){Timestamp:0,id:null,folder_id:null,Field2:[],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}",
				object.toString());
		object.setTimestamp(1L);
		assertEquals(
				"SimpleDataObject(SimSession,SimContentType,CREATED){Timestamp:0=>1,id:null,folder_id:null,Field2:[],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}",
				object.toString());
		object.commitChanges();
		assertEquals(
				"SimpleDataObject(SimSession,SimContentType,UNMODIFIED){Timestamp:1,id:null,folder_id:null,Field2:[],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}",
				object.toString());
		object.setFieldContent(2, new String[] { "Entry1", "Entry2" });
		assertEquals(
				"SimpleDataObject(SimSession,SimContentType,MODIFIED){Timestamp:1,id:null,folder_id:null,Field2:[]=>["
						+ '"' + "Entry1" + '"' + "," + '"' + "Entry2" + '"'
						+ "],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}", object.toString());
		object.commitChanges();
		assertEquals(
				"SimpleDataObject(SimSession,SimContentType,UNMODIFIED){Timestamp:1,id:null,folder_id:null,Field2:["
						+ '"' + "Entry1" + '"' + "," + '"' + "Entry2" + '"'
						+ "],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}", object.toString());
		object.setFieldContent(2, null);
		assertEquals("SimpleDataObject(SimSession,SimContentType,MODIFIED){Timestamp:1,id:null,folder_id:null,Field2:["
				+ '"' + "Entry1" + '"' + "," + '"' + "Entry2" + '"'
				+ "]=>[],Field3:null,Field4:null,Field5:null,Field6:null,Field7:null}", object.toString());
		SimpleDataObject o2 = createTestDataObjectFromString(SESSION, CONTENT_TYPE, object.toString());
		assertEquals(object.toString(), o2.toString());
	}

	private SimpleDataObject createTestObject() {
		return new SimpleDataObject(SESSION, CONTENT_TYPE);
	}
}
