/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.dataobject;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.session.*;

/**
 * Base implementation of Folder based on SimpleDataObject. When the ContentType of the
 * elements is set, it automatically updates the appropriate field (by default "module",
 * but another field may be specified in the constructor).<br>
 * Attention: There is no special handling for the other direction, i.e. if the field
 * "module" is set, the ContentType of the elements is not automatically set. This is due
 * to the fact that this object has no knowledge of available ContentTypes and therefore
 * the sync-system itself has to make sure that the ContentType of elements is correctly
 * set.
 * 
 * @author afe
 *
 */
public class FolderImpl extends SimpleDataObject implements Folder {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private ContentType _elementsContentType;

	private long _contentTimestamp, _originalContentTimestamp;

	private final int _elementsContentTypeFieldIndex;

	private final int _subfoldersFieldIndex;

	public FolderImpl(Session session, FolderContentType contentType) {
		super(session, contentType);
		if (!contentType.getID().equals(DefaultContentTypes.FOLDER_ID))
			throw new IllegalArgumentException("Illegal ContentType " + contentType.getID() + " for Folder object");
		_elementsContentTypeFieldIndex = _fieldNameToIndexMap.get("module");
		_subfoldersFieldIndex = _fieldNameToIndexMap.get("subfolders");
	}

	protected FolderImpl(FolderImpl source, boolean linkUUIDs) {
		super(source, linkUUIDs);
		_elementsContentType = source._elementsContentType;
		_contentTimestamp = source._contentTimestamp;
		_originalContentTimestamp = source._originalContentTimestamp;
		_elementsContentTypeFieldIndex = source._elementsContentTypeFieldIndex;
		_subfoldersFieldIndex = source._subfoldersFieldIndex;
	}

	@Override
	public FolderContentType getContentType() {
		return (FolderContentType) super.getContentType();
	}

	public ContentType getElementsContentType() {
		return _elementsContentType;
	}

	public void setElementsContentType(ContentType type) {
		if (_elementsContentType != null)
			throw new IllegalStateException("ContentType of elements already set");
		if (type != null) {
			_elementsContentType = type;
			setFieldContent(_elementsContentTypeFieldIndex, type.getID());
		}
	}

	public String getElementsContentTypeID() {
		if (_elementsContentType != null)
			return _elementsContentType.getID();
		return convertObjectToString(getFieldContent(_elementsContentTypeFieldIndex));
	}

	public long getContentSyncTimestamp() {
		return _contentTimestamp;
	}

	public long getOriginalContentSyncTimestamp() {
		return _originalContentTimestamp;
	}

	public void setContentSyncTimestamp(long timestamp) {
		_contentTimestamp = timestamp;
		updateChangeState(_contentTimestamp != _originalContentTimestamp);
	}

	@Override
	public void rollbackChanges() {
		_contentTimestamp = _originalContentTimestamp;
		super.rollbackChanges();
	}

	@Override
	public void commitChanges() {
		_originalContentTimestamp = _contentTimestamp;
		super.commitChanges();
	}

	@Override
	protected void updateChangeState(boolean newFieldContentIsNotEqual) {
		super.updateChangeState(newFieldContentIsNotEqual);
		if (_changeState == ChangeState.UNMODIFIED && _contentTimestamp != _originalContentTimestamp)
			_changeState = ChangeState.MODIFIED;
	}

	@Override
	protected void addSpecialFieldsToStringBuilder(StringBuilder sb) {
		super.addSpecialFieldsToStringBuilder(sb);
		sb.append(',');
		if (_formatToString)
			sb.append("\n   ");
		sb.append("ContentTimestamp:").append(_originalContentTimestamp);
		if (_contentTimestamp != _originalContentTimestamp)
			sb.append("=>").append(_contentTimestamp);
	}

	@Override
	public Folder createCopy(boolean linkUUIDs) {
		return new FolderImpl(this, linkUUIDs);
	}

	public boolean hasSubFolders() {
		Object o = _fieldContent[_subfoldersFieldIndex];
		// Return true if either o is no Boolean (e.g. null), which means "unknown" or if the boolean value is true
		return !(o instanceof Boolean) || ((Boolean) o).booleanValue();
	}
}
