/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.session.dataobject;

import java.util.*;

import com.openexchange.usm.api.session.Folder;

/**
 * Compares two folders for their hierarchical order. 
 * 
 * @author ldo
 *
 */
public class FolderHierarchyComparator implements Comparator<Folder> {

	private final List<Folder> _firstHierarchy = new ArrayList<Folder>();
	private final List<Folder> _secondHierarchy = new ArrayList<Folder>();
	private final Map<String, Folder> _foldersByID = new HashMap<String, Folder>();

	public FolderHierarchyComparator(Folder[] sortingArray) {
		for (Folder f : sortingArray) {
			if (f != null) {
				String id = f.getID();
				if (id != null)
					_foldersByID.put(id, f);
			}
		}
	}

	public int compare(Folder f1, Folder f2) {
		if (f1 == f2)
			return 0;
		if (f1 == null)
			return 1;
		if (f2 == null)
			return -1;
		buildHierarchy(_firstHierarchy, f1);
		buildHierarchy(_secondHierarchy, f2);
		int i1 = _firstHierarchy.size();
		int i2 = _secondHierarchy.size();
		// Walk from root folder towards the 2 folders until different folders are encountered
		while (i1 > 0 && i2 > 0) {
			Folder p1 = _firstHierarchy.get(--i1);
			Folder p2 = _secondHierarchy.get(--i2);
			if (p1 != p2) // Not the same folder, compare based on differing parent folder ids
				return compareIDs(p1, p2);
		}
		// Either i1 or i2 must be > 0 and the other must be 0, i.e. one of the 2 folders is the parent (directly or indirectly) of the other
		// The parent folder is "before" the child folder
		return (i1 > i2) ? 1 : -1;
	}

	private void buildHierarchy(List<Folder> hierarchy, Folder f) {
		for (hierarchy.clear(); f != null; f = findParentFolder(f))
			hierarchy.add(f);
	}

	private int compareIDs(Folder f1, Folder f2) {
		if (f1 == null)
			return f2 == null ? 0 : 1;
		return f2 == null ? -1 : compareIDs(f1.getID(), f2.getID());
	}

	private int compareIDs(String id1, String id2) {
		if (id1 == null)
			return id2 == null ? 0 : 1;
		return id2 == null ? -1 : id1.compareTo(id2);
	}

	private Folder findParentFolder(Folder child) {
		Folder parent = child.getParentFolder();
		if (parent != null)
			return parent;
		return _foldersByID.get(child.getParentFolderID());
	}
}
