/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.ox_json.impl;

import java.io.*;
import java.net.URLEncoder;
import java.util.*;

import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.methods.*;
import org.apache.commons.httpclient.methods.multipart.*;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.logging.Log;
import org.json.*;

import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.ox_json.*;

/**
 * 
 * @author afe
 *
 */
public class OXJSONAccessImpl implements OXJSONAccess {
	private static final int MAX_REDIRECT_COUNT = 10;

	private static final String IMAGE = "image/";

	private static final String TEXT_HTML = "text/html";

	private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

	private final static boolean ALWAYS_COLLECT_STATISTICS = true;

	public final static String OX_AJAX_ACCESS_STORAGE_PROPERTY = "oxJSONAccess.OXConnectionData";

	private final static int RESOURCE_BUFFER_DEFAULT_SIZE = 40000;

	private final static String ACTION_KEY = "**action**";
	private final static String JSON_KEY = "**json**";
	private final static String FILE_PART_NAME = "file";
	private final static String STRING_PART_NAME = "json";
	private final static String BINARY_ENCODING = "binary";
	private final static String TEXT_JAVASCRIPT_CONTENTTYPE = "text/javascript";
	private final static String CHARSET_UTF8 = "UTF-8";
	private final static String CONFIG = "config/";
	private final static String DATA = "data";
	private final static String HTTPS_PREFIX = "https:";
	//private final static String UDATE_ACTION = "update";

	// Check after 5 minutes if redirect still active, restricts number of log messages and redirect tests
	private static final long REDIRECT_RETEST_INTERVAL = 300000l;

	private Log _journal;

	private MultiThreadedHttpConnectionManager _httpManager;

	private Thread _backgroundConnectionCloser;

	private int _connectionTimeout;

	private String _oxJSONBaseURLasString;
	private HttpURL _oxJSONBaseURL;

	private String _postResultPattern;
	private int _actionIndex;
	private int _jsonIndex;

	private long _lastRedirect = 0L;

	public void login(Session session) throws AuthenticationFailedException, OXCommunicationException {
		logout(session, false);
		try {
			OXConnectionData result = performLogin(session);
			if (_journal.isDebugEnabled())
				_journal.debug(session + " Logged in, session id = " + result.getSessionID());
		} catch (AuthenticationFailedException e) {
			if (_journal.isDebugEnabled())
				_journal.debug(session + " Authentication failed", e);
			throw e;
		} catch (OXCommunicationException e) {
			_journal.warn(session + " Error while authenticating", e);
			throw e;
		}
	}

	public JSONObject getConfiguration(Session session, String... path) throws AuthenticationFailedException,
			OXCommunicationException {
		JSONObject config = readConfiguration(session, path);
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Retrieved OX ajax configuration");
		return config;
	}

	private JSONObject readConfiguration(Session session, String... path) throws AuthenticationFailedException,
			OXCommunicationException {
		String accessPath = CONFIG;
		if (path.length > 0) {
			StringBuilder sb = new StringBuilder(CONFIG);
			for (String part : path)
				sb.append(part).append('/');
			accessPath = sb.toString();
		}
		JSONResult result = doGet(accessPath, null, session, null);
		if (result.getResultType() != JSONResultType.JSONObject)
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_ERROR_WHILE_AUTHENTICATING_NUMBER_3,
					"Unknown OX response reading configuration");
		try {
			return result.getJSONObject().getJSONObject(DATA);
		} catch (JSONException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_ILLEGAL_DATA_BY_OXCONFIG_RESPONSE_NUMBER_4,
					"OX configuration response contains illegal data", e);
		}
	}

	private OXConnectionData performLogin(Session session) throws AuthenticationFailedException,
			OXCommunicationException {
		String user = session.getUser();
		String password = session.getPassword();
		HttpClient client = new HttpClient(_httpManager);
		PostMethod method = new PostMethod(getBaseURL() + "login?client=" + urlEncode("USM-" + session.getProtocol())
				+ "&version=" + urlEncode(USMVersion.VERSION_BUILD) + "&authId=" + UUID.randomUUID());
		method.addParameter("action", "login");
		method.addParameter("name", user);
		method.addParameter("password", password);
		JSONResult result = executeMethodOnce(session, client, method);
		if (result.getResultType() == JSONResultType.Error)
			throw new AuthenticationFailedException(OXJSONErrorCode.ACCESSIMPL_AUTHENTICATION_FAILED_NUMBER_5,
					"Authentication failed", result.getJSONObject());
		try {
			String sessionID = result.getJSONObject().getString("session");
			OXConnectionData data = new OXConnectionData(sessionID, client);
			session.setCustomProperty(OX_AJAX_ACCESS_STORAGE_PROPERTY, data);
			return data;
		} catch (JSONException e) {
			throw new AuthenticationFailedException(OXJSONErrorCode.ACCESSIMPL_RESPONSE_NOT_CONTAIN_SESSIONID_NUMBER_6,
					"Server response did not contain session ID", result.getJSONObject());
		}
	}

	public void logout(Session session) {
		logout(session, true);
	}

	private void logout(Session session, boolean doLog) {
		if (doLog && _journal.isDebugEnabled())
			_journal.debug(session + " Logging out");
		OXConnectionData data = (OXConnectionData) session.setCustomProperty(OX_AJAX_ACCESS_STORAGE_PROPERTY, null);
		if (data == null)
			return;
		GetMethod method = new GetMethod(getBaseURL() + "login?action=logout&session=" + urlEncode(data.getSessionID()));
		try {
			executeMethod(data.getHttpClient(), method);
		} catch (HttpException ignored) {
		} catch (IOException ignored) {
		} finally {
			method.releaseConnection();
		}
	}

	public JSONResult doGet(String path, String action, Session session, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException {
		OXConnectionData data = obtainConnectionData(session);
		String url = getBaseURL() + path;
		if (_journal.isDebugEnabled())
			_journal.debug(session + " GET " + url);
		GetMethod method = new GetMethod(url);
		method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
		return executeMethodAndExtractResult(session, data, method);
	}

	public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONArray array)
			throws AuthenticationFailedException, OXCommunicationException {
		return doPut(path, action, session, parameters, array.toString());
	}

	public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters,
			JSONObject object) throws AuthenticationFailedException, OXCommunicationException {
		return doPut(path, action, session, parameters, object.toString());
	}

	public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String str)
			throws AuthenticationFailedException, OXCommunicationException {
		OXConnectionData data = obtainConnectionData(session);
		String url = getBaseURL() + path;
		if (_journal.isDebugEnabled()) {
			_journal.debug(session + " PUT " + url + '\n' + str);
		}
		PutMethod method = new PutMethod(url);
		method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
		try {
			method.setRequestEntity(new StringRequestEntity(str, TEXT_JAVASCRIPT_CONTENTTYPE, CHARSET_UTF8));
		} catch (UnsupportedEncodingException e1) {
			throw new USMIllegalStateException(OXJSONErrorCode.ACCESSIMPL_UNSUPPORTED_ENCODING_NUMBER_7,
					"UnsupportedEncoding: Character encoding UTF-8 not found");
		}
		return executeMethodAndExtractResult(session, data, method);
	}

	public JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, byte[] bytes)
			throws AuthenticationFailedException, OXCommunicationException {
		OXConnectionData data = obtainConnectionData(session);
		String url = getBaseURL() + path;
		if (_journal.isDebugEnabled())
			_journal.debug(session + " PUT " + url + " with byte array");
		PutMethod method = new PutMethod(url);
		method.setQueryString(generateQueryParameters(action, data.getSessionID(), parameters));
		method.setRequestEntity(new ByteArrayRequestEntity(bytes));
		return executeMethodAndExtractResult(session, data, method);
	}

	public void activate(Log log, String oxJSONURL, String postResultPattern, int maxConnections,
			int connectionTimeout, boolean performStaleChecking) {
		String error = doActivate(log, oxJSONURL, postResultPattern, maxConnections, connectionTimeout,
				performStaleChecking);
		if (error != null) {
			error = "OXJSONAccess not activated, reason: " + error;
			log.error(error);
			deactivate(null);
			throw new USMStartupException(OXJSONErrorCode.ACCESSIMPL_OXAJAXACCESS_NOTACTIVATED_ENCODING_NUMBER_8, error);
		}
	}

	public void deactivate() {
		deactivate("USM OXJSONAccess deactivated");
	}

	private String doActivate(Log log, String oxJSONURL, String postResultPattern, int maxConnections,
			int connectionTimeout, boolean performStaleChecking) {
		_journal = log;
		_oxJSONBaseURLasString = oxJSONURL;
		try {
			_oxJSONBaseURL = (oxJSONURL.toLowerCase().startsWith(HTTPS_PREFIX)) ? new HttpsURL(oxJSONURL, CHARSET_UTF8)
					: new HttpURL(oxJSONURL, CHARSET_UTF8);
		} catch (URIException e) {
			return "OXJSONAccess base URL is no valid http(s) URL: " + _oxJSONBaseURLasString;
		}
		_postResultPattern = postResultPattern.trim();
		_actionIndex = _postResultPattern.indexOf(ACTION_KEY);
		if (_actionIndex < 0)
			return ACTION_KEY + " not found in post result pattern";
		_jsonIndex = _postResultPattern.indexOf(JSON_KEY);
		if (_jsonIndex < 0)
			return JSON_KEY + " not found in post result pattern";
		_connectionTimeout = connectionTimeout;
		_httpManager = new MultiThreadedHttpConnectionManager();
		HttpConnectionManagerParams httpParams = _httpManager.getParams();
		httpParams.setDefaultMaxConnectionsPerHost(maxConnections);
		httpParams.setMaxTotalConnections(maxConnections);
		httpParams.setConnectionTimeout(_connectionTimeout);
		httpParams.setStaleCheckingEnabled(performStaleChecking);
		_backgroundConnectionCloser = new Thread() {
			@Override
			public void run() {
				try {
					while (!isInterrupted()) {
						Thread.sleep(_connectionTimeout);
						_httpManager.closeIdleConnections(_connectionTimeout);
					}
				} catch (InterruptedException ignored) {
					// We get interrupted to indicate that this thread has to stop, so simply stop
				}
			}
		};
		_backgroundConnectionCloser.setPriority(Thread.MIN_PRIORITY);
		_backgroundConnectionCloser.setDaemon(true);
		_backgroundConnectionCloser.start();
		_journal.info("USM OXJSONAccess activated, OX-URL=" + _oxJSONBaseURLasString);
		return null;
	}

	private void deactivate(String message) {
		if (_backgroundConnectionCloser != null && _backgroundConnectionCloser.isAlive())
			_backgroundConnectionCloser.interrupt();
		_backgroundConnectionCloser = null;
		_oxJSONBaseURL = null;
		_oxJSONBaseURLasString = null;
		_postResultPattern = null;
		if (message != null)
			_journal.info(message);
		_journal = null;
	}

	private OXConnectionData obtainConnectionData(Session session) throws AuthenticationFailedException,
			OXCommunicationException {
		Object o = session.getCustomProperty(OX_AJAX_ACCESS_STORAGE_PROPERTY);
		return (o instanceof OXConnectionData) ? ((OXConnectionData) o) : performLogin(session);
	}

	private NameValuePair[] generateQueryParameters(String action, String sessionID, Map<String, String> params) {
		List<NameValuePair> list = new ArrayList<NameValuePair>();
		if (action != null)
			list.add(new NameValuePair("action", action));
		if (sessionID != null)
			list.add(new NameValuePair("session", sessionID));
		if (params != null) {
			for (Map.Entry<String, String> entry : params.entrySet())
				list.add(new NameValuePair(entry.getKey(), entry.getValue()));
		}
		list.add(new NameValuePair("timezone", "UTC"));
		return list.toArray(new NameValuePair[list.size()]);
	}

	private String getBaseURL() {
		return _oxJSONBaseURLasString;
	}

	private String urlEncode(String s) {
		try {
			return URLEncoder.encode(s, CHARSET_UTF8);
		} catch (UnsupportedEncodingException e) {
			throw new USMIllegalStateException(OXJSONErrorCode.ACCESSIMPL_UTF8_ENCODING_NOTFOUND_NUMBER_10,
					"Character encoding UTF-8 not found");
		}
	}

	private JSONResult executeMethodAndExtractResult(Session session, OXConnectionData data, HttpMethodBase method)
			throws AuthenticationFailedException, OXCommunicationException {
		JSONResult result = executeMethodOnce(session, data.getHttpClient(), method);
		// On specific error for expired/unknown session, do login and repeat
		if (shouldPerformNewLogin(result)) {
			if (_journal.isDebugEnabled())
				_journal.debug(session + " Session unknown to OX server, performing login and repeat");
			login(session);
			data = obtainConnectionData(session);
			result = executeMethodOnce(session, data.getHttpClient(), method);
		}
		return result;
	}

	private JSONResult executeMethodOnce(Session session, HttpClient client, HttpMethodBase method)
			throws OXCommunicationException {
		boolean collectOXStatistics = ALWAYS_COLLECT_STATISTICS || _journal.isDebugEnabled();
		long startTime = collectOXStatistics ? System.currentTimeMillis() : 0L;
		try {
			int statusCode = executeMethod(client, method);
			if (statusCode != HttpStatus.SC_OK)
				throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTP_CONNECTION_ERROR_11, statusCode);
			JSONResult result = OXJSONSupport.parseResult(extractResponseBodyAsString(method), null);
			if (isSynchronizationConflictError(result))
				throw new ConflictingChangeException(OXJSONErrorCode.ACCESSIMPL_CONFLICT_CHANGE_ON_OXACCESS_NUMBER_12,
						"Conflicting change on OX access", result.getJSONObject());
			return result;
		} catch (HttpException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_HTTPEXCEPTION_OCCURED_NUMBER_13, e);
		} catch (IOException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_EXECUTE_METHOD_ERROR_NUMBER_14, e);
		} finally {
			method.releaseConnection();
			if (collectOXStatistics) {
				long duration = System.currentTimeMillis() - startTime;
				long number = getSessionProperty(session, OXJSONPropertyNames.NUMBER_OF_CALLS);
				session.setCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS, Long.valueOf(number + 1L));
				long accTime = getSessionProperty(session, OXJSONPropertyNames.ACCUMULATED_TIME);
				session.setCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME, Long.valueOf(accTime + duration));
			}
		}
	}

	private long getSessionProperty(Session session, String name) {
		Object o = session.getCustomProperty(name);
		return (o instanceof Long) ? ((Long) o).longValue() : 0L;
	}

	private boolean isSynchronizationConflictError(JSONResult result) {
		return result.getResultType() == JSONResultType.Error
				&& OXJSONSupport.ERROR_CODE_OBJECT_CHANGED.equals(OXJSONSupport
						.getJSONErrorCode(result.getJSONObject()));
	}

	private boolean shouldPerformNewLogin(JSONResult result) {
		if (result.getResultType() != JSONResultType.Error)
			return false;
		String errorCode = OXJSONSupport.getJSONErrorCode(result.getJSONObject());
		return OXJSONSupport.ERROR_CODE_UNKNOWN_SESSION.equals(errorCode)
				|| OXJSONSupport.ERROR_CODE_EXPIRED_SESSION.equals(errorCode);
	}

	public OXResource getResource(Session session, String path, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException {
		GetMethod method = null;
		try {
			if (_journal.isDebugEnabled())
				_journal.debug(session + " GET " + _oxJSONBaseURLasString + " + " + path);
			OXConnectionData connectionData = obtainConnectionData(session);
			if (parameters == null) {
				HttpURL url = new HttpURL(_oxJSONBaseURL, new HttpURL(path.toCharArray()));
				method = new GetMethod();
				method.setURI(url);
			} else {
				String url = getBaseURL() + path;
				method = new GetMethod(url);
				method.setQueryString(generateQueryParameters(null, connectionData.getSessionID(), parameters));
			}
			int statusCode = executeMethod(connectionData.getHttpClient(), method);
			if (statusCode != HttpStatus.SC_OK)
				throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_15, statusCode);
			String contentType = method.getResponseHeader("Content-Type").getValue();
			if (APPLICATION_OCTET_STREAM.equals(contentType))
				return new OXResource(contentType, extractResponseBody(method));
			else if (contentType != null && contentType.startsWith(IMAGE))
				return new OXResource(contentType, extractResponseBody(method));
			else if (TEXT_HTML.equals(contentType)) {
				String httpResult = extractResponseBodyAsString(method);
				int startIndex = httpResult.indexOf("({");
				int endIndex = httpResult.lastIndexOf("})");
				JSONResult json = new JSONErrorResult(new JSONObject());
				if (startIndex > 0 && endIndex > startIndex) {
					json = checkPostResult(httpResult.substring(startIndex + 1, endIndex + 1));
				}
				throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_ERROR_ON_READING_RESOURCE,
						"Error on getting resource from Server", json.getJSONObject());
			} else {
				throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_CONTENT_TYPE_RECEIVED,
						"Invalid content type in response: " + contentType);
			}
		} catch (URIException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_16,
					"Invalid URL specified: " + path, e);
		} catch (HttpException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_17, e);
		} catch (IOException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_GETRESOURCE_ERROR_NUMBER_18, e);
		} finally {
			if (method != null)
				method.releaseConnection();
		}
	}

	public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] imageByteData, String fileContentType) throws AuthenticationFailedException,
			OXCommunicationException {
		return storeResource(path, action, session, parameters, jsonData, imageByteData, fileContentType,
				STRING_PART_NAME, FILE_PART_NAME);
	}

	public JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] imageByteData, String fileContentType, String jsonPartName, String filePartName)
			throws AuthenticationFailedException, OXCommunicationException {
		PostMethod multiPartPostMethod = null;
		try {
			OXConnectionData connectionData = obtainConnectionData(session);
			String url = getBaseURL() + path;

			if (_journal.isDebugEnabled())
				_journal.debug(session + " POST " + url);

			multiPartPostMethod = new PostMethod(url);
			multiPartPostMethod.setQueryString(generateQueryParameters(action, connectionData.getSessionID(),
					parameters));

			//"text/javascript", "UTF-8"
			List<Part> parts = new ArrayList<Part>();

			if (jsonData != null) {
				StringPart jsonPart = new StringPart(jsonPartName, jsonData.toString(), CHARSET_UTF8);
				jsonPart.setContentType(TEXT_JAVASCRIPT_CONTENTTYPE);
				parts.add(jsonPart);
			}

			FilePart filePart = new FilePart(filePartName, new ByteArrayPartSource(filePartName, imageByteData),
					fileContentType, BINARY_ENCODING);
			filePart.setTransferEncoding(BINARY_ENCODING);
			parts.add(filePart);

			multiPartPostMethod.setRequestEntity(new MultipartRequestEntity(parts.toArray(new Part[parts.size()]),
					multiPartPostMethod.getParams()));

			int statusCode = executeMethod(connectionData.getHttpClient(), multiPartPostMethod);
			if (statusCode != HttpStatus.SC_OK)
				throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_STATUSCODE_NUMBER_19, statusCode);
			String httpResult = extractResponseBodyAsString(multiPartPostMethod);
			return checkAndExtractResult(httpResult);
		} catch (URIException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_URL_NUMBER_20,
					"Invalid URL specified", e);
		} catch (HttpException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_21, e);
		} catch (IOException e) {
			throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_STORE_RESOURCE_ERROR_NUMBER_22, e);
		} finally {
			if (multiPartPostMethod != null)
				multiPartPostMethod.releaseConnection();
		}
	}

	private JSONResult checkAndExtractResult(String httpResult) throws OXCommunicationException {
		httpResult = httpResult.trim();
		int startIndex = httpResult.indexOf("({");
		int endIndex = httpResult.lastIndexOf("})");
		if (startIndex > 0 && endIndex > startIndex) {
			String json = httpResult.substring(startIndex + 1, endIndex + 1);
			return checkPostResult(json);
		}
		throw new OXCommunicationException(OXJSONErrorCode.ACCESSIMPL_INVALID_RESULTOF_POST_REQUEST_NUMBER_23,
				"Result of POST request doesn't match configured pattern");
	}

	private JSONResult checkPostResult(String json) throws OXCommunicationException {
		try {
			return OXJSONSupport.generateJSONResultForJSONObject(new JSONObject(json), JSONResultType.JSONObject);
		} catch (JSONException e) {
			throw new OXCommunicationException(
					OXJSONErrorCode.ACCESSIMPL_RESULTOF_POST_NOTCONTAIN_VALID_JSONOBJECT_NUMBER_24,
					"Result of POST request doesn't contain valid JSONObject: " + json, e);
		}
	}

	private byte[] extractResponseBody(HttpMethod method) throws IOException {
		ByteArrayOutputStream byteArrayStream = new ByteArrayOutputStream(RESOURCE_BUFFER_DEFAULT_SIZE);
		InputStream inputStream = method.getResponseBodyAsStream();
		byte[] data = new byte[4096];
		for (int count = inputStream.read(data); count > 0; count = inputStream.read(data))
			byteArrayStream.write(data, 0, count);
		byteArrayStream.close();
		return byteArrayStream.toByteArray();
	}

	private String extractResponseBodyAsString(HttpMethodBase method) throws IOException {
		return new String(extractResponseBody(method), method.getResponseCharSet());
	}

	private int executeMethod(HttpClient client, HttpMethod method) throws HttpException, IOException {
		int status = HttpStatus.SC_NOT_FOUND;
		method.setFollowRedirects(false);
		for (int i = 0; i < MAX_REDIRECT_COUNT; i++) {
			status = client.executeMethod(method);
			Header location = getRedirection(method, status);
			if (location == null) {
				if (i > 0 && _lastRedirect < System.currentTimeMillis() - REDIRECT_RETEST_INTERVAL) {
					_lastRedirect = System.currentTimeMillis();
					_journal.info("HTTP Access to JSON interface of OX-Server was redirected");
				}
				return status;
			}
			URI newURI = new URI(location.getValue(), true, method.getParams().getUriCharset());
			if (newURI.isRelativeURI())
				newURI = new URI(method.getURI(), newURI);
			method.setURI(newURI);
		}
		_journal.error("Too many HTTP redirects accessing JSON interface of OX-Server");
		return status;
	}

	private Header getRedirection(HttpMethod method, int status) {
		return (status == HttpStatus.SC_MOVED_TEMPORARILY || status == HttpStatus.SC_MOVED_PERMANENTLY
				|| status == HttpStatus.SC_SEE_OTHER || status == HttpStatus.SC_TEMPORARY_REDIRECT) ? method
				.getResponseHeader("location") : null;
	}
}
