/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.mapping.impl;

import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.apache.commons.logging.Log;

import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.util.Toolkit;

public abstract class AbstractMappingDBAccess {

	protected static final int NUMBER_OF_SESSION_ID_FIELDS = 2;
	protected final Log _journal;
	protected final Session _session;

	public AbstractMappingDBAccess(Log journal, Session session) {
		_journal = journal;
		_session = session;
	}

	protected void deleteOrRetain(String errorText, String command, String extraCommandText, String optionalFixedParam,
			String[] values) throws DatabaseAccessException, USMSQLException {
		executeUpdate("", errorText, completeComplexStatement(command + extraCommandText, values.length),
				optionalFixedParam, values);
	}

	protected void executeUpdate(String warnMessage, String errorMessage, String command, String optionalFixedParam,
			String... parameters) throws DatabaseAccessException, USMSQLException {

		EncapsulatedConnection con = _session.getWritableDBConnection();
		try {
			executeUpdate(con, warnMessage, errorMessage, command, optionalFixedParam, parameters);
		} finally {
			Toolkit.close(con);
		}

	}

	protected void executeUpdate(EncapsulatedConnection con, String warnMessage, String errorMessage, String command,
			String optionalFixedParam, String... parameters) throws USMSQLException {
		PreparedStatement statement = null;
		try {
			statement = con.prepareStatement(command);
			addSessionIdentifierParamsToStmt(statement);
			int i = NUMBER_OF_SESSION_ID_FIELDS + 1;
			if (optionalFixedParam != null)
				statement.setString(i++, optionalFixedParam);
			for (String param : parameters)
				statement.setString(i++, param);
			if (statement.executeUpdate() == 0 && warnMessage.length() > 0)
				_journal.warn(String.valueOf(_session) + ' ' + warnMessage);
		} catch (SQLException e) {
			_journal.error(String.valueOf(_session) + ' ' + errorMessage, e);
			throw new USMSQLException(MappingBundleErrorCodes.UPDATE_FAILED, _session + ": " + errorMessage, e);
		} finally {
			Toolkit.close(statement);
		}
	}

	protected void addSessionIdentifierParamsToStmt(PreparedStatement statement) throws SQLException {
		statement.setInt(1, _session.getContextId());
		statement.setInt(2, _session.getSessionId());
	}

	protected String completeComplexStatement(String prefix, int length) {
		StringBuilder sb = new StringBuilder(prefix);
		for (int i = 0; i < length; i++) {
			if (i > 0)
				sb.append(',');
			sb.append('?');
		}
		return sb.append(')').toString();
	}
}
