/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.mimemail;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.*;

import javax.mail.*;
import javax.mail.internet.*;

import org.json.*;

import com.openexchange.mail.MailException;
import com.openexchange.mail.mime.*;
import com.openexchange.mail.mime.ContentDisposition;
import com.openexchange.mail.mime.ContentType;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;

/**
 * This class converts the JSON structure with information about the mail to the mime mail. 
 * The JAVA Mail API is used for this aim.
 * @author ibr
 *
 */
public class MimeMailBuilder {

	private static final String READ_DATE = "read-date";

	private static final String UTF_8 = "UTF-8";

	private static final String PERSONAL = "personal";

	private static final String DISPOSITION_NOTIFICATION_TO_ADDRESSLIST_HEADER = "disposition-notification-to";
	private static final String RESENT_SENDER_ADDRESSLIST_HEADER = "resent-sender";
	private static final String RESENT_TO_ADDRESSLIST_HEADER = "resent-to";
	private static final String RESENT_FROM_ADDRESSLIST_HEADER = "resent-from";
	private static final String RESENT_CC_ADDRESSLIST_HEADER = "resent-cc";
	private static final String RESENT_BCC_ADDRESSLIST_HEADER = "resent-bcc";
	private static final String ERRORS_TO_ADDRESSLIST_HEADER = "errors-to";
	private static final String SENDER_ADDRESSLIST_HEADER = "sender";
	private static final String REPLY_TO_ADDRESSLIST_HEADER = "reply-to";

	private static final String ADDRESS = "address";

	private static final String CONTENTTYPE_PARAMS = "params";

	private static final String CONTENTTYPE_TYPE = "type";

	private static final String terminalSeq = "\r\n";

	private static final String BODY = "body";
	private static final String DATA = "data";

	private static final String EMPTY_STR = "";

	private static final char SEMICOLON = ';';

	private static final String NAME = "name";

	private static final String CONTENT_TYPE_UPPER = "Content-Type";
	private static final String CONTENT_TRANSFER_ENCODING_UPPER = "Content-Transfer-Encoding";

	private static final String CONTENT_TYPE = "content-type";
	private static final String CONTENT_TRANSFER_ENCODING = "content-transfer-encoding";
	private static final String CONTENT_DISPOSITION = "content-disposition";
	private static final String BASE64_ENCODING = "base64";
	//private static final String SEVENBIT_ENCODING = "7bit";

	private static final String MIME_VERSION = "mime-version";
	private static final String RECEIVED = "received";
	private static final String RETURN_PATH = "return-path";
	private static final String X_MAILER = "x-mailer";
	private static final String X_SIEVE = "x-sieve";

	private static final String FROM_ADDRESSLIST_HEADER = "from";
	private static final String TO_ADDRESSLIST_HEADER = "to";
	private static final String CC_ADDRESSLIST_HEADER = "cc";
	private static final String BCC_ADDRESSLIST_HEADER = "bcc";

	private static final String HEADERS_FIELDNAME = "headers";

	private static final String CONTENTTYPE_MULTIPART_MIXED = "multipart/mixed";
	private static final String CONTENTTYPE_MULTIPART_ALTER = "multipart/alternative";
	private static final String MULTIPART_ALTERNATIVE = "alternative";
	private static final String MULTIPART_MIXED = "mixed";
	private static final String MULTIPART = "multipart/";
	//private static final String CONTENTTYPE_TEXT_PLAIN = "text/plain";
	//private static final String CONTENTTYPE_TEXT_HTML = "text/html";
	private static final String CONTENTTYPE_TEXT_PREFIX = "text/";
	private static final String CONTENTTYPE_TEXT_PLAIN = "plain";
	private static final String CONTENTTYPE_RFC822 = "message/rfc822";
	private static final String CONTENTTYPE_APPLICATION_OCTETSTREAM = "application/octet-stream";

	private static final String CONTENT_DISPOSITION_ATTACHMENT = "ATTACHMENT";

	private static final String CHARSET = "charset";

	private USMMimeMessage _message;

	//----------------------------------------------------------------------------------------------------
	// Begin: with JAVA Mail API

	public String writeTo() throws USMJSONAPIException {
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

		try {
			_message.writeTo(outputStream);
		} catch (IOException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by output of mime mail: " + e.getMessage(), e);

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "messaging error by output of mime mail: " + e.getMessage(), e);
		} finally {
			if (outputStream != null)
				try {
					outputStream.close();
				} catch (IOException e) {
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_IO_ERROR,
							ResponseStatusCode.INTERNAL_ERROR, "IO error by output of mime mail: " + e.getMessage(), e);
				}
		}
		return outputStream.toString();
	}

	public void buildMimeMailFromJSONObject(JSONObject data) throws USMJSONAPIException {
		_message = new USMMimeMessage(Session.getInstance(new Properties()));
		buildPart(_message, data);
		try {
			_message.saveChanges();
		} catch (MessagingException e1) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_SAVE_CHANGE_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by save change of mime mail: " + e1.getMessage());
		}

	}

	public String convertJSONObjectToMimeMail(JSONObject data) throws USMJSONAPIException {
		buildMimeMailFromJSONObject(data);
		return writeTo();
	}

	private MimeMultipart buildMimePartForMultipart(JSONObject data, String contentTypeInfo, String subtype)
			throws MessagingException, JSONException, USMJSONAPIException {

		MimeMultipart multiPart = new MimeMultipart(subtype);
		if (MULTIPART_MIXED.equals(subtype))
			multiPart = new MixedMimeMultipart();

		// create and add parts
		JSONArray parts = data.getJSONArray(BODY);
		if (parts == null)
			return multiPart;

		for (int i = 0; i < parts.length(); i++) {
			JSONObject jsonpart = parts.getJSONObject(i);
			MimeBodyPart part = new MimeBodyPart();
			buildPart(part, jsonpart);
			multiPart.addBodyPart(part);
		}

		return multiPart;
	}

	private void buildPart(MimePart part, JSONObject json) throws USMJSONAPIException {
		try {
			JSONObject headers = json.getJSONObject(HEADERS_FIELDNAME);
			Object contentType = headers.has(CONTENT_TYPE) ? headers.get(CONTENT_TYPE) : CONTENTTYPE_TEXT_PREFIX + CONTENTTYPE_TEXT_PLAIN + "; "  + CHARSET + "=" + UTF_8;
			String contentTypeOfPart = buildContentTypeAsString(headers, contentType);
			if (contentTypeOfPart.startsWith(CONTENTTYPE_MULTIPART_MIXED)) {
				//add headers to message
				appendHeadersToPart(part, headers, CONTENTTYPE_MULTIPART_MIXED);
				// set multi part content
				part.setContent(buildMimePartForMultipart(json, contentTypeOfPart, MULTIPART_MIXED),
						CONTENTTYPE_MULTIPART_MIXED);
			} else if (contentTypeOfPart.startsWith(CONTENTTYPE_MULTIPART_ALTER)) {
				//add headers
				appendHeadersToPart(part, json.getJSONObject(HEADERS_FIELDNAME), CONTENTTYPE_MULTIPART_ALTER);
				// set multi part content
				part.setContent(buildMimePartForAlternative(json, contentTypeOfPart), CONTENTTYPE_MULTIPART_ALTER);
			} else if (contentTypeOfPart.startsWith(MULTIPART)) {
				//add headers to message
				appendHeadersToPart(part, headers, contentTypeOfPart);
				// set multi part content
				String subtype = contentTypeOfPart.replaceFirst(MULTIPART, "");
				part.setContent(buildMimePartForMultipart(json, contentTypeOfPart, subtype), contentTypeOfPart);
			} else {
				JSONObject body = json.has(BODY)? json.getJSONObject(BODY) : null;
				if (headers.has(CONTENT_DISPOSITION)
						&& headers.getString(CONTENT_DISPOSITION).toUpperCase().contains(CONTENT_DISPOSITION_ATTACHMENT)) {
					int posName = contentTypeOfPart.indexOf(NAME);
					String mailcontentTypeAttachment = contentTypeOfPart.substring(0, posName).trim();
					int posSep2 = mailcontentTypeAttachment.indexOf(SEMICOLON);
					if (posSep2 > 0)
						mailcontentTypeAttachment = mailcontentTypeAttachment.substring(0, posSep2);
					//add headers
					appendHeadersToPart(part, headers, contentTypeOfPart);
					// set content
					String content = body != null && body.has(DATA) ? body.getString(DATA) : "";
					String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
					if (BASE64_ENCODING.equals(encoding)) {
						if (mailcontentTypeAttachment.startsWith(CONTENTTYPE_TEXT_PREFIX))
							mailcontentTypeAttachment = CONTENTTYPE_APPLICATION_OCTETSTREAM;
						part.setContent(Toolkit.decodeBase64(content), mailcontentTypeAttachment);
						appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
					} else {
						part.setContent(content, mailcontentTypeAttachment);
					}
					part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
				} else if (contentTypeOfPart.startsWith(CONTENTTYPE_TEXT_PREFIX)) {
					try {
						//add headers
						String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
						appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
						appendHeadersToPart(part, headers, contentTypeOfPart);
						//build text part
						String content = body != null && body.has(DATA) ? body.getString(DATA) : "";
						buildTextPart(part, contentTypeOfPart, content);
					} catch (UnsupportedEncodingException e) {
						throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR,
								ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: "
										+ e.getMessage());

					}
				} else if (contentTypeOfPart.startsWith(CONTENTTYPE_RFC822)) {
					appendHeadersToPart(part, headers, contentTypeOfPart);
					USMMimeMessage nestedPart = new USMMimeMessage(Session.getInstance(new Properties()));
					buildPart(nestedPart, body);
					part.setContent(nestedPart, CONTENTTYPE_RFC822);
				}
			}
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JSONEXCEPTION_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by building of body part: " + e.getMessage());
		}

	}

	private String buildContentTypeAsString(JSONObject headers, Object paramHeader) throws JSONException,
			USMJSONAPIException {
		ContentType contentType = new ContentType();
		try {
			if (paramHeader instanceof String) {
				contentType = new ContentType((String) paramHeader);
			} else if (paramHeader instanceof JSONObject) {
				String type = ((JSONObject) paramHeader).getString(CONTENTTYPE_TYPE);
				contentType.setBaseType(type);
				JSONObject params = (((JSONObject) paramHeader).has(CONTENTTYPE_PARAMS)) ? ((JSONObject) paramHeader)
						.getJSONObject(CONTENTTYPE_PARAMS) : new JSONObject();
				for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
					String key = (String) iterator.next();
					String value = "";
					if (READ_DATE.equals(key)) {
						value = extractDateFromJSONObject(params.getJSONObject(READ_DATE));
					} else {
						value = params.getString(key);
					}
					contentType.addParameter(key, value);
				}
			}
		} catch (MailException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_TYPE_MAIL_EXC,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Mail Exception by add content-type header "
							+ e.getMessage());
		}
		return contentType.toString();
	}

	private String buildContentDispositionAsString(JSONObject headers, Object paramHeader) throws JSONException,
			USMJSONAPIException {
		ContentDisposition contentDisp = new ContentDisposition();
		try {
			if (paramHeader instanceof String)
				contentDisp = new ContentDisposition((String) headers.getString(CONTENT_DISPOSITION));
			else if (paramHeader instanceof JSONObject) {
				String type = ((JSONObject) paramHeader).getString(CONTENTTYPE_TYPE);
				contentDisp.setDisposition(type);
				JSONObject params = (((JSONObject) paramHeader).has(CONTENTTYPE_PARAMS)) ? ((JSONObject) paramHeader)
						.getJSONObject(CONTENTTYPE_PARAMS) : new JSONObject();
				for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
					String key = (String) iterator.next();
					String value = "";
					if (READ_DATE.equals(key)) {
						value = extractDateFromJSONObject(params.getJSONObject(READ_DATE));
					} else {
						value = params.getString(key);
					}
					contentDisp.addParameter(key, value);
				}
			}
		} catch (MailException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_DISP_MAIL_EXC,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Mail Exception by add content-disposition header "
							+ e.getMessage());
		}
		return contentDisp.toString();
	}

	private void buildTextPart(MimePart part, String contentTypeOfPart, String decodedContent) throws JSONException,
			MessagingException, UnsupportedEncodingException {
		int posCharset = contentTypeOfPart.indexOf(CHARSET);
		String mailcontentTypeText = contentTypeOfPart.substring(0, posCharset).trim();
		int posSep2 = mailcontentTypeText.indexOf(SEMICOLON);
		if (posSep2 > 0)
			mailcontentTypeText = mailcontentTypeText.substring(0, posSep2);
		String charset = contentTypeOfPart.substring(posCharset, contentTypeOfPart.length());
		int eqvPos = charset.indexOf('=');
		charset = charset.substring(eqvPos + 1, charset.length()).trim();
		// set content
		int posSubtype = mailcontentTypeText.indexOf('/');
		String subtype = CONTENTTYPE_TEXT_PLAIN;
		if (posSubtype > 0)
			subtype = mailcontentTypeText.substring(posSubtype + 1, mailcontentTypeText.length());
		// set content
		byte[] contentBytes;
		//decode if Content-Transfer-Encoding = Base64
		if (BASE64_ENCODING.equals(part.getEncoding()))
			contentBytes = Toolkit.decodeBase64(decodedContent);
		else
			contentBytes = decodedContent.getBytes(charset);
		part.setText(new String(contentBytes, charset), charset, subtype);
	}

	private MimeMultipart buildMimePartForAlternative(JSONObject data, String contentTypeInfo)
			throws MessagingException, JSONException, USMJSONAPIException {

		MimeMultipart multiPart = new MimeMultipart(MULTIPART_ALTERNATIVE);
		// create and add parts
		JSONArray parts = data.getJSONArray(BODY);
		if (parts == null)
			return multiPart;
		// build simple text parts: plain, html
		for (int i = 0; i < parts.length(); i++) {
			JSONObject jsonpart = parts.getJSONObject(i);
			MimeBodyPart part = new MimeBodyPart();
			JSONObject headers = jsonpart.getJSONObject(HEADERS_FIELDNAME);
			//build text part
			Object contentType = headers.get(CONTENT_TYPE);
			String contentTypeOfPart = buildContentTypeAsString(headers, contentType);

			try {
				//add headers
				part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
				appendHeadersToSimplePart(part, headers);
				//build text
				buildTextPart(part, contentTypeOfPart, jsonpart.getJSONObject(BODY).getString(DATA));
			} catch (UnsupportedEncodingException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR,
						ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: " + e.getMessage());
			}
			// set content
			multiPart.addBodyPart(part);
		}
		return multiPart;
	}

	private void appendHeadersToSimplePart(Part part, JSONObject headers) throws USMJSONAPIException {
		try {
			if (headers.has(MIME_VERSION))
				part.setHeader(MIME_VERSION, getValue(headers, MIME_VERSION, 0, null));

			if (headers.has(CONTENT_DISPOSITION)) {
				String contDispAsString = buildContentDispositionAsString(headers, headers.get(CONTENT_DISPOSITION));
				part.setDisposition(contDispAsString);
			}
			part.setHeader(CONTENT_TRANSFER_ENCODING_UPPER, getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null));
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JSON_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException by add header: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "MessagingException by add header: " + e.getMessage());
		}

	}

	static List<String> specialHeaderNamesList = new ArrayList<String>();
	static {
		specialHeaderNamesList.add(CONTENT_TYPE);
		specialHeaderNamesList.add(CONTENT_DISPOSITION);
		specialHeaderNamesList.add(CONTENT_TRANSFER_ENCODING);
		specialHeaderNamesList.add(RECEIVED);
		specialHeaderNamesList.add(RETURN_PATH);
		specialHeaderNamesList.add(X_SIEVE);
		specialHeaderNamesList.add(MIME_VERSION);
		specialHeaderNamesList.add(X_MAILER);
	}
	static List<String> knownAddressHeaderNamesList = new ArrayList<String>();

	static {
		knownAddressHeaderNamesList.add(FROM_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(CC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(BCC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(REPLY_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(SENDER_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(ERRORS_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_BCC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_CC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_FROM_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_SENDER_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(DISPOSITION_NOTIFICATION_TO_ADDRESSLIST_HEADER);
	}

	private void appendHeadersToPart(Part part, JSONObject headers, String contentType) throws USMJSONAPIException {
		try {

			part.setHeader(CONTENT_TYPE_UPPER, contentType);
			if (headers.has(CONTENT_DISPOSITION)) {
				String contDispAsString = buildContentDispositionAsString(headers, headers.get(CONTENT_DISPOSITION));
				part.setDisposition(contDispAsString);
			}
			appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, getValue(headers, CONTENT_TRANSFER_ENCODING,
					0, null));
			String received = getValue(headers, RECEIVED, 1, null);
			String received2 = getValue(headers, RECEIVED, 0, null);
			String returnPath = getValue(headers, RETURN_PATH, 0, null);
			String xSieve = getValue(headers, X_SIEVE, 0, null);

			appendHeaderConditional(part, RETURN_PATH, returnPath);

			if (headers.has(RECEIVED))
				appendHeaderConditional(part, RECEIVED, formatReceived(received));

			appendHeaderConditional(part, X_SIEVE, xSieve);

			if (headers.has(RECEIVED))
				appendHeaderConditional(part, RECEIVED, formatReceived(received2));

			if (headers.has(MIME_VERSION))
				appendHeaderConditional(part, MIME_VERSION, getValue(headers, MIME_VERSION, 0, EMPTY_STR));

			if (headers.has(X_MAILER))
				appendHeaderConditional(part, X_MAILER, getValue(headers, X_MAILER, 0, EMPTY_STR));

			//add additional headers
			Iterable<String> headerNames = JSONToolkit.keys(headers);
			for (String headerName : headerNames) {
				if (knownAddressHeaderNamesList.contains(headerName)) {
					appendHeaderConditional(part, headerName, getAddress(headers, headerName, null));
				} else if (!specialHeaderNamesList.contains(headerName)) {
					appendHeaderConditional(part, headerName, getValue(headers, headerName, 0, null));
				}
			}

		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JSON_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException by add header: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "MessagingException by add header: " + e.getMessage());
		}
	}

	private String getValue(JSONObject headers, String key, int i, String defaultVal) throws JSONException {
		if (!headers.has(key))
			return defaultVal;
		Object value = headers.get(key);
		if (value == null)
			return defaultVal;
		if (value instanceof String)
			return (String) value;
		else if (value instanceof JSONArray) {
			JSONArray array = (JSONArray) value;
			if (i >= array.length())
				return defaultVal;
			try {
				return array.getString(i);
			} catch (JSONException e) {
				return defaultVal;
			}
		} else if (value instanceof JSONObject) {
			String v = extractDateFromJSONObject((JSONObject) value);
			return (v == null) ? value.toString() : v;
		}
		return defaultVal;
	}

	private String getAddress(JSONObject headers, String key, String defaultVal) throws JSONException,
			USMJSONAPIException {
		if (!headers.has(key))
			return defaultVal;
		Object value = headers.get(key);
		if (value == null)
			return defaultVal;
		if (value instanceof String)
			return value.toString();
		else if (value instanceof JSONArray) {
			JSONArray array = (JSONArray) value;
			if (0 >= array.length())
				return defaultVal;
			try {
				StringBuilder address = new StringBuilder();
				for (int j = 0; j < array.length(); j++) {
					String addressPart = array.getJSONObject(j).getString(ADDRESS);
					String personalPart = array.getJSONObject(j).optString(PERSONAL, null);
					QuotedInternetAddress ia = new QuotedInternetAddress(addressPart, personalPart, UTF_8);
					address.append(ia.toString());
					if (j < array.length() - 1)
						address.append(",");
				}
				return address.toString();
			} catch (JSONException e) {
				return defaultVal;
			} catch (AddressException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADDRESS_EXCEPTION,
						ResponseStatusCode.INTERNAL_ERROR, "mail address exception: " + e.getMessage());
			} catch (UnsupportedEncodingException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_2,
						ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: " + e.getMessage());
			}

		}
		return defaultVal;
	}

	private void appendHeaderConditional(Part part, String header, String value) throws MessagingException {
		if (value != null && !"null".equals(value)) {
			try {
				part.setHeader(header, MimeUtility.encodeText(value, UTF_8, null));
			} catch (UnsupportedEncodingException e) {
				part.setHeader(header, value);
			}
		}
	}

	private String extractDateFromJSONObject(JSONObject dateObject) {
		try {
			if (dateObject.has("date"))
				return dateObject.getString("date");
			if (dateObject.has("utc")) {
				//				DateFormat df = DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.LONG);
				//				return df.format(new Date(dateObject.getLong("utc")));
				return new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z").format(new Date(dateObject.getLong("utc")));
			}
		} catch (JSONException e) {
		}
		return null;
	}

	private String formatReceived(String rec) {
		if (rec.length() < 2)
			return rec;
		StringBuilder buffer = new StringBuilder(60);
		int posBy = rec.indexOf("by");
		int posFor = rec.indexOf("for");
		if (posFor > 0) {

			buffer.append(rec.substring(0, posBy - 1)).append(terminalSeq);
			buffer.append("	").append(rec.substring(posBy, posFor - 1)).append(terminalSeq);
			buffer.append("	").append(rec.substring(posFor, rec.length()));
		} else {
			int posWith = rec.indexOf("with ");
			if (posWith < 0)
				return rec;
			int pos = rec.indexOf(SEMICOLON, posWith);
			if (pos < 0 || (pos + 1) >= rec.length())
				return rec;
			buffer.append(rec.substring(0, posBy - 1)).append(terminalSeq);
			buffer.append("	").append(rec.substring(posBy, pos + 1)).append(terminalSeq);
			buffer.append("	").append(rec.substring(pos + 2, rec.length()));
		}
		return buffer.toString();
	}

	//----------------------------------------------------------------------------------------------------
	//End	

}
