/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json;

import java.io.*;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.*;

import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;

import com.openexchange.usm.api.contenttypes.ContentTypeManager;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.configuration.*;
import com.openexchange.usm.connector.commands.CommandHandler;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.ox_json.OXJSONPropertyNames;
import com.openexchange.usm.util.Toolkit;

public class USMJSONServlet extends HttpServlet {

	private static final int ONE_SECOND = 1000;

	private static final String COMMAND = "action";

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final transient JSONSessionInitializer _sessionInitializer = new JSONSessionInitializer(this);

	private transient SessionManager _sessionManager;

	private transient ContentTypeManager _contentTypeManager;

	private transient Log _journal;

	private transient HttpService _httpService;

	private transient ConfigurationManager _configurationManager;

	private transient String _servletAlias;

	private transient long[] _pingTimestamps;

	private transient int _pingTimestampsIndex = -1;

	private transient int _minPingInterval;

	private transient int _maxPingInterval;

	private transient int _maxPingRequestsPerSecond;

	private transient int _pingSampleSize;

	/**
	 * the activate method; register the httpServer object, set the Servlet alias
	 * @param sessionManager
	 * @param contentTypeManager
	 * @param log
	 * @param httpService
	 * @param configurationManager
	 */
	public void activate(SessionManager sessionManager, ContentTypeManager contentTypeManager, Log log,
			HttpService httpService, ConfigurationManager configurationManager) {
		_sessionManager = sessionManager;
		_contentTypeManager = contentTypeManager;
		_journal = log;
		_httpService = httpService;
		_configurationManager = configurationManager;
		try {
			_servletAlias = _configurationManager.getProperty(ConfigurationProperties.JSON_SERVLET_ALIAS_PROPERTY,
					ConfigurationProperties.JSON_SERVLET_ALIAS_DEFAULT, true);
			_minPingInterval = _configurationManager.getProperty(
					ConfigurationProperties.JSON_SERVLET_PING_MIN_INTERVAL,
					ConfigurationProperties.JSON_SERVLET_PING_MIN_INTERVAL_DEFAULT, true);
			_maxPingInterval = _configurationManager.getProperty(
					ConfigurationProperties.JSON_SERVLET_PING_MAX_INTERVAL,
					ConfigurationProperties.JSON_SERVLET_PING_MAX_INTERVAL_DEFAULT, true);
			_maxPingRequestsPerSecond = _configurationManager.getProperty(
					ConfigurationProperties.JSON_SERVLET_PING_MAX_REQUEST_PER_SECOND,
					ConfigurationProperties.JSON_SERVLET_PING_MAX_REQUEST_PER_SECOND_DEFAULT, true);
			_pingSampleSize = (int) Math.ceil(ONE_SECOND / (double) _maxPingRequestsPerSecond);
			_pingTimestamps = new long[_pingSampleSize * _maxPingRequestsPerSecond];
		} catch (USMInvalidConfigurationException e1) {
			logAndThrowStartupError(ConnectorBundleErrorCodes.INVALID_ALIAS_FOR_SERVLET,
					ConfigurationProperties.JSON_SERVLET_ALIAS_PROPERTY, e1);
		}

		try {
			_httpService.registerServlet(_servletAlias, this, null, _httpService.createDefaultHttpContext());
			_journal.info("Connector servlet registered under alias " + _servletAlias);
			_journal.info("Open-Xchange USM/JSON Version " + USMJSONVersion.VERSION + " (Build "
					+ USMJSONVersion.BUILD_NUMBER + ") successfully started");
		} catch (ServletException e) {
			logAndThrowStartupError(ConnectorBundleErrorCodes.SERVLET_REGISTRATION_FAILED,
					"Servlet registration failed", e);
		} catch (NamespaceException e) {
			logAndThrowStartupError(ConnectorBundleErrorCodes.SERVLET_ALIAS_IN_USE, "Servlet alias already in use:"
					+ _servletAlias, e);
		}
	}

	private void logAndThrowStartupError(int errorCode, String detailMessage, Throwable cause) {
		String message = "Open-Xchange USM-Connector couldn't be started: " + detailMessage;
		_journal.error(message, cause);
		deactivate();
		throw new USMStartupException(errorCode, message, cause);
	}

	/**
	 * unregister http service, set the servlet alias to null
	 */
	public void deactivate() {
		if (_servletAlias != null) {
			_httpService.unregister(_servletAlias);
			_journal.info("Connector servlet unregistered from alias " + _servletAlias);
			_servletAlias = null;
		} else {
			_journal.info("Connector servlet unregistered");
		}
		_sessionManager = null;
		_contentTypeManager = null;
		_configurationManager = null;
		_httpService = null;
		_journal = null;
		_pingTimestamps = null;
	}

	@Override
	protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		long startTime = System.currentTimeMillis();
		String command = getCommandName(request);
		if (_journal.isDebugEnabled())
			_journal.debug("HTTP PUT " + command);
		try {
			ResponseObject result;
			CommandHandler handler = null;
			boolean printedError = false;
			try {
				JSONObject parameters = extractJSONObjectFromRequest(request);
				handler = CommandDelegateFactory.createCommandHandler(this, request.getSession(), command, parameters);
				result = handler.handleRequest();
			} catch (USMJSONAPIException e) {
				result = new ResponseObject(e);
			} catch (Exception e) {
				JSONObject errorDetailsObject = new JSONObject();
				USMException.appendStacktrace(e, errorDetailsObject);
				result = new ResponseObject(ResponseStatusCode.INTERNAL_ERROR,
						ConnectorBundleErrorCodes.EXCEPTION_IN_SERVLET, e.getMessage(), errorDetailsObject, null);
				_journal.error(computeLogMessage(command, handler, result), e);
				printedError = true;
			}
			switch (result.getStatusCode()) {
				case SUCCESS:
				case ACCESS_DENIED:
				case BAD_REQUEST:
				case SYNC_FAILED:
				case UNKNOWN_SESSION:
				case UNKNOWN_SYNCID:
				case UNKNOWN_UUID:
				case WRONG_MISSING_PARAMETERS:
				case UNSUPPORTED_VERSION:
					if (_journal.isDebugEnabled()) {
						String message = computeLogMessage(command, handler, result);
						USMJSONException e = result.getCause();
						if (e instanceof Throwable) {
							_journal.debug(message, (Throwable) e);
						} else {
							_journal.debug(message);
						}
					}
					break;
				case DATABASE_ERROR:
				case INTERNAL_ERROR:
				case OX_SERVER_ERROR:
				default:
					if (!printedError) {
						String message = computeLogMessage(command, handler, result);
						USMJSONException e = result.getCause();
						if (e instanceof Throwable) {
							_journal.error(message, (Throwable) e);
						} else {
							_journal.error(result.toString(3));
							_journal.error(message);
						}
					}
					break;
			}
			byte[] data = result.toString().getBytes(ServletConstants.UTF_ENCODING_8);
			response.setContentType(ServletConstants.TEXT_JAVASCRIPT);
			response.setContentLength(data.length);
			ServletOutputStream os = response.getOutputStream();
			os.write(data, 0, data.length);
			os.flush();
			os.close();
			if (_journal.isDebugEnabled())
				_journal.debug(generateTimeLog(handler, command, System.currentTimeMillis() - startTime));
		} catch (Exception error) {
			_journal.error("Open-Xchange USM/JSON: Internal server error in " + command, error);
			response.sendError(HttpStatus.SC_INTERNAL_SERVER_ERROR);
		}
	}

	private String computeLogMessage(String command, CommandHandler handler, ResponseObject result) {
		Session session = (handler == null) ? null : handler.getSession();
		StringBuilder sb = new StringBuilder();
		sb.append(session).append(' ').append(command).append(' ').append(result.getStatusCode());
		if (result.getStatusCode() != ResponseStatusCode.SUCCESS) {
			sb.append(' ').append(result.getErrorCode());
			if (result.getErrorMessage() != null)
				sb.append(' ').append(result.getErrorMessage());
		}
		return sb.toString();
	}

	private String getCommandName(HttpServletRequest request) {
		String[] result = request.getParameterValues(COMMAND);
		if (result != null && result.length == 1)
			return result[0];
		String command = request.getPathInfo();
		if (command.length() > 0 && command.charAt(0) == '/')
			command = command.substring(1);
		// XXX This is only a temporary workaround for a bug in the OX ServletContainer
		//		int index = command.lastIndexOf('/');
		//		if (index >= 0)
		//			command = command.substring(index + 1);
		return command;
	}

	private JSONObject extractJSONObjectFromRequest(HttpServletRequest request) throws USMJSONAPIException {
		// Only use the content-type before a ";", ignore any extra information
		String contentType = request.getContentType();
		if (contentType != null) {
			int semiColon = contentType.indexOf(';');
			if (semiColon >= 0)
				contentType = contentType.substring(0, semiColon);
		}
		if (!ServletConstants.TEXT_JAVASCRIPT.equals(contentType))
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.BAD_REQUEST, ResponseStatusCode.BAD_REQUEST,
					"Request type not \"text/javascript\"");
		//read the JSON object from the request
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(
					new InputStreamReader(request.getInputStream(), ServletConstants.UTF_ENCODING_8));
			char[] buffer = new char[4096];
			StringBuilder sb = new StringBuilder(4096);
			for (int count = reader.read(buffer); count >= 0; count = reader.read(buffer))
				sb.append(buffer, 0, count);
			JSONObject requestObject = new JSONObject(sb.toString());
			return requestObject;
		} catch (UnsupportedEncodingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.UTF_8_NOT_SUPPORTED,
					ResponseStatusCode.INTERNAL_ERROR, "UTF-8 Encoding is not supported");
		} catch (IOException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.ERROR_ON_READ_REQUEST,
					ResponseStatusCode.BAD_REQUEST, "Error while reading request");
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.BAD_JSONOBJECT_IN_REQUEST,
					ResponseStatusCode.BAD_REQUEST, "Invalid JSONObject as parameter", e);
		} finally {
			Toolkit.close(reader);
		}
	}

	public SessionManager getSessionManager() {
		return _sessionManager;
	}

	public ContentTypeManager getContentTypeManager() {
		return _contentTypeManager;
	}

	public Log getJournal() {
		return _journal;
	}

	public ConfigurationManager getConfigurationManager() {
		return _configurationManager;
	}

	public Session getSession(String user, String password, String device) throws USMException {
		return _sessionManager.getSession(user, password, ServletConstants.PROTOCOL, device, _sessionInitializer);
	}

	public long getPingInterval() {
		int nextIndex = _pingTimestampsIndex + 1 >= _pingTimestamps.length ? 0 : _pingTimestampsIndex + 1;
		if (_pingTimestamps[nextIndex] == 0)
			return _minPingInterval;
		long deltaT = _pingTimestamps[_pingTimestampsIndex] - _pingTimestamps[nextIndex];
		if (deltaT < 1)
			deltaT = 1;
		if (deltaT >= ONE_SECOND * _pingSampleSize)
			return _minPingInterval;
		else {
			long newInterval = _minPingInterval * ONE_SECOND / deltaT;
			return newInterval > _maxPingInterval ? _maxPingInterval : newInterval;
		}
	}

	public void newPingRequestArrived() {
		_pingTimestampsIndex++;
		if (_pingTimestampsIndex == _pingTimestamps.length)
			_pingTimestampsIndex = 0;
		_pingTimestamps[_pingTimestampsIndex] = System.currentTimeMillis();
	}

	private String generateTimeLog(CommandHandler handler, String command, long duration) {
		Session session = (handler == null) ? null : handler.getSession();
		if (session == null)
			return "No session: " + command + " executed in " + duration + " ms.";
		Object numberOfCalls = session.getCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS);
		Object accumulatedTime = session.getCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME);
		if (numberOfCalls == null || accumulatedTime == null)
			return session + ": " + command + " executed in " + duration + " ms, no OX calls made.";
		return session + ": " + command + " executed in " + duration + " ms, " + numberOfCalls + " OX calls took "
				+ accumulatedTime + " ms.";
	}
}
