/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;

import java.util.*;

import javax.servlet.http.HttpSession;

import org.json.JSONArray;
import org.json.JSONObject;

import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.json.*;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.FolderHierarchyComparator;

/**
 * Handler for the SyncUpdate USM-JSON-Command.
 * @author ldo
 *
 */
public class SyncUpdateHandler extends SyncCommandHandler {
	private final static String[] REQUIRED_PARAMETERS = { SESSIONID, SYNCID };
	private final static String[] OPTIONAL_PARAMETERS = {
			FOLDERID,
			CREATED,
			MODIFIED,
			DELETED,
			LIMIT,
			CONFLICT_RESOLUTION };

	private Folder _folderToSync;
	private long _syncID;

	public SyncUpdateHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters)
			throws USMJSONAPIException {
		super(servlet, httpSession, parameters);
	}

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		String folderUUID = getStringParameter(FOLDERID, null);
		int limit = getOptionalLimit();
		ConflictResolution conflictResolution = getOptionalConflictResolution();
		_syncID = getSyncID();
		SyncResult result = computeSyncUpdate(folderUUID, _syncID, limit, conflictResolution);
		long syncId = result.getTimestamp();
		Folder folder = null;
		if (folderUUID != null)
			folder = getFolderByUUID(folderUUID);
		if (_recurrenceUUIDMap.size() > 0 && folder != null) {
			//make additional update to retrieve all created exception objects on the server,
			//and replace the uuids created on the server with the original uuids sent from the client  
			retrieveExceptionsFromServerAndSetUUIDs(syncId, folder);
		}
		syncId = makeDelayedCallsAndSaveNewState(limit, result, folder, syncId);
		JSONObject response = createResponseFromSyncResults(result, syncId);
		return new ResponseObject(ResponseStatusCode.SUCCESS, response);
	}

	@Override
	protected DataObject getCachedElement(String id) throws DatabaseAccessException, USMSQLException {
		if (_cachedElements == null && _folderToSync != null)
			_cachedElements = new DataObjectSet(_session.getCachedFolderElements(_folderToSync.getID(), _folderToSync
					.getElementsContentType(), _syncID));
		return super.getCachedElement(id);
	}

	private void retrieveExceptionsFromServerAndSetUUIDs(long syncId, Folder folder) throws USMJSONAPIException {
		try {
			for (Map.Entry<DataObject, UUID> entry : _recurrenceUUIDMap.entrySet()) {
				if (entry.getKey().getChangeState() != ChangeState.MODIFIED) {
					_newStateToSave = getNewStateToSave(folder, syncId, entry.getKey(), entry.getValue());
				}
			}
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_ERROR_APP_EXCEPTION, e);
		}
	}

	private SyncResult computeSyncUpdate(String folderID, long syncId, int limit, ConflictResolution conflictResolution)
			throws USMJSONAPIException {
		try {
			if (folderID == null) {
				//update folder hierarchy
				Folder[] folderArray = createFolderChangesArray(syncId);
				Arrays.sort(folderArray, new FolderHierarchyComparator(folderArray));
				return _session.syncFolderChangesWithServer(syncId, limit, null, true, conflictResolution, folderArray);
			}
			_folderToSync = getFolderByUUID(folderID);
			_clientChangesArray = createElementChangesArray(_folderToSync, syncId, conflictResolution);
			if (_syncIdForUpdate > 0) {
				syncId = _syncIdForUpdate;
				_syncIdForUpdate = 0L;
			}
			return _session.syncChangesWithServer(_folderToSync.getID(), syncId, limit, null, true, conflictResolution,
					_clientChangesArray);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (SlowSyncRequiredException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SYNC_UPDATE_INVALID_SYNCID,
					ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID "
							+ syncId
							+ " for "
							+ (folderID == null ? "folder hierarchy" : ((_folderToSync != null) ? (_folderToSync
									.getID()
									+ "/" + folderID) : folderID)));
		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_INTERNAL_ERROR, e);
		}
	}

	private Folder[] createFolderChangesArray(long syncId) throws DatabaseAccessException, USMSQLException,
			USMJSONAPIException, SlowSyncRequiredException {
		Folder[] originalFolders = _session.getCachedFolders(syncId);
		if (originalFolders == null)
			throw new SlowSyncRequiredException(-1, null);
		_allFolders = originalFolders;
		Folder[] createdFolders = readCreatedFolders(originalFolders);
		Folder[] modifiedFolders = readModifiedFolders(originalFolders);
		Folder[] deletedFolders = readDeletedFolders(originalFolders);
		Folder[] result = new Folder[createdFolders.length + modifiedFolders.length + deletedFolders.length];
		int index = 0;
		for (Folder folder : createdFolders) {
			result[index++] = folder;
		}
		for (Folder folder : modifiedFolders) {
			result[index++] = folder;
		}
		for (Folder folder : deletedFolders) {
			result[index++] = folder;
		}
		return result;
	}

	private DataObject[] createElementChangesArray(Folder folder, long syncId, ConflictResolution conflictResolution)
			throws USMException {
		DataObject[] originalElements = _session.getCachedFolderElements(folder.getID(), folder
				.getElementsContentType(), syncId);
		if (originalElements == null)
			throw new SlowSyncRequiredException(-1, null);
		DataObject[] createdElements = readCreatedFolderElements(folder, syncId, conflictResolution);
		DataObject[] modifiedElements;
		ContentType elementsContentType = folder.getElementsContentType();
		if (elementsContentType == null)
			return EMPTY_DATAOBJECT_ARRAY;
		if (elementsContentType.getCode() == DefaultContentTypes.MAIL_CODE) {
			modifiedElements = readModifiedMailDataObjects(folder, elementsContentType, originalElements, syncId);
		} else {
			modifiedElements = readModifiedFolderElements(folder, originalElements, createdElements);
		}
		DataObject[] deletedElements = readDeletedFolderElements(folder, originalElements);
		DataObject[] result = new DataObject[createdElements.length + modifiedElements.length + deletedElements.length];
		int index = 0;
		for (DataObject object : createdElements) {
			result[index++] = object;
		}
		for (DataObject object : modifiedElements) {
			result[index++] = object;
		}
		for (DataObject object : deletedElements) {
			result[index++] = object;
		}
		return result;
	}

	private DataObject[] readModifiedMailDataObjects(Folder folder, ContentType contentType,
			DataObject[] originalObjects, long syncId) throws USMException {
		if (!_parameters.has(MODIFIED))
			return EMPTY_DATAOBJECT_ARRAY;
		JSONArray list = getJSONArray(_parameters, MODIFIED);
		int size = list.length();
		List<DataObject> result = new ArrayList<DataObject>(size);
		DataObject[] syncStateToStore = new DataObject[originalObjects.length];
		syncStateToStore = originalObjects;
		for (int i = 0; i < size; i++) {
			JSONObject data = getJSONObject(list, i);
			DataObject object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, originalObjects);
			_syncIdForUpdate = updateMailObject(folder, contentType, originalObjects, syncId, result, data, object,
					_syncIdForUpdate, syncStateToStore);
		}
		_syncIdForUpdate = _session.storeSyncState(getOriginalSyncID(), syncId, folder.getID(), syncStateToStore);
		DataObject[] savedElements = _session.getCachedFolderElements(folder.getID(), contentType, _syncIdForUpdate);
		for (int i = 0; i < size; i++) {
			JSONObject data = getJSONObject(list, i);
			DataObject object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, savedElements);
			result.add(object);
		}
		DataObject[] resultArray = new DataObject[result.size()];
		resultArray = result.toArray(resultArray);
		return resultArray;
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

	@Override
	protected long getOriginalSyncID() {
		return _syncID;
	}
}
