/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;

import java.util.*;

import javax.servlet.http.HttpSession;

import org.json.*;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.json.*;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.mimemail.MimeMailBuilder;
import com.openexchange.usm.session.dataobject.*;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;

public abstract class SyncCommandHandler extends NormalCommandHandler {

	protected final static DataObject[] EMPTY_DATAOBJECT_ARRAY = new DataObject[0];

	protected Map<DataObject, UUID> _recurrenceUUIDMap = new HashMap<DataObject, UUID>();
	protected Map<DataObject, UUID> _exceptionChangesForDelayedCall = new HashMap<DataObject, UUID>();
	protected DataObject[] _clientChangesArray;

	private List<JSONObject> _mailDiffsClientAndServer = new ArrayList<JSONObject>();
	private Map<String, PIMAttachments> _clientAttachmentsMap = new HashMap<String, PIMAttachments>();

	protected DataObjectSet _newStateToSave;
	private boolean _newStateToSaveChanged = false;

	protected long _syncIdForUpdate = 0L;

	protected DataObjectSet _cachedElements;

	public SyncCommandHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters)
			throws USMJSONAPIException {
		super(servlet, httpSession, parameters);
	}

	protected Folder[] readCreatedFolders(Folder[] originalFolders) throws USMJSONAPIException {
		return readFolders(CREATED, originalFolders, true);
	}

	protected Folder[] readModifiedFolders(Folder[] originalFolders) throws USMJSONAPIException {
		return readFolders(MODIFIED, originalFolders, false);
	}

	protected Folder[] readDeletedFolders(Folder[] originalFolders) throws USMJSONAPIException {
		return convertToFolderArray(readDeletedDataObjects(getFolderContentType(), originalFolders));
	}

	protected Folder[] readFolders(String key, Folder[] originalFolders, boolean newObjects) throws USMJSONAPIException {
		return convertToFolderArray(readDataObjects(null, getFolderContentType(), key, originalFolders, newObjects));
	}

	protected DataObject[] readCreatedFolderElements(Folder folder, long syncId, ConflictResolution conflictResolution)
			throws USMJSONAPIException {
		ContentType elementsContentType = folder.getElementsContentType();
		if (elementsContentType == null)
			return EMPTY_DATAOBJECT_ARRAY;
		DataObject[] result = (elementsContentType.getCode() == DefaultContentTypes.MAIL_CODE) ? readCreatedMailDataObjects(
				folder, elementsContentType, conflictResolution, syncId)
				: readDataObjects(folder, elementsContentType, CREATED, null, true);
		for (DataObject o : result)
			o.setParentFolder(folder);
		return result;
	}

	protected DataObject[] readModifiedFolderElements(Folder folder, DataObject[] originalElements,
			DataObject[] creations) throws USMJSONAPIException {
		DataObject[] modifications = readDataObjects(folder, folder.getElementsContentType(), MODIFIED,
				originalElements, false);
		return _recurrenceUUIDMap.isEmpty() ? modifications : addModificationsForExceptions(modifications,
				originalElements, creations);
	}

	// TODO Optimize ?
	protected DataObject[] addModificationsForExceptions(DataObject[] originalModifications,
			DataObject[] originalElements, DataObject[] creations) throws USMJSONAPIException {
		List<DataObject> modifications = new ArrayList<DataObject>(originalModifications.length
				+ _recurrenceUUIDMap.size());
		for (int i = 0; i < originalModifications.length; i++) {
			modifications.add(originalModifications[i]);
		}
		List<DataObject> toRemoveFromReccurenceMap = new ArrayList<DataObject>();

		for (DataObject exceptionObject : _recurrenceUUIDMap.keySet()) {
			UUID seriesUUID = _recurrenceUUIDMap.get(exceptionObject);
			boolean delayed = false;
			DataObject seriesObjectFromExistingElements;
			try {
				seriesObjectFromExistingElements = getSeriesObjectByUUID(originalElements, seriesUUID);
				if (seriesObjectFromExistingElements.getChangeState() != ChangeState.UNMODIFIED) {
					delayed = true;
				}
			} catch (USMJSONAPIException e) {
				seriesObjectFromExistingElements = getSeriesObjectByUUID(creations, seriesUUID); //find the series object from the current client creations 
				delayed = true;
			}
			String serieID = seriesObjectFromExistingElements.getID();
			DataObject seriesObject = seriesObjectFromExistingElements.createCopy(false);
			seriesObject.commitChanges(); //copy is needed because there may be more exception created in one call
			for (int i = 0; i < exceptionObject.getContentType().getFields().length; i++) {
				seriesObject.setFieldContent(i, exceptionObject.getFieldContent(i));
			}
			seriesObject.setID(serieID);
			seriesObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(serieID));

			if (delayed) {
				seriesObject.setUUID(exceptionObject.getUUID());
				_exceptionChangesForDelayedCall.put(seriesObject, seriesUUID);
				toRemoveFromReccurenceMap.add(exceptionObject);
			} else {
				exceptionObject.setID(serieID);
				exceptionObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(serieID));
				modifications.add(seriesObject);
			}
		}
		for (DataObject dataObject : toRemoveFromReccurenceMap) {
			_recurrenceUUIDMap.remove(dataObject); //remove from recurrence_map all exception which were delayed for next sync calls
		}
		return modifications.toArray(new DataObject[modifications.size()]);
	}

	protected long makeDelayedCallsAndSaveNewState(int limit, SyncResult result, Folder folder, long syncId)
			throws USMJSONAPIException {
		if (folder == null || folder.getElementsContentType() == null
				|| folder.getID().equals(folder.getElementsContentTypeID())) //is Dummy content type folder
			return syncId;

		makeDelayedSyncForExceptions(syncId, limit, folder);

		if (_newStateToSave == null) {
			try {
				// TODO Depending on syncId, we might already have the cached folder elements somewhere
				_newStateToSave = new DataObjectSet(_session.getCachedFolderElements(folder.getID(), folder
						.getElementsContentType(), syncId));
			} catch (USMException e) {
				throw USMJSONAPIException.createInternalError(
						ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_RETRIEVE_CACHED_ELEMENTS, e);
			}
		}

		_newStateToSaveChanged = false; //needed because the PIMAttachments are always compared just by timestamp, and we make changes also to the attachments list and keep the old timestamp
		handlePIMAttachments(folder, result, syncId);

		if (_newStateToSaveChanged || !_newStateToSave.isEqualTo(result.getNewState()))
			try {
				syncId = _session
						.storeSyncState(getOriginalSyncID(), syncId, folder.getID(), _newStateToSave.toArray());
			} catch (USMException e) {
				throw USMJSONAPIException.createInternalError(
						ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_SAVE_SYNC_STATE, e);
			}
		return syncId;
	}

	protected void makeDelayedSyncForExceptions(long syncID, int limit, Folder folder) throws USMJSONAPIException {
		if (_exceptionChangesForDelayedCall.isEmpty())
			return;
		try {
			ContentType elementsContentType = folder.getElementsContentType();
			for (DataObject exception : _exceptionChangesForDelayedCall.keySet()) {
				// TODO XXX OPTIMIZE THIS !
				DataObject[] currentFolderElements = elementsContentType.getTransferHandler().readFolderContent(folder,
						_session.getFieldFilter(elementsContentType));
				for (DataObject dataObject : currentFolderElements) {
					_session.insertStoredUUID(dataObject);
				}
				UUID recurrenceUUID = _exceptionChangesForDelayedCall.get(exception);
				DataObject exceptionToSend = refreshExceptionFromCachedElements(currentFolderElements, exception,
						recurrenceUUID);
				if (exceptionToSend != null) {
					elementsContentType.getTransferHandler().writeUpdatedDataObject(exceptionToSend,
							exceptionToSend.getTimestamp());
					_newStateToSave = getNewStateToSave(folder, syncID, exceptionToSend, recurrenceUUID);
				}
			}
			return;
		} catch (SlowSyncRequiredException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SYNC_UPDATE_INVALID_SYNCID_2,
					ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_INTERNAL_ERROR_2, e);
		}
	}

	/**
	 * New state to save in case of delayed calls for appointment exceptions.
	 * @param folder
	 * @param syncID
	 * @param exceptionToSend
	 * @param seriesUUID
	 * @return
	 * @throws USMException
	 */
	protected DataObjectSet getNewStateToSave(Folder folder, long syncID, DataObject exceptionToSend, UUID seriesUUID)
			throws USMException {
		// TODO Might need optimization
		SyncResult syncResult = _session.syncChangesWithServer(folder.getID(), syncID, Session.NO_LIMIT, null, false,
				null);
		setTheClientUUIDsInObjectsCreatedOnServer(syncResult, exceptionToSend);
		DataObjectSet oldState;
		if (_newStateToSave == null || _newStateToSave.size() == 0)
			oldState = new DataObjectSet(_session.getCachedFolderElements(folder.getID(), folder
					.getElementsContentType(), syncID));
		else
			oldState = _newStateToSave;
		DataObjectSet result = new DataObjectSet(oldState);
		for (DataObject dataObject : syncResult.getNewState()) {
			if (exceptionToSend.getUUID().equals(dataObject.getUUID()) || seriesUUID.equals(dataObject.getUUID())) {
				//remove the old state of this object if existing
				result.remove(dataObject.getUUID());
				result.add(dataObject);
			}
		}
		return result;
	}

	private void setTheClientUUIDsInObjectsCreatedOnServer(SyncResult result, DataObject exception)
			throws USMException, USMJSONAPIException {
		DataObject o = findDataObjectWithMatchingRecurrenceIdAndStartDate(exception, result.getNewState());
		if (o != null) {
			try {
				_session.storeUUID(o.getContentType(), Integer.parseInt(o.getID()), exception.getUUID());
			} catch (NumberFormatException e) {
				throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.SYNC_UPDATE_NUMBER_ERROR,
						new USMException(ConnectorBundleErrorCodes.SYNC_UPDATE_NUMBER_ERROR, e));
			}
			o.setUUID(exception.getUUID());
		}
	}

	private boolean matchRecurrenceIdAndStartDate(DataObject o1, DataObject o2) {
		Object recurrence_id1 = o1.getFieldContent("recurrence_id");
		if (recurrence_id1 == null)
			return false;
		Object startDate1 = o1.getFieldContent("start_date");
		if (startDate1 == null)
			return false;
		Object recurrence_id2 = o2.getFieldContent("recurrence_id");
		if (!recurrence_id1.equals(recurrence_id2))
			return false;
		Object startDate2 = o2.getFieldContent("start_date");
		return startDate1.equals(startDate2);
	}

	private DataObject findDataObjectWithMatchingRecurrenceIdAndStartDate(DataObject o, DataObject[] list) {
		for (DataObject object : list) {
			if (matchRecurrenceIdAndStartDate(object, o))
				return object;
		}
		return null;
	}

	private DataObject refreshExceptionFromCachedElements(DataObject[] cachedFolderElements, DataObject exception,
			UUID recurrenceUUID) {
		// TODO Optimize this, do not iterate over all elements (depending on optimizations of makeDelayedSyncForExceptions)
		for (int i = 0; i < cachedFolderElements.length; i++) {
			if (recurrenceUUID != null && cachedFolderElements[i] != null
					&& recurrenceUUID.equals(cachedFolderElements[i].getUUID())) {
				DataObject temp = DataObjectUtil.copyAndModify(cachedFolderElements[i], exception, false);
				temp.setID(cachedFolderElements[i].getID());
				temp.setParentFolderID(cachedFolderElements[i].getParentFolderID());
				temp.setFieldContent(RECURRENCE_ID, Integer.parseInt(cachedFolderElements[i].getID()));
				temp.setUUID(exception.getUUID());
				temp.setTimestamp(cachedFolderElements[i].getTimestamp());
				return temp;
			}
		}
		return null;
	}

	protected DataObject[] readDeletedFolderElements(Folder folder, DataObject[] originalElements)
			throws USMJSONAPIException {
		return readDeletedDataObjects(folder.getElementsContentType(), originalElements);
	}

	private Folder[] convertToFolderArray(DataObject[] dataObjects) {
		Folder[] result = new Folder[dataObjects.length];
		for (int i = 0; i < dataObjects.length; i++)
			result[i] = (Folder) dataObjects[i];
		return result;
	}

	private DataObject[] readDataObjects(Folder folder, ContentType contentType, String key,
			DataObject[] originalObjects, boolean newObjects) throws USMJSONAPIException {
		if (!_parameters.has(key))
			return EMPTY_DATAOBJECT_ARRAY;
		JSONArray list = getJSONArray(_parameters, key);
		int size = list.length();
		List<DataObject> result = new ArrayList<DataObject>(size);
		Map<DataObject, UUID> parentFolderUUIDMap = new HashMap<DataObject, UUID>();

		for (int i = 0; i < size; i++) {
			JSONObject data = getJSONObject(list, i);
			DataObject object = getDataObjectFromJSONObjectAndOriginalObjects(contentType, data, newObjects ? null
					: originalObjects);
			ContentTypeField[] fields = contentType.getFields();
			for (String prop : JSONToolkit.keys(data)) {
				if (OBJECT_TYPE.equals(prop)) {
					checkObjectTypeInJSON(contentType, data);
				} else if (FOLDER_UUID.equals(prop) && contentType.getID().equals(DefaultContentTypes.FOLDER_ID)) {
					String folder_uuid = getString(data, prop);
					if (folder_uuid != null && folder_uuid.length() > 0)
						parentFolderUUIDMap.put(object, UUID.fromString(folder_uuid));
				} else if (CHANGE_EXCEPTIONS.equals(prop)) {
					addExceptionsToClientChanges(result, object, getJSONArray(data, CHANGE_EXCEPTIONS),
							originalObjects, contentType);
				} else if (ATTACHMENTS.equals(prop)) {
					addAttachmentsToObject(object, getJSONArray(data, ATTACHMENTS), getString(data, UUID_KEY));
				} else if (CONTACT_IMAGE_DATA.equals(prop)) {
					storeContactImage(object, data);
				} else if (DISTRIBUTION_LIST.equals(prop)) {
					addDistributionListToObject(object, data, contentType, fields);
				} else if (CommandConstants.TITLE.equals(prop) && MODIFIED.equals(key)
						&& DefaultContentTypes.FOLDER_ID.equals(contentType.getID())
						&& DefaultContentTypes.MAIL_ID.equals(((Folder) object).getElementsContentTypeID())) {
					//when the title of a mail folder is changed than change the id too
					String id = object.getID();
					id = id.substring(0, id.lastIndexOf('/') + 1) + getString(data, prop);
					object.setID(id);
					storeField(object, fields, data, prop);
				} else if (!UUID_KEY.equals(prop)) {
					storeField(object, fields, data, prop);
				}
			}
			result.add(object);
		}
		DataObject[] resultArray = new DataObject[result.size()];
		resultArray = result.toArray(resultArray);
		for (Map.Entry<DataObject, UUID> entry : parentFolderUUIDMap.entrySet()) {
			setParentFolder(folder, entry.getKey(), entry.getValue(), resultArray, originalObjects);
		}
		return resultArray;
	}

	private void addDistributionListToObject(DataObject object, JSONObject data, ContentType contentType,
			ContentTypeField[] fields) throws USMJSONAPIException {
		try {
			JSONArray dataArray = getJSONArray(data, DISTRIBUTION_LIST);
			for (int i = 0; i < dataArray.length(); i++) {
				JSONObject memberObj = dataArray.getJSONObject(i);
				if (memberObj.has(UUID_KEY)) {
					UUID uuid = UUID.fromString(memberObj.getString(UUID_KEY));
					int id = _session.getMappedObjectId(contentType, uuid);
					if (id != 0)
						memberObj.put(CommandConstants.ID, String.valueOf(id));
				}
			}
			storeField(object, fields, data, DISTRIBUTION_LIST);

		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(1, e); //TODO XXX Add unique error code!
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(1, e); //TODO XXX Add unique error code!
		}
	}

	private void addAttachmentsToObject(DataObject object, JSONArray attachmentsArray, String objectUUID)
			throws USMJSONAPIException {
		Object attachmentsFieldContent = object.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
		long timestamp = 0L;
		PIMAttachment[] oldAttachments = new PIMAttachment[0];
		if (attachmentsFieldContent != null) {
			timestamp = ((PIMAttachments) attachmentsFieldContent).getTimestamp();
			oldAttachments = ((PIMAttachments) attachmentsFieldContent).getAttachments();
		}
		PIMAttachment[] attachments = new PIMAttachment[attachmentsArray.length()];
		try {
			for (int i = 0; i < attachmentsArray.length(); i++) {
				JSONObject attObject = attachmentsArray.getJSONObject(i);
				UUID uuid = extractUUIDFromString(attObject.getString(UUID_KEY));
				byte[] data = Toolkit.decodeBase64(attObject.optString(CommandConstants.DATA_KEY));
				attachments[i] = new PIMAttachment(uuid, attObject, data);
			}
			if (!Arrays.equals(attachments, oldAttachments)) {
				_clientAttachmentsMap.put(objectUUID, new PIMAttachments(timestamp, attachments));
			}
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_READ_ATTACHMENTS,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Invalid attachment data", e);
		}
	}

	private DataObject[] readCreatedMailDataObjects(Folder folder, ContentType contentType,
			ConflictResolution conflictResolution, long syncId) throws USMJSONAPIException {
		if (!_parameters.has(CREATED))
			return EMPTY_DATAOBJECT_ARRAY;
		DataObject[] originalObjects = null;
		try {
			if (syncId > 0)
				originalObjects = _session.getCachedFolderElements(folder.getID(), contentType, syncId);
			else
				originalObjects = _session.getCachedFolderElements(folder.getID(), contentType);
		} catch (DatabaseAccessException e) {
			//do nothing, no cached objects are found
		} catch (USMSQLException e) {
			//do nothing, no cached objects are found
		}
		JSONArray list = getJSONArray(_parameters, CREATED);
		int size = list.length();
		List<DataObject> result = new ArrayList<DataObject>(size);
		boolean hadNewMail = false;
		List<DataObject> newState = new ArrayList<DataObject>();
		DataObject[] cachedMails = null;
		try {
			for (int i = 0; i < size; i++) {
				JSONObject data = getJSONObject(list, i);
				DataObject object = getMailDataObjectFromJSONObjectAndOriginalObjects(contentType, data,
						originalObjects);
				if (object.getChangeState() == ChangeState.UNMODIFIED) {
					//objects exists in cache
					JSONObject diffs = getDifferencesClientAndServerMail(data, object, contentType);
					if (diffs != null) {
						//if client mail is the same as the server mail - do nothing
						if (conflictResolution == null)
							conflictResolution = _session.getConflictResolution();
						switch (conflictResolution) {
							case ERROR:
							case ERROR_DELETE_OVER_CHANGE:
								throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_SYNC_CONFLICT_2,
										ResponseStatusCode.SYNC_FAILED, "Conflicting changes", diffs);
							case USE_CLIENT_DELETE_OVER_CHANGE:
							case USE_CLIENT:
								updateMailObject(folder, contentType, originalObjects, syncId, result, data, object,
										syncId, originalObjects);
								break;
							case USE_SERVER_DELETE_OVER_CHANGE:
							case USE_SERVER:
								//ignore changes from client
								_mailDiffsClientAndServer.add(diffs);
								readMailFromJSONObjAndReturnFlags(folder, contentType, data, object);
								result.add(object);
								break;
							default:
								// TODO Throw an exception for unknown ConflictResolution
								break;
						}
					} else {
						result.add(object);
					}
				} else {
					//object doesn't exist in cache
					if (!hadNewMail) {
						// TODO Optimize this
						cachedMails = _session.getCachedFolderElements(folder.getID(), contentType, syncId);
						if (cachedMails != null) {
							for (DataObject o : cachedMails)
								newState.add(o);
						}
						hadNewMail = true;
					}
					int flags = readMailFromJSONObjAndReturnFlags(folder, contentType, data, object);
					if (object.getParentFolderID() == null)
						object.setParentFolder(folder);
					createNewMIMEMailOnServer(contentType, data, object, flags);
					if (originalObjects == null)
						result.add(object);
					object = object.createCopy(true);
					object.commitChanges();
					newState.add(object);
				}
			}
			if (hadNewMail && cachedMails != null)
				_syncIdForUpdate = _session.storeSyncState(getOriginalSyncID(), syncId, folder.getID(), newState
						.toArray(new DataObject[newState.size()]));
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_SAVE_EMAILS_SYNC_STATE, e);
		}
		return result.toArray(new DataObject[result.size()]);
	}

	private JSONObject getDifferencesClientAndServerMail(JSONObject clientData, DataObject serverObject,
			ContentType contentType) throws USMJSONAPIException {
		JSONObject result = null;
		String uuid = null;
		JSONObject extraMailFieldsOnServer = null;
		if (existExternalMailFieldsInJSONObject(clientData)) {
			try {
				//contentType.getTransferHandler().readDataObject(serverObject, _session.getFieldFilter(contentType));
				BitSet extraFields = getExtraFields(contentType);
				extraMailFieldsOnServer = ((MailContentType) contentType).readStructuredMail(serverObject, extraFields);
			} catch (USMException e) {
				throw USMJSONAPIException.createInternalError(
						ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_READ_MAIL_FROM_SERVER, e);
			}
		}
		for (String prop : JSONToolkit.keys(clientData)) {
			if (OBJECT_TYPE.equals(prop)) {
				checkObjectTypeInJSON(contentType, clientData);
			} else if (FOLDER_UUID.equals(prop)) {
				String folder_uuid = getString(clientData, prop);
				if (folder_uuid != null && folder_uuid.length() > 0) {
					Folder parentFolder = getFolderByUUID(folder_uuid);

					if (!serverObject.getParentFolderID().equals(parentFolder.getID())) {
						result = addDifferencesToJSON(result, folder_uuid, serverObject.getParentFolder().getUUID()
								.toString(), FOLDER_UUID);
					}
				}
			} else if (UUID_KEY.equals(prop)) {
				uuid = getString(clientData, prop);
			} else if (ExternalMailContentTypeFields.getFieldIndex(prop) < 0) {
				Object clientField;
				try {
					clientField = clientData.get(prop);
				} catch (JSONException e) {
					throw USMJSONAPIException.createJSONError(
							ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_1, e);
				}
				Object serverField = serverObject.getFieldContent(prop);
				if (!clientField.equals(serverField)) {
					result = addDifferencesToJSON(result, clientField, serverField, prop);
				}
			} else if (ExternalMailContentTypeFields.getFieldIndex(prop) > 0) {
				Object clientField;
				try {
					clientField = clientData.get(prop);
					Object serverField = extraMailFieldsOnServer.get(prop);
					if (clientField instanceof JSONObject && serverField instanceof JSONObject) {
						if (!JSONToolkit.equals((JSONObject) clientField, (JSONObject) serverField))
							result = addDifferencesToJSON(result, clientField, serverField, prop);
					} else if (clientField instanceof JSONArray && serverField instanceof JSONArray) {
						if (!JSONToolkit.equals((JSONArray) clientField, (JSONArray) serverField))
							result = addDifferencesToJSON(result, clientField, serverField, prop);
					} else {
						if (!clientField.equals(serverField))
							result = addDifferencesToJSON(result, clientField, serverField, prop);
					}
				} catch (JSONException e) {
					throw USMJSONAPIException.createJSONError(
							ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_2, e);
				}
			}
		}
		if (result != null)
			try {
				JSONObject details = ((JSONArray) result.get(CONFLICTS)).getJSONObject(0);
				details.put(UUID_KEY, uuid);
			} catch (JSONException e) {
				throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_INVALID_JSON_PROPERTY_2, e);
			}
		return result;
	}

	private JSONObject addDifferencesToJSON(JSONObject jo, Object clientValue, Object serverValue, String key) {
		try {
			if (jo == null)
				jo = new JSONObject();
			JSONArray conflicts;
			if (jo.has(CONFLICTS))
				conflicts = (JSONArray) jo.get(CONFLICTS);
			else
				conflicts = new JSONArray();
			JSONObject details;
			if (conflicts.length() > 0)
				details = conflicts.getJSONObject(0);
			else
				details = new JSONObject();
			JSONObject clientObject;
			if (details.has(CLIENT))
				clientObject = (JSONObject) details.get(CLIENT);
			else
				clientObject = new JSONObject();
			JSONObject serverObject;
			if (details.has(SERVER))
				serverObject = (JSONObject) details.get(SERVER);
			else
				serverObject = new JSONObject();

			clientObject.put(key, clientValue);
			serverObject.put(key, serverValue);
			details.put(CLIENT, clientObject);
			details.put(SERVER, serverObject);
			conflicts.put(0, details);
			jo.put(CONFLICTS, conflicts);
		} catch (JSONException e) {
			//do nothing
		}
		return jo;
	}

	protected boolean existExternalMailFieldsInJSONObject(JSONObject data) {
		for (int j = 0; j < ExternalMailContentTypeFields.getFields().length; j++) {
			if (data.has(ExternalMailContentTypeFields.getFields()[j]))
				return true;
		}
		return false;
	}

	protected long updateMailObject(Folder folder, ContentType contentType, DataObject[] originalObjects, long syncId,
			List<DataObject> result, JSONObject data, DataObject object, long _syncIdForUpdate,
			DataObject[] syncStateToStore) throws USMJSONAPIException {
		if (existExternalMailFieldsInJSONObject(data)) {
			try {
				contentType.getTransferHandler().readDataObject(object, _session.getFieldFilter(contentType));
				BitSet extraFields = getExtraFields(contentType);
				JSONObject extraMailFieldsOnServer = ((MailContentType) contentType).readStructuredMail(object,
						extraFields);
				for (Iterator<?> iterator = extraMailFieldsOnServer.keys(); iterator.hasNext();) {
					String key = (String) iterator.next();
					if (!data.has(key)) //put the fields that are not modified from client but needed to create the new MIME mail
						data.put(key, extraMailFieldsOnServer.get(key));
				}
				//create the new mime mail on the server
				DataObject newObj = contentType.newDataObject(_session);
				newObj.setUUID(object.getUUID());
				newObj.setParentFolderID(object.getParentFolderID());
				int flags = readMailFromJSONObjAndReturnFlags(folder, contentType, data, newObj);
				createNewMIMEMailOnServer(contentType, data, newObj, flags);

				int index = 0;
				for (DataObject originalObject : originalObjects) {
					if (!originalObject.equals(object)) {
						syncStateToStore[index] = originalObject;
					} else {
						syncStateToStore[index] = newObj;
					}
					index++;
				}
				//delete the old mail on server
				contentType.getTransferHandler().writeDeletedDataObject(object);
				if (_syncIdForUpdate < 0)
					result.add(newObj); //_syncIDForUpdate is < 0 only from syncInit command

			} catch (USMException e) {
				throw USMJSONAPIException.createInternalError(
						ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_READ_MAIL_FROM_SERVER, e);
			} catch (JSONException e) {
				throw USMJSONAPIException.createJSONError(
						ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_UPDATE_MAIL_FROM_SERVER, e);
			}
		} else {
			readMailFromJSONObjAndReturnFlags(folder, contentType, data, object);

			try {
				contentType.getTransferHandler().writeUpdatedDataObject(object, syncId);
			} catch (USMException e) {
				throw USMJSONAPIException.createInternalError(
						ConnectorBundleErrorCodes.SYNC_UPDATE_CAN_NOT_UPDATE_EMAIL_ON_SERVER, e);
			}
		}
		return _syncIdForUpdate;
	}

	protected void createNewMIMEMailOnServer(ContentType contentType, JSONObject data, DataObject object, int flags)
			throws USMJSONAPIException {
		MimeMailBuilder mimeMailBuilder = new MimeMailBuilder();
		String mailMIMEdataBlock = mimeMailBuilder.convertJSONObjectToMimeMail(data);
		try {
			String mailID = ((MailContentType) contentType).createNewMail(object.getParentFolderID(), flags,
					mailMIMEdataBlock, _session);
			object.setID(mailID);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_CANNOT_CREATE_MAIL, e);
		}
	}

	protected int readMailFromJSONObjAndReturnFlags(Folder folder, ContentType contentType, JSONObject data,
			DataObject object) throws USMJSONAPIException {
		int flags = 0;
		ContentTypeField[] fields = contentType.getFields();
		for (String prop : JSONToolkit.keys(data)) {
			if (OBJECT_TYPE.equals(prop)) {
				checkObjectTypeInJSON(contentType, data);
			} else if (FOLDER_UUID.equals(prop)) {
				String folder_uuid = getString(data, prop);
				if (folder_uuid != null && folder_uuid.length() > 0) {
					Folder parentFolder = getFolderByUUID(folder_uuid);
					checkMatchSyncAndParentFolder(folder, parentFolder);
					object.setParentFolder(parentFolder);
				}
			} else if (FLAGS.equals(prop)) {
				flags = getInt(data, prop);
				storeField(object, fields, data, prop);
			} else if (!UUID_KEY.equals(prop) && ExternalMailContentTypeFields.getFieldIndex(prop) < 0) {
				storeField(object, fields, data, prop);
			}
		}
		return flags;
	}

	private void checkObjectTypeInJSON(ContentType contentType, JSONObject data) throws USMJSONAPIException {
		String type = getString(data, OBJECT_TYPE);
		if (!contentType.getID().equals(type))
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_DATA_OBJECT_TYPE_MISMATCH,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "DataObject type " + type + " not of expected type "
							+ contentType.getID());
	}

	private void addExceptionsToClientChanges(List<DataObject> result, DataObject seriesObject, JSONArray exceptions,
			DataObject[] originalObjects, ContentType contentType) throws USMJSONAPIException {
		//		Long[] change_exceptions = new Long[exceptions.length()];
		for (int i = 0; i < exceptions.length(); i++) {
			JSONObject data = getJSONObject(exceptions, i);
			DataObject exceptionObject = getDataObjectFromJSONObjectAndOriginalObjectsIfExisting(contentType, data,
					originalObjects);
			for (String prop : JSONToolkit.keys(data)) {
				if (OBJECT_TYPE.equals(prop)) {
					checkObjectTypeInJSON(contentType, data);
				} else if (ATTACHMENTS.equals(prop)) {
					addAttachmentsToObject(exceptionObject, getJSONArray(data, ATTACHMENTS), getString(data, UUID_KEY));
				} else if (!UUID_KEY.equals(prop)) {
					storeField(exceptionObject, contentType.getFields(), data, prop);
				}
			}
			//			change_exceptions[i] = (Long) (exceptionObject.getFieldContent("recurrence_date_position") == null ? exceptionObject
			//					.getFieldContent("start_date")
			//					: exceptionObject.getFieldContent("recurrence_date_position"));
			exceptionObject.setFieldContent(RECURRENCE_ID, Integer.parseInt(seriesObject.getID()));
			exceptionObject.setParentFolderID(seriesObject.getParentFolderID());
			if (exceptionObject.getChangeState() == ChangeState.MODIFIED)
				result.add(exceptionObject);
			else if (exceptionObject.getChangeState() != ChangeState.UNMODIFIED)
				_recurrenceUUIDMap.put(exceptionObject, seriesObject.getUUID());
		}
	}

	private void setParentFolder(Folder folder, DataObject o, UUID uuid, DataObject[] firstList, DataObject[] secondList)
			throws USMJSONAPIException {
		if (setParentFolder(folder, o, uuid, firstList) && setParentFolder(folder, o, uuid, secondList))
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_UNKNOWN_FOLDER_UUID,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown folder_uuid " + uuid);
	}

	private boolean setParentFolder(Folder folder, DataObject o, UUID uuid, DataObject[] list)
			throws USMJSONAPIException {
		// TODO Optimize, use DataObjectSet instead of array to avoid linear search
		if (list != null) {
			for (DataObject p : list) {
				if (p.getUUID().equals(uuid)) {
					Folder f = (Folder) p;
					checkMatchSyncAndParentFolder(folder, f);
					o.setParentFolder(f);
					return false;
				}
			}
		}
		return true;
	}

	private void checkMatchSyncAndParentFolder(Folder syncFolder, Folder parentFolder) throws USMJSONAPIException {
		if (syncFolder != null && !syncFolder.equals(parentFolder))
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_SYNC_AND_PARENT_FOLDER_NO_MATCH,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS,
					"Objects have different parent folder as the one currently synchronized");
	}

	private DataObject[] readDeletedDataObjects(ContentType contentType, DataObject[] originalObjects)
			throws USMJSONAPIException {
		if (!_parameters.has(DELETED))
			return EMPTY_DATAOBJECT_ARRAY;
		JSONArray list = getJSONArray(_parameters, DELETED);
		int size = list.length();
		DataObject[] result = new DataObject[size];
		for (int i = 0; i < size; i++) {
			DataObject object = getDataObjectByUUID(originalObjects, extractUUIDFromString(getString(list, i)));
			object.setChangeState(ChangeState.DELETED);
			result[i] = object;
		}
		return result;
	}

	protected DataObject getDataObjectFromJSONObjectAndOriginalObjects(ContentType contentType, JSONObject data,
			DataObject[] originalObjects) throws USMJSONAPIException {
		UUID uuid = extractUUIDFromString(getString(data, UUID_KEY));
		if (originalObjects != null)
			return getDataObjectByUUID(originalObjects, uuid);
		DataObject result = contentType.newDataObject(_session);
		result.setUUID(uuid);
		return result;
	}

	private DataObject getMailDataObjectFromJSONObjectAndOriginalObjects(ContentType contentType, JSONObject data,
			DataObject[] originalObjects) throws USMJSONAPIException {
		UUID uuid = extractUUIDFromString(getString(data, UUID_KEY));
		if (originalObjects != null) {
			try {
				DataObject result = getDataObjectByUUID(originalObjects, uuid);
				return result;
			} catch (USMJSONAPIException e) {
				//do nothing, continue with code
			}
		}
		DataObject result = contentType.newDataObject(_session);
		result.setUUID(uuid);
		return result;
	}

	private DataObject getDataObjectFromJSONObjectAndOriginalObjectsIfExisting(ContentType contentType,
			JSONObject data, DataObject[] originalObjects) throws USMJSONAPIException {
		UUID uuid = extractUUIDFromString(getString(data, UUID_KEY));
		DataObject result;
		if (originalObjects != null) {
			result = getExceptionObjectByUUID(originalObjects, uuid);
			if (result != null)
				return result;
		}
		result = contentType.newDataObject(_session);
		result.setUUID(uuid);
		return result;
	}

	// TODO Merge with getDataObjectByUUID
	private DataObject getExceptionObjectByUUID(DataObject[] objects, UUID uuid) throws USMJSONAPIException {
		if (objects != null) {
			for (DataObject o : objects) {
				if (o.getUUID().equals(uuid)) {
					if (o.getChangeState() == ChangeState.UNMODIFIED)
						return o;
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_MULTIPLE_ACTIONS_ON_DATA_OBJECT,
							ResponseStatusCode.WRONG_MISSING_PARAMETERS,
							"More than one action requested on DataObject with uuid " + uuid);
				}
			}
		}
		return null;
	}

	private boolean isAppException(DataObject dataObject) {
		if (dataObject.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE) {
			return false;
		}
		String id = dataObject.getID();
		Number recurrence_id = (dataObject.getFieldContent(RECURRENCE_ID) == null) ? null : (Number) (dataObject
				.getFieldContent(RECURRENCE_ID));
		return (recurrence_id != null && recurrence_id.intValue() != 0 && !id.equals(recurrence_id.toString()));
	}

	// TODO Merge with getDataObjectByUUID
	private DataObject getSeriesObjectByUUID(DataObject[] objects, UUID uuid) throws USMJSONAPIException {
		if (objects != null) {
			for (DataObject o : objects) {
				if (o.getUUID().equals(uuid)) {
					return o;
				}
			}
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_DATA_OBJECT_NOT_FOUND_2,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No DataObject with uuid " + uuid);
	}

	private void storeContactImage(DataObject object, JSONObject element) throws USMJSONAPIException {
		try {
			String picture = element.isNull(CONTACT_IMAGE_DATA) ? null : element.getString(CONTACT_IMAGE_DATA);
			object.setFieldContent(CONTACT_IMAGE_DATA, new Image(System.currentTimeMillis(), IMAGE_JPEG, Toolkit
					.decodeBase64(picture)));
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_ILLEGAL_PROPERTY_VALUE,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Illegal value for property " + CONTACT_IMAGE_DATA, e);
		}
	}

	// TODO Possible optimization: Create key->index map for folder to synchronize 
	private void storeField(DataObject object, ContentTypeField[] fields, JSONObject element, String key)
			throws USMJSONAPIException {
		for (ContentTypeField field : fields) {
			if (field.getFieldName().equals(key)) {
				try {
					object.setFieldContent(key, field.getFieldType().extractFromJSONObject(_session, element, key));
					return;
				} catch (JSONException e) {
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_ILLEGAL_PROPERTY_VALUE,
							ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Illegal value for property " + key, e);
				}
			}
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_UNKNOWN_PROPERTY,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown property " + key);
	}

	protected JSONObject createResponseFromSyncResults(SyncResult result) throws USMJSONAPIException {
		return createResponseFromSyncResults(result, null);
	}

	protected JSONObject createResponseFromSyncResults(SyncResult result, Long syncID) throws USMJSONAPIException {
		try {
			DataObject[] newState = result.getNewState();
			_allFolders = newState;
			JSONObject response = new JSONObject();
			if (syncID == null)
				syncID = result.getTimestamp();
			response.put(SYNCID, syncID);
			if (result.isIncomplete())
				response.put(MORE_AVAILABLE, Boolean.TRUE);
			JSONArray created = new JSONArray();
			JSONArray modified = new JSONArray();
			JSONArray deleted = new JSONArray();
			Map<ContentType, BitSet> extraFieldsMap = new HashMap<ContentType, BitSet>();
			boolean folders = true;
			DataObject[] changes = result.getChanges();
			Folder[] folderChanges = new Folder[changes.length];
			for (int i = 0; i < changes.length; i++) {
				if (changes[i] instanceof Folder)
					folderChanges[i] = (Folder) changes[i];
				else {
					folders = false;
					break;
				}
			}
			if (folders) {
				Arrays.sort(folderChanges, new FolderHierarchyComparator(folderChanges));
				changes = folderChanges;
			}
			Map<DataObject, List<DataObject>> exceptionsMap = _newStateToSave == null ? createExceptionsMap(newState)
					: createExceptionsMap(_newStateToSave.toArray());
			for (DataObject o : changes) {
				switch (o.getChangeState()) {
					case CREATED:
						JSONObject extraFieldsObject = addExtraFieldsToDataObject(o, extraFieldsMap);
						if (DefaultContentTypes.CONTACTS_ID.equals(o.getContentType().getID())) {
							setImageByContacts(o);
						}
						if (!isAppException(o)) {
							JSONObject jObj = storeDataObjectWithExtraFieldsToJSONObject(exceptionsMap, o,
									extraFieldsObject);
							created.put(jObj);
						}
						break;
					case DELETED:
						if (!isAppException(o) || (isAppException(o) && !masterAppExistsInChanges(o, changes))) {
							deleted.put(o.getUUID().toString());
						}
						break;
					case MODIFIED:
						if (DefaultContentTypes.CONTACTS_ID.equals(o.getContentType().getID())) {
							setImageByContacts(o);
						}
						if (!isAppException(o)) {
							JSONObject jo = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
							if (jo.length() > 2) // only add element if any actual change is present (i.e. more than "uuid" and "objectType" is set)
								modified.put(jo);
						} else {
							if (!masterAppExistsInChanges(o, changes)) {
								JSONObject jo = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
								if (jo.length() > 2)
									modified.put(jo);
							}
						}
						break;
					case UNMODIFIED:
						// Should not happen
						break;
				}
			}
			addMailModifsExtraFields(modified);
			if (created.length() > 0)
				response.put(CREATED, created);
			if (modified.length() > 0)
				response.put(MODIFIED, modified);
			if (deleted.length() > 0)
				response.put(DELETED, deleted);
			Map<DataObject, USMException> errorMap = result.getErrors();
			if (!errorMap.isEmpty()) {
				addErrorsToResponse(response, extraFieldsMap, errorMap);
			}
			return response;
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_SYNC_RESULT_JSON_ERROR, e);
		} catch (USMException e1) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_SYNC_RESULT_USM_ERROR, e1);
		}
	}

	private boolean masterAppExistsInChanges(DataObject exception, DataObject[] changes) {
		boolean masterAppExistInChanges = false;
		int recID = (Integer) exception.getFieldContent(RECURRENCE_ID);
		for (DataObject change : changes) {
			if (change.getID().equals(String.valueOf(recID)))
				masterAppExistInChanges = true;
		}
		return masterAppExistInChanges;
	}

	private void addErrorsToResponse(JSONObject response, Map<ContentType, BitSet> extraFieldsMap,
			Map<DataObject, USMException> errorMap) throws USMJSONAPIException, JSONException {
		JSONObject errors = new JSONObject();
		for (Map.Entry<DataObject, USMException> entry : errorMap.entrySet()) {
			if (entry.getKey().getContentType().getID() == DefaultContentTypes.CALENDAR_ID
					&& entry.getKey().getChangeState() == ChangeState.CREATED) {
				checkMeetingRequestCreationConflict(entry, errors, extraFieldsMap, response);
			} else {
				addErrorToMap(extraFieldsMap, errors, entry);
			}
		}
		if (errors.length() > 0)
			response.put(ERRORS, errors);
	}

	private void addErrorToMap(Map<ContentType, BitSet> extraFieldsMap, JSONObject errors,
			Map.Entry<DataObject, USMException> entry) throws USMJSONAPIException, JSONException {
		DataObject object = entry.getKey();
		JSONObject extraFieldsObject = addExtraFieldsToDataObject(object, extraFieldsMap);
		errors.put(object.getUUID().toString(), new ResponseObject(null, storeDataObjectWithExtraFieldsToJSONObject(
				null, object, extraFieldsObject), entry.getValue()));
	}

	private void checkMeetingRequestCreationConflict(Map.Entry<DataObject, USMException> entry, JSONObject errors,
			Map<ContentType, BitSet> extraFieldsMap, JSONObject response) throws USMJSONAPIException, JSONException {
		DataObject object = entry.getKey();
		String uid = (String) object.getFieldContent(CommandConstants.UID);
		if (uid == null) {
			addErrorToMap(extraFieldsMap, errors, entry);
			return;
		}
		Session session = object.getSession();
		AppointmentContentType contentType = (AppointmentContentType) object.getContentType();
		String oxID = null;
		try {
			oxID = contentType.resolveUID(session, uid);
		} catch (USMException e1) {
			addErrorToMap(extraFieldsMap, errors, entry);
			return;
		}
		if (oxID == null) {
			addErrorToMap(extraFieldsMap, errors, entry);
			return;
		}
		boolean objectFoundInList = false;
		try {
			DataObject[] allApp = getAllAppointments();
			for (DataObject app : allApp) {
				if (oxID.equals(app.getID())) {
					objectFoundInList = true;
					DataObject appCopy = DataObjectUtil.copyAndModify(app, object, true);
					appCopy.setID(app.getID());
					appCopy.setParentFolderID(app.getParentFolderID());
					contentType.getTransferHandler().writeUpdatedDataObject(appCopy, System.currentTimeMillis());
					JSONArray deletedArray = response.has(DELETED) ? response.getJSONArray(DELETED) : new JSONArray();
					deletedArray.put(object.getUUID().toString());
					response.put(DELETED, deletedArray);
					break;
				}
			}
		} catch (USMException e) {
			createNewObjectWithNewUID(object, extraFieldsMap, errors, entry);
			return;
		}
		if (!objectFoundInList) {
			createNewObjectWithNewUID(object, extraFieldsMap, errors, entry);
		}
	}

	private void createNewObjectWithNewUID(DataObject object, Map<ContentType, BitSet> extraFieldsMap,
			JSONObject errors, Map.Entry<DataObject, USMException> entry) throws USMJSONAPIException, JSONException {
		String newUid = UUID.randomUUID().toString();
		object.setFieldContent(CommandConstants.UID, newUid);
		try {
			object.getContentType().getTransferHandler().writeNewDataObject(object);
			object.getSession().storeUUID(object.getContentType(), Integer.parseInt(object.getID()), object.getUUID());
		} catch (USMException e) {
			addErrorToMap(extraFieldsMap, errors, entry);
		}
	}

	private JSONObject storeDataObjectWithExtraFieldsToJSONObject(Map<DataObject, List<DataObject>> exceptionsMap,
			DataObject o, JSONObject extraFieldsObject) throws JSONException, USMJSONAPIException {
		JSONObject jObj = storeCompleteDataObjectInJSONObject(o, exceptionsMap);
		if (extraFieldsObject != null)
			storeExtraFieldsObjectInJSONObject(jObj, extraFieldsObject);
		return jObj;
	}

	// TODO Add description of the purpose of this method
	private void addMailModifsExtraFields(JSONArray modified) throws JSONException {
		for (JSONObject diffs : _mailDiffsClientAndServer) {
			JSONArray conflicts = (JSONArray) diffs.get(CONFLICTS);
			JSONObject details = conflicts.getJSONObject(0);
			JSONObject serverObject = (JSONObject) details.get(SERVER);
			String uuidDiffs = details.getString(UUID_KEY);
			//check if already exists in modifications
			JSONObject modObj = null;
			int originalIndex = modified.length();
			for (int i = 0; i < modified.length(); i++) {
				JSONObject modObjFromArray = modified.getJSONObject(i);
				String uuid = modObjFromArray.getString(UUID_KEY);
				if (uuid.equals(uuidDiffs)) {
					modObj = modObjFromArray;
					originalIndex = i;
				}
			}
			if (modObj == null)
				modObj = new JSONObject();
			for (String prop : JSONToolkit.keys(serverObject)) {
				modObj.put(prop, serverObject.get(prop));
			}
			modObj.put(OBJECT_TYPE, "mail");
			modObj.put(UUID_KEY, uuidDiffs);
			modified.put(originalIndex, modObj);
		}
	}

	private void setImageByContacts(DataObject o) {
		Image v = (Image) o.getFieldContent(CONTACT_IMAGE_DATA);
		if (o.getChangeState() != ChangeState.MODIFIED ? v != null : o.isFieldModified(CONTACT_IMAGE_DATA)) {
			try {
				byte[] pictureBytes = ((ContactContentType) o.getContentType()).getPictureData(o, "jpeg",
						Integer.MAX_VALUE);
				if (pictureBytes != null)
					o.setFieldContent(CONTACT_IMAGE_DATA, new Image(o.getTimestamp(), IMAGE_JPEG, pictureBytes));
			} catch (USMException e) {
				_servlet.getJournal().warn("error by read of image data - " + e.getErrorCode() + ": " + e.getMessage(),
						e);
			}
		}
	}

	private Map<DataObject, List<DataObject>> createExceptionsMap(DataObject[] changes) {
		Map<DataObject, List<DataObject>> result = new HashMap<DataObject, List<DataObject>>();
		for (DataObject dataObject : changes) {
			if (isAppException(dataObject)) {
				addExceptionToChangeExceptionsList(dataObject, changes, result);
			}
		}
		return result;
	}

	// TODO Optimize this, use a DataObjectSet for changes to avoid linear search
	private void addExceptionToChangeExceptionsList(DataObject exception, DataObject[] changes,
			Map<DataObject, List<DataObject>> result) {
		String recurrence_id = String.valueOf(exception.getFieldContent(RECURRENCE_ID));
		for (DataObject dataObject : changes) {
			if (dataObject.getContentType().getCode() != DefaultContentTypes.CALENDAR_CODE) {
				continue;
			}
			String id = dataObject.getID();
			if (id.equals(recurrence_id)) {
				List<DataObject> exceptionsList = result.get(dataObject);
				if (exceptionsList == null)
					exceptionsList = new ArrayList<DataObject>();
				if (!exceptionsList.contains(exception))
					exceptionsList.add(exception);
				result.put(dataObject, exceptionsList);
			}
		}
	}

	private JSONObject addExtraFieldsToDataObject(DataObject o, Map<ContentType, BitSet> extraFieldsMap)
			throws USMJSONAPIException {
		ContentType type = o.getContentType();
		BitSet extraFields = extraFieldsMap.get(type);
		if (extraFields == null) {
			extraFields = getExtraFields(type);
			extraFieldsMap.put(type, extraFields);
		}

		if (!extraFields.isEmpty()) {
			try {
				if (!DefaultContentTypes.MAIL_ID.equals(type.getID())) {
					type.getTransferHandler().readDataObject(o, extraFields);
					return null;
				} else {
					return readStructuredMail(o, extraFields, (MailContentType) type);
				}
			} catch (USMException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_CANNOT_READ_EXTRA_FIELDS,
						ResponseStatusCode.INTERNAL_ERROR,
						"Extra information for new object could not be read from OX server", e);
			}
		}
		return null;
	}

	// Used to store all currently available appointments in case more than 1 MeetingRequest is executed in 1 sync
	// TODO Probably better to use a DataObjectSet to store all appointments for faster lookup by id / uuid
	private DataObject[] _allAppointments = null;

	private DataObject[] getAllAppointments() throws USMException {
		if (_allAppointments == null) {
			AppointmentContentType appointmentType = (AppointmentContentType) _servlet.getContentTypeManager()
					.getContentType(DefaultContentTypes.CALENDAR_ID);
			_allAppointments = appointmentType.getAllAppointments(_session, _session.getFieldFilter(appointmentType));
		}
		return _allAppointments;
	}

	private JSONObject readStructuredMail(DataObject o, BitSet extraFields, MailContentType contentType)
			throws USMJSONAPIException {
		JSONObject structuredMail = null;
		try {
			structuredMail = contentType.readStructuredMail(o, extraFields);
			//meeting requests
			JSONObject headers = structuredMail.optJSONObject(CommandConstants.MAIL_HEADERS);
			if (headers != null && o.getChangeState() == ChangeState.CREATED) {
				String module = headers.optString(CommandConstants.MAIL_HEADER_OX_MODULE);
				String type = headers.optString(CommandConstants.MAIL_HEADER_OX_TYPE);
				String id = headers.optString(CommandConstants.MAIL_HEADER_OX_OBJ_ID);
				if (CommandConstants.MAIL_HEADER_OX_MODULE_APPOINTMENTS.equals(module)
						&& CommandConstants.MAIL_HEADER_OX_TYPE_NEW.equals(type) && id != null) {
					//					ContentType calendarContentType = _servlet.getContentTypeManager().getContentType(
					//							DefaultContentTypes.CALENDAR_ID);
					DataObject appObj = null;
					if (_allAppointments == null)
						getAllAppointments();
					for (int i = 0; i < _allAppointments.length; i++) {
						if (id.equals(_allAppointments[i].getID())) {
							appObj = _allAppointments[i];
							break;
						}
					}
					if (appObj != null) {
						_session.insertStoredUUID(appObj);
						//						calendarContentType.getTransferHandler().readDataObject(appObj,
						//								_session.getFieldFilter(calendarContentType));
						structuredMail.put(CommandConstants.MEETING_REQUEST,
								storeCompleteDataObjectInJSONObject(appObj));
					}
				}
			}
		} catch (OXCommunicationException e) {
			//the server returned an error: create one multipart email which describes the error and contains the original mail as attachment
			return getErrorMailStructure(o, e, contentType);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_GET_MAIL_OR_MEETING_REQUEST, e);

		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(
					ConnectorBundleErrorCodes.COMMAND_SYNC_CAN_NOT_ADD_MEETING_REQUEST_TO_MAIL, e);
		}
		return structuredMail;
	}

	private JSONObject getErrorMailStructure(DataObject o, OXCommunicationException e, MailContentType contentType) {
		JSONObject error = new JSONObject();
		JSONObject header = new JSONObject();
		JSONArray body = new JSONArray();
		JSONObject contentTypeMail = new JSONObject();
		try {
			contentTypeMail.put(TYPE, MULTIPART_MIXED);
			header.put(CONTENT_TYPE, contentTypeMail);
			header.put(SUBJECT, SERVER_INFO_ERROR_ON_READING_EMAIL);
			error.put(HEADERS, header);
			//add first part
			JSONObject errorPart = new JSONObject();
			JSONObject headerErrorPart = new JSONObject();
			JSONObject bodyErrorPart = new JSONObject();
			JSONObject contentTypeErrorPart = new JSONObject();
			contentTypeErrorPart.put(TYPE, TEXT_PLAIN);
			JSONObject paramsObject = new JSONObject();
			paramsObject.put(CHARSET, UTF_8);
			contentTypeErrorPart.put(PARAMS, paramsObject);
			headerErrorPart.put(CONTENT_TYPE, contentTypeErrorPart);
			bodyErrorPart.put(DATA_KEY, e.getMessage() + ": " + e.getJSONError());
			bodyErrorPart.put(ID, "1.1");
			errorPart.put(HEADERS, headerErrorPart);
			errorPart.put(BODY, bodyErrorPart);
			body.put(errorPart);
			try {
				byte[] originalMail = contentType.readMailSource(o, _session.getFieldFilter(contentType));
				//add original email as attachment part
				JSONObject mailPart = new JSONObject();
				JSONObject headerMailPart = new JSONObject();
				JSONObject bodyMailPart = new JSONObject();
				JSONObject contentTypeMailPart = new JSONObject();
				contentTypeMailPart.put(TYPE, APPLICATION_OCTET_STREAM);
				paramsObject = new JSONObject();
				paramsObject.put(NAME, ORIGINAL_MAIL_EML);
				contentTypeMailPart.put(PARAMS, paramsObject);
				headerMailPart.put(CONTENT_TYPE, contentTypeMailPart);
				JSONObject contentDispMailPart = new JSONObject();
				contentDispMailPart.put(TYPE, ATTACHMENT);
				paramsObject = new JSONObject();
				paramsObject.put(FILENAME, ORIGINAL_MAIL_EML);
				contentDispMailPart.put(CommandConstants.PARAMS, paramsObject);
				headerMailPart.put(CONTENT_DISPOSITION, contentDispMailPart);
				headerMailPart.put(CONTENT_TRANSFER_ENCODING, BASE64);
				bodyMailPart.put(DATA_KEY, Toolkit.encodeBase64(originalMail));
				bodyMailPart.put(ID, "1.2");
				mailPart.put(HEADERS, headerMailPart);
				mailPart.put(BODY, bodyMailPart);
				body.put(mailPart);
			} catch (USMException e1) {
				//can not read email, create just the error part
			}
			error.put(CommandConstants.BODY, body);
		} catch (JSONException e2) {
			//error on creating error mail; return empty object //XXX: or exception???
		}
		return error;
	}

	@SuppressWarnings("unchecked")
	private void storeExtraFieldsObjectInJSONObject(JSONObject jObj, JSONObject extraFieldsObject) throws JSONException {
		for (Iterator iterator = extraFieldsObject.keys(); iterator.hasNext();) {
			String key = (String) iterator.next();
			jObj.put(key, extraFieldsObject.get(key));
		}
	}

	// TODO Optimize, use extra field map which is initialized once at the beginning of the syncInit/Update for the folder
	// TODO The info if the ContentType has attachments_last_modified_utc can be stored in a boolean field so that it has to be computed only once
	protected boolean hasAttachmentsLastModified(ContentType contentType) {
		if (contentType != null && contentType.supportsPIMAttachments()) {
			ContentTypeField[] fields = contentType.getFields();
			for (int i = 0; i < fields.length; i++) {
				if (ATTACHMENTS_LAST_MODIFIED.equals(fields[i].getFieldName()))
					return _session.getFieldFilter(contentType).get(i);
			}
		}
		return false;
	}

	// TODO Is it possible to only retrieve current server attachments once, maybe use a map for already retrieved data ?
	protected void handlePIMAttachments(Folder folder, SyncResult result, long newSyncId) throws USMJSONAPIException {
		ContentType elementsContentType = folder.getElementsContentType();
		if (!hasAttachmentsLastModified(elementsContentType))
			return;
		//handle the client changes
		try {
			for (int i = 0; i < _clientChangesArray.length; i++) {
				DataObject clientObject = _clientChangesArray[i];
				handlePIMAttClientChanges(newSyncId, folder, elementsContentType, clientObject);
			}
			for (DataObject exceptionObject : _exceptionChangesForDelayedCall.keySet()) {
				handlePIMAttForAppExceptions(newSyncId, folder, elementsContentType, exceptionObject);
			}
			for (DataObject exceptionObject : _recurrenceUUIDMap.keySet()) {
				handlePIMAttForAppExceptions(newSyncId, folder, elementsContentType, exceptionObject);
			}
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_ON_CREATE_DELETE_ATTACHMENTS, e);
		}

		//handle the server changes
		try {
			handlePIMAttachmentsServerChanges(result);
		} catch (USMException e1) {
			throw USMJSONAPIException.createInternalError(
					ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_ON_RETRIEVE_ATTACHMENTS_FORM_SERVER, e1);
		}
	}

	private void handlePIMAttForAppExceptions(long newSyncId, Folder folder, ContentType elementsContentType,
			DataObject exceptionObject) throws USMException, DatabaseAccessException, USMSQLException {
		DataObject o = _newStateToSave.get(exceptionObject.getUUID());
		if (o != null)
			handlePIMAttClientChanges(newSyncId, folder, elementsContentType, o);
	}

	private void handlePIMAttClientChanges(long newSyncId, Folder folder, ContentType elementsContentType,
			DataObject clientObject) throws USMException, DatabaseAccessException, USMSQLException {
		if (clientObject.getChangeState() != ChangeState.DELETED) {
			PIMAttachments clientAttachments = _clientAttachmentsMap.get(clientObject.getUUID().toString());
			if (clientAttachments == null)
				return;
			PIMAttachments serverAttachments = elementsContentType.getAllAttachments(clientObject);
			PIMAttachments oldServerAttachments = null;
			if (clientObject.getID() != null) {
				DataObject cachedObjectState = getCachedElement(clientObject.getID());
				if (cachedObjectState != null)
					oldServerAttachments = (PIMAttachments) cachedObjectState
							.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
			}
			if (oldServerAttachments == null)
				oldServerAttachments = new PIMAttachments(0, new PIMAttachment[0]);
			handlePIMAttClientCreations(oldServerAttachments, serverAttachments, clientObject, clientAttachments,
					newSyncId);
			handlePIMAttClientDeletions(oldServerAttachments, serverAttachments, clientObject, clientAttachments,
					newSyncId);
		}
	}

	private void handlePIMAttachmentsServerChanges(SyncResult result) throws USMException {
		for (DataObject serverObject : result.getChanges()) {
			ContentType contentType = serverObject.getContentType();
			if (serverObject.getChangeState() != ChangeState.DELETED && contentType.supportsPIMAttachments()) {
				if (serverObject.getChangeState() == ChangeState.CREATED) {
					PIMAttachments serverAttachments = (PIMAttachments) contentType.getAllAttachments(serverObject);
					if (serverAttachments != null)
						serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
					//save the new attachments in _newStateToSave
					addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);
				} else if (serverObject.isFieldModified(ATTACHMENTS_LAST_MODIFIED)) {
					DataObject cachedObjectState = getCachedElement(serverObject.getID());
					PIMAttachments currentAttachments = (PIMAttachments) serverObject
							.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
					if (cachedObjectState != null) {
						PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState
								.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
						if (currentAttachments != null && oldSavedAttachments != null)
							serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, new PIMAttachments(
									currentAttachments.getTimestamp(), oldSavedAttachments.getAttachments()));
						else if (currentAttachments == null && oldSavedAttachments != null) {
							PIMAttachments att = new PIMAttachments(0, new PIMAttachment[0]);
							//empty attachments field - means all attachments have been deleted on the server
							serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, att);
							addTheServerAttachmentsToNewSyncState(serverObject, att);
						}
					}
					PIMAttachments serverAttachments = (PIMAttachments) contentType.getAllAttachments(serverObject);
					if (serverAttachments != null)
						serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
					//save the new attachments in _newStateToSave
					addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);

				} else if (isNumOfAttachmentsChanged(serverObject)) {
					//some attachment has been deleted, but is it not the newest one and lastModifiedOfNewesAttachmentUTC is not changed
					PIMAttachments serverAttachments = (PIMAttachments) contentType.getAllAttachments(serverObject);
					if (serverAttachments != null)
						serverObject.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
					//save the new attachments in _newStateToSave
					addTheServerAttachmentsToNewSyncState(serverObject, serverAttachments);

				} else {
					//save the old attachments (if existing) in the _newStateToSave
					DataObject cachedObjectState = getCachedElement(serverObject.getID());
					if (cachedObjectState != null) {
						PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState
								.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
						addTheServerAttachmentsToNewSyncState(serverObject, oldSavedAttachments);
					}
				}
			}
		}
		for (DataObject serverObject : _newStateToSave) {
			ContentType contentType = serverObject.getContentType();
			if (serverObject.getChangeState() == ChangeState.UNMODIFIED && contentType.supportsPIMAttachments()) {
				//save the old attachments (if existing) in the _newStateToSave
				DataObject cachedObjectState = getCachedElement(serverObject.getID());
				if (cachedObjectState != null) {
					PIMAttachments oldSavedAttachments = (PIMAttachments) cachedObjectState
							.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
					PIMAttachments currentAttachments = (PIMAttachments) serverObject
							.getFieldContent(ATTACHMENTS_LAST_MODIFIED);
					if (oldSavedAttachments != null && oldSavedAttachments.size() > 0
							&& (currentAttachments == null || currentAttachments.size() == 0))
						addTheServerAttachmentsToNewSyncState(serverObject, oldSavedAttachments);
				}
			}
		}
	}

	private boolean isNumOfAttachmentsChanged(DataObject serverObject) {
		Number originalValue = (Number) serverObject.getOriginalFieldContent(CommandConstants.NUMBER_OF_ATTACHMENTS);
		Number value = (Number) serverObject.getFieldContent(CommandConstants.NUMBER_OF_ATTACHMENTS);
		if ((originalValue == null || originalValue.intValue() == 0) && (value == null || value.intValue() == 0))
			return false;
		return serverObject.isFieldModified(CommandConstants.NUMBER_OF_ATTACHMENTS);
	}

	private void addTheServerAttachmentsToNewSyncState(DataObject serverObject, PIMAttachments serverAttachments) {
		if (serverAttachments == null || _newStateToSave == null || serverObject == null)
			return;
		DataObject o = _newStateToSave.get(serverObject.getID());
		if (o != null) {
			o.setFieldContent(ATTACHMENTS_LAST_MODIFIED, serverAttachments);
			_newStateToSaveChanged = true;
		}
	}

	/**
	 * Checks if the server still has some attachments which don't exist on the client any more : delete on server
	 * @param oldServerState
	 * @param newServerState
	 * @param clientObject
	 * @param clientAttachments
	 * @throws USMException
	 */
	private void handlePIMAttClientDeletions(PIMAttachments oldServerAttachments, PIMAttachments serverAttachments,
			DataObject clientObject, PIMAttachments clientAttachments, long syncId) throws USMException {
		if (serverAttachments == null)
			return;
		for (int k = 0; k < serverAttachments.size(); k++) {
			PIMAttachment serverAttachment = serverAttachments.getAttachment(k);
			boolean existsInOldServerState = existsInAttachmentsList(serverAttachment, oldServerAttachments);
			if (existsInOldServerState || oldServerAttachments == null) {
				boolean existsInClientState = false;
				for (int j4 = 0; j4 < clientAttachments.size(); j4++) {
					PIMAttachment clientAttachment = clientAttachments.getAttachment(j4);
					if (clientAttachment.equalsByUUID(serverAttachment)) {
						existsInClientState = true;
					}
				}
				if (!existsInClientState) {
					long timestamp = clientObject.getContentType().deleteAttachments(clientObject, serverAttachment);
					removePIMAttachmentFromNewServerState(clientObject, serverAttachment, timestamp,
							oldServerAttachments);
				}
			}
		}
	}

	/**
	 * Checks if the client has sent some attachments which still don't exist on server and creates them on  the server.
	 * @param oldServerState
	 * @param newServerAttachmentsState
	 * @param clientObject
	 * @param clientAttachments
	 * @throws USMException
	 */
	private void handlePIMAttClientCreations(PIMAttachments serverAttachments,
			PIMAttachments newServerAttachmentsState, DataObject clientObject, PIMAttachments clientAttachments,
			long syncId) throws USMException {
		if (clientAttachments == null)
			return;
		for (int i = 0; i < clientAttachments.size(); i++) {
			PIMAttachment clientAttachment = clientAttachments.getAttachment(i);
			boolean existsInOldServerState = existsInAttachmentsList(clientAttachment, serverAttachments);
			if (!existsInOldServerState) {
				boolean existsInNewServerState = existsInAttachmentsList(clientAttachment, newServerAttachmentsState);
				if (!existsInNewServerState) {
					long timestamp = clientObject.getContentType().createNewAttachment(clientObject, clientAttachment);
					addPIMAttachmentToNewServerState(clientObject, clientAttachment, timestamp, serverAttachments);
				}
			}
		}
	}

	private void addPIMAttachmentToNewServerState(DataObject clientObject, PIMAttachment clientAttachment,
			long timestamp, PIMAttachments oldAttachments) throws USMException {
		DataObject o = _newStateToSave.get(clientObject.getID());
		if (o != null) {
			List<PIMAttachment> newAttachmentsList = new ArrayList<PIMAttachment>();
			for (int j = 0; oldAttachments != null && j < oldAttachments.size(); j++) {
				if (oldAttachments.getAttachment(j).equalsByUUID(clientAttachment)) {
					newAttachmentsList.add(clientAttachment);
				} else {
					newAttachmentsList.add(oldAttachments.getAttachment(j));
				}
			}
			if (!newAttachmentsList.contains(clientAttachment))
				newAttachmentsList.add(clientAttachment);
			refreshNewStateToSaveWithNewPIMAttachemnts(timestamp, o, newAttachmentsList);
		}
	}

	private void refreshNewStateToSaveWithNewPIMAttachemnts(long timestamp, DataObject o,
			List<PIMAttachment> newAttachmentsList) throws USMException {
		int numberOfAttachments = newAttachmentsList.size();
		o.setFieldContent(ATTACHMENTS_LAST_MODIFIED, timestamp == 0 ? null : new PIMAttachments(timestamp,
				newAttachmentsList.toArray(new PIMAttachment[numberOfAttachments])));
		o.setFieldContent(NUMBER_OF_ATTACHMENTS, numberOfAttachments);
		_newStateToSaveChanged = true;
	}

	private void removePIMAttachmentFromNewServerState(DataObject objectOnServer, PIMAttachment clientAttachment,
			long timestamp, PIMAttachments serverAttachments) throws USMException {
		DataObject o = _newStateToSave.get(objectOnServer.getID());
		if (o != null) {
			List<PIMAttachment> newAttList = new ArrayList<PIMAttachment>();
			for (PIMAttachment attachment : serverAttachments.getAttachments()) {
				if (!attachment.equalsByUUID(clientAttachment))
					newAttList.add(attachment);
			}
			refreshNewStateToSaveWithNewPIMAttachemnts(timestamp, o, newAttList);
		}
	}

	// TODO Move this to PIMAttachments ?
	private boolean existsInAttachmentsList(PIMAttachment attachment, PIMAttachments attachmentsList) {
		for (int k = 0; attachmentsList != null && k < attachmentsList.size(); k++) {
			if (attachmentsList.getAttachment(k).equalsByUUID(attachment)) {
				return true;
			}
		}
		return false;
	}

	protected USMJSONAPIException generateConflictException(SynchronizationConflictException e) {
		try {
			JSONObject details = new JSONObject();
			JSONArray conflicts = new JSONArray();
			for (ConflictingChange conflict : e.getConflictingChanges()) {
				JSONObject data = new JSONObject();
				DataObject clientObject = conflict.getClientChange();
				DataObject serverObject = conflict.getServerChange();
				data.put(UUID_KEY, clientObject.getUUID().toString());
				data.put(CLIENT, getDataObjectDifferences(clientObject, serverObject));
				data.put(SERVER, getDataObjectDifferences(serverObject, clientObject));
				conflicts.put(data);
			}
			details.put(CONFLICTS, conflicts);
			return new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_SYNC_CONFLICT,
					ResponseStatusCode.SYNC_FAILED, e.getMessage(), details);
		} catch (USMJSONAPIException e1) {
			return e1;
		} catch (JSONException e1) {
			return USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.COMMAND_SYNC_ERROR_JSON_ERROR, e1);
		}
	}

	private Object getDataObjectDifferences(DataObject object, DataObject other) throws JSONException,
			USMJSONAPIException {
		if (object.getChangeState() == ChangeState.DELETED)
			return DELETED;
		JSONObject diff = storeDataObjectFieldsInJSONObject(DataObjectUtil.copyAndModify(other, object, true));
		if (object.getUUID() != null && other.getUUID() != null && !object.getUUID().equals(other.getUUID()))
			diff.put(UUID_KEY, object.getUUID().toString());
		addOptionalFolderUUID(object, diff);
		return diff;
	}

	protected DataObject getCachedElement(String id) throws DatabaseAccessException, USMSQLException {
		return _cachedElements == null ? null : _cachedElements.get(id);
	}

	protected abstract long getOriginalSyncID();
}
