/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;

import javax.servlet.http.HttpSession;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.ObjectChanges;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * Handler for the USM-Ping Command. 
 * @author ldo
 *
 */
public class PingHandler extends NormalCommandHandler {

	private static final String INTERVAL = "interval";
	private final static String[] REQUIRED_PARAMETERS = { SESSIONID };
	private final static String[] OPTIONAL_PARAMETERS = { OBJECTIDS };

	public PingHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters)
			throws USMJSONAPIException {
		super(servlet, httpSession, parameters);
		_servlet.newPingRequestArrived();
	}

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {

		JSONArray objectUUIDs = _parameters.has(OBJECTIDS) ? getJSONArray(_parameters, OBJECTIDS) : null;
		boolean watchFolderHierarchyChanges = true;
		if (objectUUIDs != null) {
			watchFolderHierarchyChanges = isContextUUIDRequested(objectUUIDs);
		}
		try {
			ObjectChanges changes = getSession().waitForChanges(watchFolderHierarchyChanges,
					toOXIdsStringArray(objectUUIDs), 0);
			JSONArray resultArray = null;
			if (changes != null) {
				resultArray = toUUIDsJSONArray(changes.getContentChangeParentIDs());
				if (changes.hasFolderStructureChanged())
					resultArray.put(getSession().getContextUUID().toString());
			}
			JSONObject data = new JSONObject();
			if(resultArray != null && resultArray.length() > 0) data.put(OBJECTIDS, resultArray);
			data.put(INTERVAL, _servlet.getPingInterval());
			return new ResponseObject(ResponseStatusCode.SUCCESS, data);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_INTERNAL_ERROR, e);
		} catch (InterruptedException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.PING_WAIT_FOR_CHANGES_INTERRUPTED_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "waiting for changes interrupted", e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.PING_JSON_ERROR, e);
		} 
	}

	private boolean isContextUUIDRequested(JSONArray objectUUIDs) throws USMJSONAPIException {
		for (int i = 0; i < objectUUIDs.length(); i++) {
			String uuid = getString(objectUUIDs, i);
			if(isContextUUID(uuid))
				return true;
		}
		return false;
	}

	private boolean isContextUUID(String uuid) throws USMJSONAPIException {
		try {
			if (getSession().getContextUUID().equals(extractUUIDFromString(uuid)))
				return true;
		} catch (DatabaseAccessException e) {
			USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_NO_DB_ACCESS, e);
		} catch (USMSQLException e) {
			USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.PING_SQL_ERROR, e);
		}
		return false;
	}

	private String[] toOXIdsStringArray(JSONArray jsonArray) throws USMJSONAPIException {
		if (jsonArray == null)
			return null;
		String[] stringArray = new String[jsonArray.length() - (isContextUUIDRequested(jsonArray)? 1: 0)];
		int j = 0;
		for (int i = 0; i < jsonArray.length(); i++) {
			String uuid = getString(jsonArray, i);
			if (!isContextUUID(uuid)) {
				Folder folder = getFolderByUUID(uuid);
				stringArray[j++] = folder.getID();
			}
		}
		return stringArray;
	}
	
	private JSONArray toUUIDsJSONArray(String[] idsArray) throws DatabaseAccessException, USMSQLException,
			JSONException {
		JSONArray result = new JSONArray();
		for (int i = 0; i < idsArray.length; i++) {
			Folder cachedFolder = getSession().getCachedFolder(idsArray[i]);
			if (cachedFolder != null) {
				result.put(i, cachedFolder.getUUID().toString());
			}
		}
		return result;
	}
	

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

}
