/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;

import java.util.*;

import javax.servlet.http.HttpSession;

import org.json.*;

import com.openexchange.usm.api.contenttypes.DefaultContentTypes;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.json.*;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;

/**
 * Handler for the MoveItems USM-JSON-Command.
 * @author ldo
 *
 */
public class MoveItemsHandler extends SyncCommandHandler {

	private static final String LAST_MODIFIED_OF_NEWEST_ATTACHMENT_UTC = "lastModifiedOfNewestAttachmentUTC";

	private enum ErrorStatusCodes {
		/**
		 * Move Items destination doesn't exist on the server. 
		 */
		DESTINATION_NOT_EXIST,

		/**
		 * Move Items source doesn't exist on the server. 
		 */
		SOURCE_NOT_EXIST,

		/**
		 * No create objects permission in the destination folder (Move Items) for all objects
		 */
		DESTINATION_NO_PERMISSION,

		/**
		 * No delete permission in the source folder (Move Items) for all objects
		 */
		SOURCE_NO_PERMISSION_ALL_OBJECTS,

		/**
		 * No delete permission in the source folder (Move Items) for own objects
		 */
		SOURCE_NO_PERMISSION_OWN_OBJECTS,

		/**
		 * Other error. 
		 */
		OTHER;

		public int getStatusCode() {
			return ordinal() + 1;
		}
	}

	private static final String ERRORS = "errors";
	private static final String ERROR_STATUS = "errorStatus";
	private static final String ERROR_DETAILS = "errorDetails";
	private static final String OWN_RIGHTS = "own_rights";
	private final static String[] REQUIRED_PARAMETERS = {
			SESSIONID,
			SOURCEFOLDERID,
			SOURCESYNCID,
			DESTINATIONFOLDERID,
			DESTINATIONSYNCID,
			OBJECTIDS };
	private final static String[] OPTIONAL_PARAMETERS = {};

	private ErrorStatusCodes _errorStatus = ErrorStatusCodes.OTHER;

	private long _originalSyncID;

	public MoveItemsHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters)
			throws USMJSONAPIException {
		super(servlet, httpSession, parameters);
	}

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		String sourceFolderUUID = getStringParameter(SOURCEFOLDERID);
		String destinationFolderUUID = getStringParameter(DESTINATIONFOLDERID);
		Folder sourceFolder = getFolderByUUID(sourceFolderUUID);
		Folder destinationFolder = getFolderByUUID(destinationFolderUUID);
		long sourceFolderSyncID = getLong(_parameters, SOURCESYNCID);
		long destinationSyncID = getLong(_parameters, DESTINATIONSYNCID);
		if (sourceFolderUUID.equals(destinationFolderUUID))
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_SAME_SOURCE_DEST,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Source and Destination are the same.");
		JSONArray objectsToMove = getJSONArray(_parameters, OBJECTIDS);

		//move items
		JSONObject response = moveItems(sourceFolder, sourceFolderSyncID, destinationFolder, destinationSyncID,
				objectsToMove);
		return new ResponseObject(ResponseStatusCode.SUCCESS, response);
	}

	private boolean checkExisting(Folder folder) {
		BitSet fields = new BitSet();
		try {
			folder.getContentType().getTransferHandler().readFolder(folder, fields);
		} catch (USMException e) {
			return false;
		}
		return true;
	}

	private boolean checkCreatePermissionsAllObjects(int own_rights) {
		return (own_rights & 0x7F) >= 2;
	}

	private boolean checkDeletePermissionsAllObjects(int own_rights) {
		return (own_rights & 0xFE00000) >= 0x400000;
	}

	private boolean checkDeletePermissionsOwnObjects(int own_rights) {
		return (own_rights & 0xFE00000) >= 0x200000;
	}

	private int getOwnRights(Folder folder) {
		BitSet requestedFields = new BitSet();
		int fieldIndex = folder.getFieldIndex(OWN_RIGHTS);
		requestedFields.set(fieldIndex);
		try {
			folder.getContentType().getTransferHandler().readFolder(folder, requestedFields);
		} catch (USMException e) {
			return 0;
		}
		int own_rights = ((Integer) folder.getFieldContent(fieldIndex)).intValue();
		return own_rights;
	}

	private JSONObject moveItems(Folder sourceFolder, long sourceFolderSyncID, Folder destinationFolder,
			long destinationSyncID, JSONArray objectsToMove) throws USMJSONAPIException {
		JSONObject responseData = new JSONObject();
		DataObject[] changes = null;
		try {
			if (!sourceFolder.getElementsContentTypeID().equals(destinationFolder.getElementsContentTypeID()))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_NOT_MATCHING_FOLERS,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Source and Destination are not of same type.");
			DataObject[] sourceElements = _session.getCachedFolderElements(sourceFolder.getID(), sourceFolder
					.getElementsContentType(), sourceFolderSyncID);
			if (sourceElements == null)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_INVALID_SYNC_ID,
						ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
			DataObject[] destinationElements = _session.getCachedFolderElements(destinationFolder.getID(),
					destinationFolder.getElementsContentType(), destinationSyncID);
			if (destinationElements == null)
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_INVALID_SYNC_ID_2,
						ResponseStatusCode.UNKNOWN_SYNCID, "Unknown SyncID");
			_originalSyncID = sourceFolderSyncID;
			List<DataObject> changesList = new ArrayList<DataObject>();
			for (int i = 0; i < objectsToMove.length(); i++) {
				UUID objectUUID = extractUUIDFromString(getString(objectsToMove, i));
				DataObject object = getDataObjectByUUID(sourceElements, objectUUID);
				object.setParentFolder(destinationFolder);
				changesList.add(object);
			}
			changes = new DataObject[changesList.size()];
			changes = changesList.toArray(changes);
			SyncResult syncResult = _session.syncChangesWithServer(sourceFolder.getID(), sourceFolderSyncID, -1, null,
					false, ConflictResolution.ERROR, changes);
			Map<DataObject, USMException> errors = syncResult.getErrors();
			DataObject[] newSourceState = getNewSourceStateToSave(sourceFolder, sourceFolderSyncID, changes, errors);
			long newSourceSyncID = _session.storeSyncState(sourceFolderSyncID, syncResult.getTimestamp(), sourceFolder
					.getID(), newSourceState);
			responseData.put(SOURCESYNCID, newSourceSyncID);

			if (errors != null && !errors.isEmpty()) {
				JSONObject notmoved = new JSONObject();
				//reset to other
				_errorStatus = checkFolderExistance(sourceFolder, destinationFolder);
				int destinationRigths = getOwnRights(destinationFolder);
				int sourceRights = getOwnRights(sourceFolder);
				for (Map.Entry<DataObject, USMException> entry : errors.entrySet()) {
					DataObject dataObject = entry.getKey();
					USMException exception = entry.getValue();
					addErrorToNotMovedArray(sourceRights, destinationRigths, notmoved, dataObject, exception);
				}
				responseData.put(ERRORS, notmoved);
			}
			_originalSyncID = destinationSyncID;
			if (_errorStatus != ErrorStatusCodes.DESTINATION_NOT_EXIST) {
				DataObject[] newDestinationState = getNewDestinationStateToSave(destinationFolder, destinationSyncID,
						changes);
				long newDestinationSyncID = _session.storeSyncState(destinationSyncID, destinationSyncID,
						destinationFolder.getID(), newDestinationState);
				responseData.put(DESTINATIONSYNCID, newDestinationSyncID);
			} else {
				responseData.put(DESTINATIONSYNCID, destinationSyncID);
			}

		} catch (SynchronizationConflictException e) {
			throw generateConflictException(e);
		} catch (USMSQLException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR, e);
		} catch (DatabaseAccessException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_2, e);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (OXCommunicationException e) {
			return createErrorResponse(sourceFolderSyncID, destinationSyncID, e, sourceFolder, destinationFolder,
					changes);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_3, e);
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.MOVE_ITEMS_JSON_ERROR, e);
		}
		return responseData;
	}

	private void addErrorToNotMovedArray(int sourceRights, int destinationRights, JSONObject notmoved,
			DataObject dataObject, USMException exception) throws JSONException {
		JSONObject errorObject = new JSONObject();
		errorObject.put(ERROR_DETAILS, exception.getErrorDetailsForJSONResponse() == null ? exception
				.getOxErrorForJSONResponse() : exception.getErrorDetailsForJSONResponse());
		errorObject.put(ERROR_STATUS, getErrorStatusCode(destinationRights, sourceRights, dataObject).getStatusCode());
		notmoved.put(dataObject.getUUID().toString(), errorObject);
	}

	private JSONObject createErrorResponse(long sourceFolderSyncID, long destinationSyncID, USMException e,
			Folder sourceFolder, Folder destinationFolder, DataObject[] changes) {
		JSONObject errorResponse = new JSONObject();
		try {
			errorResponse.put(SOURCESYNCID, sourceFolderSyncID);
			JSONObject notmoved = new JSONObject();
			//reset to other
			_errorStatus = checkFolderExistance(sourceFolder, destinationFolder);
			int destinationRigths = getOwnRights(destinationFolder);
			int sourceRights = getOwnRights(sourceFolder);
			for (int i = 0; changes != null && i < changes.length; i++) {
				addErrorToNotMovedArray(sourceRights, destinationRigths, notmoved, changes[i], e);
			}
			errorResponse.put(ERRORS, notmoved);
			errorResponse.put(DESTINATIONSYNCID, destinationSyncID);
		} catch (JSONException e1) {
			//do nothing
		}
		return errorResponse;
	}

	private ErrorStatusCodes getErrorStatusCode(int destinationRigths, int sourceRights, DataObject errorDataObject) {
		if (_errorStatus == ErrorStatusCodes.OTHER) {
			if (!checkCreatePermissionsAllObjects(destinationRigths))
				return ErrorStatusCodes.DESTINATION_NO_PERMISSION;
			if (isOwnObject(errorDataObject) && !checkDeletePermissionsOwnObjects(sourceRights))
				return ErrorStatusCodes.SOURCE_NO_PERMISSION_OWN_OBJECTS;
			if (!checkDeletePermissionsAllObjects(sourceRights))
				return ErrorStatusCodes.SOURCE_NO_PERMISSION_ALL_OBJECTS;
		}
		return _errorStatus;
	}

	private ErrorStatusCodes checkFolderExistance(Folder sourceFolder, Folder destinationFolder) {
		//check special error cases
		if (!checkExisting(destinationFolder))
			return ErrorStatusCodes.DESTINATION_NOT_EXIST;
		if (!checkExisting(sourceFolder))
			return ErrorStatusCodes.SOURCE_NOT_EXIST;
		return ErrorStatusCodes.OTHER;
	}

	private boolean isOwnObject(DataObject object) {
		if (DefaultContentTypes.MAIL_CODE == object.getContentType().getCode())
			return true; //mails can be only own objects
		int userId = object.getSession().getUserIdentifier();
		String createdBy = (String) object.getFieldContent("created_by");
		return createdBy.equals(String.valueOf(userId));
	}

	private DataObject[] getNewDestinationStateToSave(Folder folder, long syncID, DataObject[] requestedChanges)
			throws USMException {
		SyncResult syncResult = _session.syncChangesWithServer(folder.getID(), syncID, Session.NO_LIMIT, null, false,
				null);
		DataObject[] oldState = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(),
				syncID);
		List<DataObject> resultList = new ArrayList<DataObject>();
		resultList.addAll(Arrays.asList(oldState));
		for (DataObject dataObject : syncResult.getChanges()) {
			if (dataObject.getChangeState() == ChangeState.CREATED) {
				addElementIfInClientChanges(folder, requestedChanges, resultList, dataObject);
			}
		}
		return resultList.toArray(new DataObject[resultList.size()]);
	}

	private DataObject[] getNewSourceStateToSave(Folder folder, long syncID, DataObject[] requestedChanges,
			Map<DataObject, USMException> errors) throws USMException {
		DataObject[] oldState = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType(),
				syncID);
		List<DataObject> resultList = new ArrayList<DataObject>();
		resultList.addAll(Arrays.asList(oldState));
		for (DataObject dataObject : requestedChanges) {
			if (!errors.containsKey(dataObject)) {
				for (DataObject oldObject : oldState) {
					if (oldObject.getID().equals(dataObject.getOriginalID()))
						resultList.remove(oldObject);
				}
			}
		}
		return resultList.toArray(new DataObject[resultList.size()]);
	}

	private void addElementIfInClientChanges(Folder folder, DataObject[] requestedChanges, List<DataObject> resultList,
			DataObject dataObject) {
		DataObject objectToChange = dataObject.createCopy(false);
		for (DataObject sourceElement : requestedChanges) {
			if (sourceElement.getParentFolderID().equals(dataObject.getParentFolderID())
					&& sourceElement.getID().equals(dataObject.getID())) {
				objectToChange.setUUID(sourceElement.getUUID());
				if (!folder.getElementsContentTypeID().equals(DefaultContentTypes.MAIL_ID)) {
					objectToChange.setFieldContent(LAST_MODIFIED_OF_NEWEST_ATTACHMENT_UTC, sourceElement
							.getFieldContent(LAST_MODIFIED_OF_NEWEST_ATTACHMENT_UTC));
				}
				resultList.add(objectToChange);
				return;
			}
		}
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

	@Override
	protected long getOriginalSyncID() {
		return _originalSyncID;
	}
}
