/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.*;

import java.util.*;

import javax.servlet.http.HttpSession;

import org.json.*;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.ChangeState;
import com.openexchange.usm.api.session.ConflictResolution;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.json.*;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.ox_json.OXJSONPropertyNames;
import com.openexchange.usm.util.BitSetEncoder;
import com.openexchange.usm.util.JSONToolkit;

/**
 * General Handler for all  USM-JSON-Commands. 
 * All command handlers for specific commands should extend this handler. 
 * @author ldo
 *
 */
public abstract class BaseCommandHandler implements CommandHandler {

	protected final USMJSONServlet _servlet;
	protected final HttpSession _httpSession;
	protected final JSONObject _parameters;

	protected Session _session;

	public BaseCommandHandler(USMJSONServlet servlet, HttpSession httpSession, JSONObject parameters)
			throws USMJSONAPIException {
		_servlet = servlet;
		_httpSession = httpSession;
		_parameters = parameters;
		int requiredCount = 0;
		for (String key : JSONToolkit.keys(parameters)) {
			if (checkParameter(key))
				requiredCount++;
		}
		if (requiredCount != getRequiredParameters().length) {
			for (String key : getRequiredParameters()) {
				if (!parameters.has(key))
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_MISSING_REQUIRED_PARAMETER,
							ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Missing required parameter " + key);
			}
		}
	}

	private boolean checkParameter(String key) throws USMJSONAPIException {
		for (String requiredKey : getRequiredParameters()) {
			if (key.equals(requiredKey))
				return true;
		}
		for (String requiredKey : getOptionalParameters()) {
			if (key.equals(requiredKey))
				return false;
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_UNKNOWN_PARAMETER,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Unknown parameter " + key);
	}

	/* (non-Javadoc)
	 * @see com.openexchange.usm.connector.commands.CommandHandler#handleRequest(org.json.JSONObject, com.openexchange.usm.json.ConnectorServlet, javax.servlet.http.HttpSession)
	 */
	public ResponseObject handleRequest() throws USMJSONAPIException {
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NOT_IMPLEMENTED,
				ResponseStatusCode.INTERNAL_ERROR, "No handler for request");
	}

	protected boolean getBooleanParameter(String name) throws USMJSONAPIException {
		return getBooleanParameter(name, false);
	}

	protected boolean getBooleanParameter(String name, boolean defaultValue) throws USMJSONAPIException {
		return _parameters.has(name) ? getBoolean(_parameters, name) : defaultValue;
	}

	protected String getStringParameter(String name) throws USMJSONAPIException {
		return getStringParameter(name, null);
	}

	protected String getStringParameter(String name, String defaultValue) throws USMJSONAPIException {
		return _parameters.has(name) ? getString(_parameters, name) : defaultValue;
	}

	protected boolean hasSyncID() {
		return _parameters.has(SYNCID);
	}

	protected long getSyncID() throws USMJSONAPIException {
		return getLong(_parameters, SYNCID);
	}

	protected int getOptionalLimit() throws USMJSONAPIException {
		try {
			if (!_parameters.has(LIMIT))
				return Session.NO_LIMIT;
			int limit = _parameters.getInt(LIMIT);
			if (limit > 0)
				return limit;
			if (limit == 0)
				return -1;
		} catch (JSONException e) {
			// fall through
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_BAD_LIMIT_PARAMETER,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Bad value for parameter " + LIMIT);
	}

	protected ConflictResolution getOptionalConflictResolution() throws USMJSONAPIException {
		try {
			if (!_parameters.has(CONFLICT_RESOLUTION))
				return null;
			return ConflictResolution.valueOf(_parameters.getString(CONFLICT_RESOLUTION));
		} catch (IllegalArgumentException e) {
			// fall through
		} catch (JSONException e) {
			// fall through
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_BAD_CONFLICT_RESOLUTION_PARAMETER,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Bad value for parameter " + CONFLICT_RESOLUTION);
	}

	protected UUID extractUUIDFromString(String uuid) throws USMJSONAPIException {
		try {
			if (uuid != null)
				return UUID.fromString(uuid);
		} catch (IllegalArgumentException e) {
			// Fall through to error handling
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_INVALID_UUID,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Not a valid UUID: " + uuid);
	}

	protected JSONArray getJSONArray(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getJSONArray(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_JSON_ARRAY_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No JSONArray for key " + key);
		}
	}

	protected JSONObject getJSONObject(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getJSONObject(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_JSON_OBJECT_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No JSONObject for key " + key);
		}
	}

	protected boolean getBoolean(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getBoolean(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_BOOLEAN_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No boolean for key " + key);
		}
	}

	protected String getString(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getString(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_STRING_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No String for key " + key);
		}
	}

	protected long getLong(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getLong(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_LONG_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No long for key " + key);
		}
	}

	protected int getInt(JSONObject o, String key) throws USMJSONAPIException {
		try {
			return o.getInt(key);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_INT_IN_JSON_OBJECT,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No int for key " + key);
		}
	}

	protected JSONObject getJSONObject(JSONArray a, int index) throws USMJSONAPIException {
		try {
			return a.getJSONObject(index);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_JSON_OBJECT_IN_JSON_ARRAY,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No JSONObject in JSONArray at index " + index);
		}
	}

	protected String getString(JSONArray a, int index) throws USMJSONAPIException {
		try {
			return a.getString(index);
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_NO_STRING_IN_JSON_ARRAY,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No String in JSONArray for index " + index);
		}
	}

	protected BitSet getExtraFields(ContentType type) {
		String extraFieldStorage = _session.getPersistentField(JSONSessionInitializer.getTypePropertyKey(type));
		return (extraFieldStorage == null || extraFieldStorage.length() == 0) ? new BitSet() : BitSetEncoder
				.fromBase64(extraFieldStorage);
	}

	protected JSONObject getConfiguration() throws USMJSONAPIException {
		boolean availableContentTypes = getBooleanParameter(AVAILABLE_CONTENT_TYPES);
		boolean syncContentTypes = getBooleanParameter(SYNC_CONTENT_TYPES);
		ContentTypeManager contentTypeManager = _servlet.getContentTypeManager();
		try {
			JSONObject config = new JSONObject();
			//mandatory
			config.put(USER, _session.getUser());
			config.put(DEVICE, _session.getDevice());
			config.put(TIME_ZONE, _session.getUserTimeZone().getID());
			config.put(START_DATE, String.valueOf(_session.getStartDate()));
			config.put(END_DATE, String.valueOf(_session.getEndDate()));
			config.put(FOLDER_TREE, _session.getPersistentField(FolderContentType.FOLDER_TREE));
			config.put(CONFLICT_RESOLUTION, _session.getConflictResolution());
			config.put(CONTEXT_UUID, _session.getContextUUID().toString());
			config.put(USERID, _session.getUserIdentifier());

			config.put(CUSTOM_PROPERTIES, getCustomProperties());
			//optional
			if (syncContentTypes) {
				config.put(SYNC_CONTENT_TYPES, getSyncContentTypes(contentTypeManager));
			}
			if (availableContentTypes) {
				config.put(AVAILABLE_CONTENT_TYPES, getAvailableContentTypes(contentTypeManager));
			}
			return config;
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.GET_CONFIG_JSON_ERROR, e);
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.GET_CONFIG_INTERNAL_ERROR, e);
		}
	}

	private JSONObject getCustomProperties() throws JSONException {
		JSONObject result = new JSONObject();
		for (Map.Entry<String, String> entry : _session.getPersistentFields().entrySet()) {
			String key = entry.getKey();
			if (key.startsWith(CLIENT_FIELD_PREFIX))
				result.put(key.substring(CLIENT_FIELD_PREFIX.length()), entry.getValue());
		}
		return result;
	}

	protected JSONObject getAvailableContentTypes(ContentTypeManager contentTypeManager) throws JSONException,
			USMException {
		JSONObject contentTypesObject = new JSONObject();
		ContentType[] availableContentTypes = contentTypeManager.getRegisteredContentTypes();

		for (ContentType type : availableContentTypes) {
			ContentTypeField[] fields = type.getFields();
			BitSet availableFields = new BitSet();
			availableFields.set(0, fields.length, true);
			JSONSessionInitializer.excludeInvalidFieldsFromFilter(type, availableFields);
			JSONArray fieldsArray = new JSONArray();
			for (int i = 0; i < fields.length; i++) {
				if (availableFields.get(i)) {
					String fieldName = fields[i].getFieldName();
					if (CommandConstants.ATTACHMENTS_LAST_MODIFIED.equals(fieldName))
						fieldsArray.put(CommandConstants.ATTACHMENTS);
					else
						fieldsArray.put(fieldName);
				}
			}
			if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
				for (int i = 0; i < ExternalMailContentTypeFields.getFields().length; i++) {
					fieldsArray.put(ExternalMailContentTypeFields.getFields()[i]);
				}
			}

			contentTypesObject.put(type.getID(), fieldsArray);
		}
		return contentTypesObject;
	}

	private JSONObject getSyncContentTypes(ContentTypeManager contentTypeManager) throws JSONException, USMException {
		JSONObject contentTypesObject = new JSONObject();
		ContentType[] availableContentTypes = contentTypeManager.getRegisteredContentTypes();

		for (ContentType type : availableContentTypes) {
			BitSet bitSet = _session.getFieldFilter(type);
			if (bitSet != null) {
				if (!DefaultContentTypes.MAIL_ID.equals(type.getID()))
					bitSet.or(getExtraFields(type));
				ContentTypeField[] fields = type.getFields();
				JSONArray fieldsArray = new JSONArray();
				for (int i = 0; i < fields.length; i++) {
					if (bitSet.get(i)) {
						String fieldName = fields[i].getFieldName();
						if (CommandConstants.ATTACHMENTS_LAST_MODIFIED.equals(fieldName))
							fieldsArray.put(CommandConstants.ATTACHMENTS);
						else
							fieldsArray.put(fieldName);

					}
				}

				if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
					BitSet extraFieldsBitSet = getExtraFields(type);
					for (int i = 0; i < ExternalMailContentTypeFields.getFields().length; i++) {
						if (extraFieldsBitSet.get(i))
							fieldsArray.put(ExternalMailContentTypeFields.getFields()[i]);
					}
				}
				contentTypesObject.put(type.getID(), fieldsArray);
			}
		}
		return contentTypesObject;
	}

	public Session getSession() {
		return _session;
	}

	protected void setSession(Session session) {
		_session = session;
		_session.setCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS, null);
		_session.setCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME, null);
	}

	protected abstract String[] getRequiredParameters();

	protected abstract String[] getOptionalParameters();
	
	
	// TODO Optimization: Use DataObjectSet instead of array for faster lookup
	protected DataObject getDataObjectByUUID(DataObject[] objects, UUID uuid) throws USMJSONAPIException {
		if (objects != null) {
			for (DataObject o : objects) {
				if (o.getUUID().equals(uuid)) {
					if (o.getChangeState() == ChangeState.UNMODIFIED)
						return o;
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_MULTIPLE_ACTIONS_ON_DATA_OBJECT,
							ResponseStatusCode.WRONG_MISSING_PARAMETERS,
							"More than one action requested on DataObject with uuid " + uuid);
				}
			}
		}
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_DATA_OBJECT_NOT_FOUND,
				ResponseStatusCode.WRONG_MISSING_PARAMETERS, "No DataObject with uuid " + uuid);
	}
	
	protected Folder getFolderByUUID(String folderUUID) throws USMJSONAPIException {
		if (DefaultContentTypes.GROUPS_ID.equals(folderUUID) || DefaultContentTypes.RESOURCES_ID.equals(folderUUID))
			return _session.getDummyContentTypeFolder(_servlet.getContentTypeManager().getContentType(folderUUID));

		UUID uuid = extractUUIDFromString(folderUUID);
		Folder[] folders = null;
		try {
			folders = _session.getCachedFolders();
		} catch (USMException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_CANNOT_FIND_FOLDER_BY_UUID,
					e);
		}
		if (folders != null)
			return (Folder) getDataObjectByUUID(folders, uuid);
		throw new USMJSONAPIException(ConnectorBundleErrorCodes.COMMAND_UNKNOWN_FOLDER_ID,
				ResponseStatusCode.UNKNOWN_UUID, "Unknown folderid");
	}
}
