/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.attachments;

import java.util.*;

import org.apache.commons.logging.Log;
import org.json.*;

import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.contenttypes.util.UtilConstants;
import com.openexchange.usm.ox_json.*;

/**
 * Handler for the interaction with the OX server for PIM Attachments.
 * 
 * @author ldo
 *
 */
public class PIMAttachmentTransferHandler {

	private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

	private static final String CONTENT_TYPE = "content_type";

	private static final String FILE_PART_NAME = "file_0";

	private static final String JSON_PART_NAME = "json_0";

	private static final String ACTION_DOCUMENT = "document";

	private static final String ACTION_DETACH = "detach";

	private static final String ACTION_ATTACH = "attach";

	private static final String OX_AJAX_PATH = "attachment";

	private final Log _journal;
	private final OXJSONAccess _ajaxAccess;

	public PIMAttachmentTransferHandler(Log journal, OXJSONAccess ajaxAccess) {
		_journal = journal;
		_ajaxAccess = ajaxAccess;
	}

	public long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Creating attachment for object" + object.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		JSONObject jsonAtt = attachment.toJSONObject();
		String[] names = PIMAttachment.FIELDS.values().toArray(new String[PIMAttachment.FIELDS.size()]);
		JSONObject requestBody;
		try {
			requestBody = new JSONObject(jsonAtt, names);
			requestBody.put(UtilConstants.ATTACHED, object.getID());
			requestBody.put(UtilConstants.FOLDER, object.getParentFolderID());
			requestBody.put(UtilConstants.MODULE, object.getContentType().getCode());
		} catch (JSONException e1) {
			requestBody = null;
			throw new USMException(PIMAttachmentErrorCodes.CAN_NOT_CREATE_REQUEST_BODY,
					"Create New Attachment: Can not create request body.");
		}
		JSONResult result = _ajaxAccess.storeResource(OX_AJAX_PATH, ACTION_ATTACH, object.getSession(), parameters,
				requestBody, attachment.getData(), attachment.getMimeType(), JSON_PART_NAME, FILE_PART_NAME);
		JSONArray array = extractArrayResult(result);
		try {
			attachment.setOxId(array.getInt(0));
		} catch (JSONException e) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.ID_NOT_EXISTING,
					"The OX Server did not return the id of the new attachment");
		}
		long timestamp = getLastModifiedOfNewestAttachment(object);
		return timestamp;
	}

	public PIMAttachments getAllAttachments(DataObject object) throws USMException {
		return getAllAttachments(object, null, null);
	}

	private String createColumnsParameter() {
		StringBuilder sb = new StringBuilder(100);
		for (Integer fieldId : PIMAttachment.FIELDS.keySet()) {
			sb.append(fieldId);
			sb.append(",");
		}
		return sb.toString();
	}

	public PIMAttachments getAllAttachments(DataObject object, String sort, String order) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Read all attachments for object" + object.getID());
		if (object.getID() == null) //this means that the object was created by client and still doesn't have ox id
			return new PIMAttachments(0, new PIMAttachment[0]);
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ATTACHED, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.MODULE, String.valueOf(object.getContentType().getCode()));
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter());
		if (sort != null) {
			parameters.put(UtilConstants.SORT, sort);
			if (order != null)
				parameters.put(UtilConstants.ORDER, order);
		}

		JSONResult result = _ajaxAccess.doGet(OX_AJAX_PATH, UtilConstants.ACTION_ALL, object.getSession(), parameters);
		JSONArray array = extractArrayResult(result);
		if (array.length() == 0)
			return null;
		PIMAttachment[] pimAtts = new PIMAttachment[array.length()];
		for (int i = 0; i < array.length(); i++) {
			try {
				JSONArray attachment = array.getJSONArray(i);
				UUID uuid = UUID.randomUUID();
				byte[] data = getAttachmentData(object, attachment.getInt(0));
				pimAtts[i] = new PIMAttachment(uuid, attachment, data);
			} catch (JSONException e) {
				throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_1,
						"The result doesn't consist of JSONObjects");
			}
		}
		long timestamp = extractTimestamp(result);
		setSavedUUIDs(object, pimAtts);
		return new PIMAttachments(timestamp, pimAtts);
	}

	private void setSavedUUIDs(DataObject object, PIMAttachment[] pimAtts) {
		PIMAttachments oldAttachments = (PIMAttachments) object
				.getFieldContent(UtilConstants.ATTACHMENTS_LAST_MODIFIED);
		if (oldAttachments == null)
			return;
		for (int i = 0; i < pimAtts.length; i++) {
			for (int j = 0; j < oldAttachments.getAttachments().length; j++) {
				if (pimAtts[i].getOxId() == oldAttachments.getAttachment(j).getOxId())
					pimAtts[i].setUUID(oldAttachments.getAttachment(j).getUUID());
			}
		}
	}

	private JSONArray extractArrayResult(JSONResult result) throws OXCommunicationException, InternalUSMException {
		if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.ERROR_ON_STORE_RESOURCE_1,
					"OX server returned error", result.getJSONObject());
		}
		if (result.getResultType() != JSONResultType.JSONObject)
			throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_2,
					"OX server didn't send expected JSONObject");
		try {
			return result.getJSONObject().getJSONArray(UtilConstants.RESULT_DATA);
		} catch (JSONException e) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_3,
					"OX server did not send data array", e);
		}
	}

	private long extractTimestamp(JSONResult result) throws OXCommunicationException, InternalUSMException {
		if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.ERROR_ON_STORE_RESOURCE_2,
					"OX server returned error", result.getJSONObject());
		}
		if (result.getResultType() != JSONResultType.JSONObject)
			throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_5,
					"OX server didn't send expected JSONObject");
		try {
			return result.getJSONObject().getLong(UtilConstants.TIMESTAMP);
		} catch (JSONException e) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_6,
					"OX server didn't send timestamp", e);
		}
	}

	public long deleteAttachments(DataObject object, PIMAttachment[] attachmentsToDelete) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Delete attachments for object" + object.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ATTACHED, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.MODULE, String.valueOf(object.getContentType().getCode()));
		JSONArray requestBody = new JSONArray();
		for (int i = 0; i < attachmentsToDelete.length; i++) {
			requestBody.put(attachmentsToDelete[i].getOxId());
		}
		JSONResult result = _ajaxAccess
				.doPut(OX_AJAX_PATH, ACTION_DETACH, object.getSession(), parameters, requestBody);
		if (result.getResultType() == JSONResultType.Error) {
			throw new OXCommunicationException(PIMAttachmentErrorCodes.INVALID_OX_RESULT_4,
					"OX returned error on deleting attachments", result.getJSONObject());
		}
		//long timestamp = extractTimestamp(result);
		//TODO: remove this call when OX changes the response in create attachment!!!
		long timestamp = getLastModifiedOfNewestAttachment(object);
		return timestamp;
	}

	private long getLastModifiedOfNewestAttachment(DataObject object) throws USMException {
		DataObject objectCopy = object.createCopy(false);
		BitSet fields = new BitSet();
		fields.set(object.getFieldIndex(UtilConstants.ATTACHMENTS_LAST_MODIFIED));
		objectCopy.getContentType().getTransferHandler().readDataObject(objectCopy, fields);
		PIMAttachments attachments = (PIMAttachments) objectCopy
				.getFieldContent(UtilConstants.ATTACHMENTS_LAST_MODIFIED);
		long timestamp = attachments == null ? 0 : attachments.getTimestamp();
		return timestamp;
	}

	public byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException {
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.ATTACHED, object.getID());
		parameters.put(UtilConstants.MODULE, String.valueOf(object.getContentType().getCode()));
		parameters.put(UtilConstants.ID, String.valueOf(attachmentId));
		parameters.put(UtilConstants.ACTION, ACTION_DOCUMENT);
		parameters.put(CONTENT_TYPE, APPLICATION_OCTET_STREAM);
		OXResource result = _ajaxAccess.getResource(object.getSession(), OX_AJAX_PATH, parameters);
		return result.getData();
	}

}
