/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.mail.impl;

import java.util.*;

import org.apache.commons.logging.Log;
import org.json.*;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.contenttypes.util.*;
import com.openexchange.usm.ox_json.*;

public class MailContentTypeTransferHandlerImpl extends AbstractTransferHandler {

	private static final String FORCE = "force";
	private static final String FOLDER = "folder";
	private static final String TEXT = "text";
	private static final String ONE = "1";
	private static final String HARDDELETE = "harddelete";
	private static final String RAW = "raw";
	private static final String VIEW = "view";
	private static final String FLAGS = "flags";
	private static final String UNSEEN = "unseen";
	private static final String SRC = "src";
	private static final String SAVE = "save";
	private static final String ZERO = "0";

	private final BitSet _getAllBitSet = new BitSet();
	private final int _idIndex;
	private final int _receivedDateIndex;

	public MailContentTypeTransferHandlerImpl(MailContentType contentType, Log log, OXJSONAccess ajaxAccess) {
		super(contentType, log, ajaxAccess);
		_idIndex = getFieldIndex(MailConstants.FIELD_ID);
		_receivedDateIndex = getFieldIndex(MailConstants.RECEIVED_DATE);
		_getAllBitSet.set(_idIndex);
		_getAllBitSet.set(_receivedDateIndex);
	}

	private int getFieldIndex(String fieldId) {
		ContentTypeField[] fields = _contentType.getFields();
		for (int i = 0; i < fields.length; i++) {
			if (fields[i].getFieldName().equals(fieldId))
				return i;
		}
		throw new USMStartupException(USMContentTypesMailErrorCodes.REQUIRED_FIELDS_MISSING_ERROR,
				"Required field is missing from ContentType: " + fieldId);
	}

	public void sendMail(String mail, Session session) throws USMException {
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(SRC, ONE);
		JSONResult result = _ajaxAccess.doPut(UtilConstants.MAIL_PATH, UtilConstants.ACTION_NEW, session, parameters,
				mail);
		checkResult(result, JSONResultType.JSONObject);
	}

	public Object getMailAttachment(Session session, String folderId, String mailId, String attachmentId)
			throws USMException {
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.FOLDER, folderId);
		parameters.put(UtilConstants.ID, mailId);
		parameters.put(MailConstants.ATTACHMENT, attachmentId);
		parameters.put(SAVE, ONE);
		parameters.put(UtilConstants.ACTION, MailConstants.ATTACHMENT);

		OXResource result = _ajaxAccess.getResource(session, UtilConstants.MAIL_PATH, parameters);
		return result;
	}

	public String createNewMail(String folderId, int flag, String mailMIMEdataBlock, Session session)
			throws USMException {
		//TODO: use import instead of new (see svn history!)
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(SRC, ONE);
		parameters.put(UtilConstants.FOLDER, folderId);
		parameters.put(FLAGS, String.valueOf(flag));
		parameters.put(FORCE, ONE);
		JSONResult result = _ajaxAccess.doPut(UtilConstants.MAIL_PATH, UtilConstants.ACTION_NEW, session, parameters,
				mailMIMEdataBlock);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		try {
			return o.getJSONObject(UtilConstants.RESULT_DATA).getString(UtilConstants.RESULT_ID);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER4, null,
					"Missing result ID on new " + _contentType.getID(), e);
		}
	}

	public DataObject getReplyForward(Session session, String messageID, String folderID, String action)
			throws USMException {
		DataObject mail = _contentType.newDataObject(session);
		mail.setID(messageID);
		mail.setParentFolderID(folderID);
		mail.commitChanges();
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, messageID);
		parameters.put(UtilConstants.FOLDER, folderID);
		parameters.put(UNSEEN, ONE);
		parameters.put(VIEW, TEXT);

		JSONResult result = _ajaxAccess.doGet(UtilConstants.MAIL_PATH, action, session, parameters);
		if (result.getResultType() == JSONResultType.Error)
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER2, result
					.getJSONObject());
		BitSet reqFields = createRequestedFields(_contentType);
		reqFields.set(18); //attachments
		updateDataObjectFromJSONResult(mail, reqFields, result);
		mail.setID(messageID);
		mail.setParentFolderID(folderID);
		return mail;
	}

	private BitSet createRequestedFields(ContentType type) {
		ContentTypeField[] fields = type.getFields();
		BitSet requestedFields = new BitSet();
		for (int i = 0; i < fields.length; i++) {
			if (fields[i].getFieldID() >= 0) {
				requestedFields.set(i);
			}
		}
		return requestedFields;
	}

	@Override
	public void writeDeletedDataObject(DataObject object) throws USMException {
		writeDeletedDataObject(object, Boolean.TRUE.equals(object.getSession().getCustomProperty(
				Session.MAIL_HARDDELETE_PROPERTY)));
	}

	private void writeDeletedDataObject(DataObject object, boolean hardDelete) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Delete mail " + object.getID());
		checkTypeToWrite(object, DefaultContentTypes.MAIL_ID);
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(object.getTimestamp()));
		if (hardDelete)
			parameters.put(HARDDELETE, ONE);
		JSONObject jObj = new JSONObject();
		JSONArray array = new JSONArray();
		try {
			jObj.put(UtilConstants.ID, object.getID());
			jObj.put(UtilConstants.FOLDER, object.getParentFolderID());

			array.put(jObj);
		} catch (JSONException e) {
			throw new USMIllegalArgumentException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER2,
					"Duplicate or null entry in provided key/value pairs");
		}
		JSONResult result = _ajaxAccess.doPut(UtilConstants.MAIL_PATH, UtilConstants.ACTION_DELETE,
				object.getSession(), parameters, array);
		checkForNotDeletedObjects(object, result);
	}

	@Override
	public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp)
			throws USMException {
		HashMap<String, String> parameters = new HashMap<String, String>();
		//XXX Not IMAP - Response: Just an empty JSON array is going to be returned since this action cannot be applied to IMAP. 

		//parameters.put(UtilConstants.FOLDER, folder.getID());
		//parameters.put(UtilConstants.COLUMNS, createColumnsParameter(folder, requestedFields));
		//parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));

		if (_journal.isDebugEnabled())
			_journal.debug(folder.getSession() + " Read updates in " + folder.getID());
		JSONResult result = _ajaxAccess.doGet(UtilConstants.MAIL_PATH, UtilConstants.ACTION_UPDATES, folder
				.getSession(), parameters);
		checkResult(result, JSONResultType.JSONObject);
		return getListOfUpdates(folder, requestedFields, result);
	}

	@Override
	public void writeNewDataObject(DataObject object) throws USMException {
		if (_journal.isErrorEnabled())
			_journal.error(object.getSession() + " Creating a mail isn't support");
		throw new UnsupportedContentOperationException(USMContentTypesMailErrorCodes.CREATE_ERROR,
				"Creating a mail is not supported");
	}

	@Override
	public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
		checkTypeToWrite(object, DefaultContentTypes.MAIL_ID);
		HashMap<String, String> parameters = new HashMap<String, String>();

		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getOriginalParentFolderID());
		//XXX is it to use? In HTTP Ajax API timestamp isn't use for mail update
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));

		Number color = (Number) object.getFieldContent(MailConstants.MAIL_UPDATE_COLOR_LABEL);
		if (object.isFieldModified(MailConstants.MAIL_UPDATE_FLAGS)) {
			Number flagValue = (Number) object.getFieldContent(MailConstants.MAIL_UPDATE_FLAGS);
			int flagsToSet = flagValue == null ? 0 : flagValue.intValue();
			int flagsToRemove = MailContentType.ALL_FLAGS - flagsToSet;
			if (flagsToSet > 0)
				updateFlags(object, parameters, color, flagsToSet, true);
			if (flagsToRemove > 0)
				updateFlags(object, parameters, color, flagsToRemove, false);
		} else if (object.isFieldModified(MailConstants.MAIL_UPDATE_COLOR_LABEL)) {
			updateFlags(object, parameters, color, 0, true);
		}
		JSONObject requestBody = new JSONObject();
		if (object.isFieldModified(UtilConstants.PARENTFOLDER_ID)) {
			try {
				requestBody.put(UtilConstants.PARENTFOLDER_ID, object.getParentFolderID());
			} catch (JSONException ignored) {
				// can not happen
			}
		}
		JSONResult result = _ajaxAccess.doPut(UtilConstants.MAIL_PATH, UtilConstants.ACTION_UPDATE,
				object.getSession(), parameters, requestBody);
		if (result.getResultType() == JSONResultType.Error) {
			checkConflictingChange(result.getJSONObject());
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER5, result
					.getJSONObject());
		} else if (result.getResultType() == JSONResultType.JSONObject
				&& object.isFieldModified(UtilConstants.PARENTFOLDER_ID) && result.getJSONObject().has("data")) {
			//if the email is moved to another folder, update the ID of the object with the new id
			JSONObject data = result.getJSONObject().optJSONObject("data");
			if (data != null) {
				String id = data.optString(UtilConstants.ID, null);
				if (id != null)
					object.setID(id);
			}
		}
	}

	private void updateFlags(DataObject object, Map<String, String> parameters, Number color, int flag, boolean set)
			throws AuthenticationFailedException, OXCommunicationException, InternalUSMException {
		if (flag > 256) {
			updateFlags(object, parameters, color, 256, set);
			flag &= 255;
		}
		JSONObject requestBody = new JSONObject();
		try {
			requestBody.put(MailConstants.MAIL_UPDATE_FLAGS, flag);
			requestBody.put(MailConstants.MAIL_UPDATE_VALUE, set);
			if (color != null)
				requestBody.put(MailConstants.MAIL_UPDATE_COLOR_LABEL, color);
		} catch (JSONException ignored) {
			// can not happen, no null key
		}
		JSONResult result = _ajaxAccess.doPut(UtilConstants.MAIL_PATH, UtilConstants.ACTION_UPDATE,
				object.getSession(), parameters, requestBody);
		if (result.getResultType() == JSONResultType.Error) {
			checkConflictingChange(result.getJSONObject());
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER4, result
					.getJSONObject());
		}
	}

	@Override
	public void readDataObject(DataObject object, BitSet requestedFields) throws USMException {
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UNSEEN, ONE);
		parameters.put(VIEW, RAW);
		readDataObject(object, requestedFields, parameters);
	}

	public JSONObject readStructuredMail(DataObject object, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Read structured mail with id " + object.getID());
		checkTypeToWrite(object, _contentType.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UNSEEN, ONE);
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_GET_STRUCTURE, object
				.getSession(), parameters);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject resultObject = null;
		try {
			resultObject = result.getJSONObject().getJSONObject("data");
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER6, object,
					"read structured mail: data  not present", e);
		}
		return resultObject;
	}

	public byte[] readMailSource(DataObject object, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Read mail with id " + object.getID());
		checkTypeToWrite(object, _contentType.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UNSEEN, ONE);
		parameters.put(SRC, ONE);
		parameters.put(SAVE, ONE);
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getParentFolderID());
		parameters.put(UtilConstants.ACTION, UtilConstants.ACTION_GET);
		return _ajaxAccess.getResource(object.getSession(), getOXAjaxAccessPath(), parameters).getData();
	}

	@Override
	protected String getOXAjaxAccessPath() {
		return UtilConstants.MAIL_PATH;
	}

	@Override
	public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
		if (AbstractTransferHandler.CHECK_ACCESS_RIGHTS_BEFORE_FOLDER_READ && !hasFolderReadRights(folder))
			return EMPTY_LIST;
		Session session = folder.getSession();
		long startDate = session.getStartDate();
		long endDate = session.getEndDate();
		int mailLimit = session.getMailLimit();
		if (startDate == 0L && endDate == Long.MAX_VALUE && mailLimit <= 0)
			return super.readFolderContent(folder, requestedFields);
		DataObject[] idList = super.readFolderContent(folder, _getAllBitSet);
		List<DataObject> filteredList = new ArrayList<DataObject>();
		for (DataObject o : idList) {
			Long t = (Long) o.getFieldContent(_receivedDateIndex);
			if (t == null || (t >= startDate && t <= endDate))
				filteredList.add(o);
		}
		if (filteredList.isEmpty())
			return EMPTY_LIST;
		if (filteredList.size() > mailLimit) {
			Collections.sort(filteredList, new Comparator<DataObject>() {
				public int compare(DataObject a, DataObject b) {
					// Sort descending, if no time is present, treat as newest
					Long ta = (Long) a.getFieldContent(_receivedDateIndex);
					Long tb = (Long) b.getFieldContent(_receivedDateIndex);
					if (ta == null) {
						return (tb == null) ? 0 : -1;
					} else if (tb == null) {
						return 1;
					} else {
						return tb.compareTo(ta);
					}
				}
			});
			filteredList.subList(mailLimit, filteredList.size()).clear();
		}
		JSONArray ids = new JSONArray();
		for (DataObject o : filteredList) {
			JSONObject element = new JSONObject();
			try {
				element.put(FOLDER, folder.getID());
				element.put(UtilConstants.ID, o.getID());
			} catch (JSONException e) {
				// Ignored because it can not happen
			}
			ids.put(element);
		}
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		try {
			JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_LIST, session,
					parameters, ids);
			DataObject[] listOfResults = getListOfResults(folder, requestedFields, result);
			return extractDeletedMails(session, listOfResults);
		} catch (OXCommunicationException e) {
			//catch the OXCommunication exception and than get the access rights
			if (hasFolderReadRights(folder)) {
				throw e;
			} else {
				DataObject[] emptyList = new DataObject[0];
				return emptyList;
			}
		}
	}

	private DataObject[] extractDeletedMails(Session session, DataObject[] listOfResults) {
		Object o = session.getCustomProperty(MailContentType.FILTER_MARKED_AS_DELETED_MAILS);
		boolean filterMails = (o instanceof Boolean) ? ((Boolean) o) : false;
		if (filterMails) {
			List<DataObject> filteredMails = new ArrayList<DataObject>();
			for (int i = 0; i < listOfResults.length; i++) {
				if ((((Integer) listOfResults[i].getFieldContent(MailConstants.FLAGS)) & 0x2) == 0) {
					filteredMails.add(listOfResults[i]);
				}
			}
			listOfResults = filteredMails.toArray(new DataObject[filteredMails.size()]);
		}
		return listOfResults;
	}
}
