/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.contacts.impl;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.URLDecoder;
import java.util.*;

import javax.imageio.ImageIO;

import org.apache.commons.logging.Log;
import org.json.*;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.*;
import com.openexchange.usm.datatypes.contacts.Image;
import com.openexchange.usm.datatypes.contacts.ImageDataType;
import com.openexchange.usm.ox_json.*;
import com.openexchange.usm.util.Toolkit;

/**
 * 
 * @author ibr
 *
 */
public class ContactContentTypeTransferHandler extends AbstractTransferHandler {

	private static final String OR_SEARCH = "orSearch";
	private static final String EMAIL_AUTO_COMPLETE = "emailAutoComplete";
	// The field "image1" is a complex element that is represented by several fields in OX.
	// This String determines which extra columns have to be fetched on a folder read.
	private static final String EXTRA_IMAGE_COLUMNS = ",597,606,601";
	private static final String ACTION_SEARCH = "search";

	private final int _image1FieldIndex;

	public ContactContentTypeTransferHandler(ContactContentType contentType, Log log, OXJSONAccess ajaxAccess) {
		super(contentType, log, ajaxAccess);
		ContentTypeField[] fields = contentType.getFields();
		for (int i = 0; i < fields.length; i++) {
			if (ImageDataType.IMAGE1.equals(fields[i].getFieldName())) {
				_image1FieldIndex = i;
				return;
			}
		}
		throw new IllegalStateException("No image1 field in ContactContentType");
	}

	@Override
	protected String getOXAjaxAccessPath() {
		return UtilConstants.CONTACT_PATH;
	}

	public Map<String, DataObject[]> resolveRecipients(List<String> toList, Session session, BitSet requestedFields)
			throws AuthenticationFailedException, OXCommunicationException, UnsupportedContentOperationException {
		Map<String, DataObject[]> result = new HashMap<String, DataObject[]>();
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		JSONObject requestBody = new JSONObject();

		for (String to : toList) {
			try {
				requestBody.put(ContactConstants.LAST_NAME, to);
				requestBody.put(ContactConstants.FIRST_NAME, to);
				requestBody.put(ContactConstants.DISPLAY_NAME, to);
				requestBody.put(EMAIL_AUTO_COMPLETE, Boolean.TRUE.toString());
				requestBody.put(OR_SEARCH, Boolean.TRUE.toString());
			} catch (JSONException e) {
				_journal.error(session + " Can not create request body for Resolving recipients", e);
			}

			JSONResult jResult = _ajaxAccess.doPut(getOXAjaxAccessPath(), ACTION_SEARCH, session, parameters,
					requestBody);
			//fill List
			result.put(to, parseRecipientsResult(jResult, session, requestedFields));
		}
		return result;
	}

	private DataObject[] parseRecipientsResult(JSONResult jResult, Session session, BitSet requestedFields) {
		JSONArray jsonar;

		try {
			jsonar = jResult.getJSONObject().getJSONArray(UtilConstants.RESULT_DATA);
		} catch (JSONException e1) {
			return new DataObject[0];
		}
		if (jsonar.length() == 0)
			return new DataObject[0];

		ArrayList<DataObject> dataObjects = new ArrayList<DataObject>();
		for (int j = 0; j < jsonar.length(); j++) {
			// get array which contains the information specified by the corresponding identifiers in the columns parameter
			try {
				JSONArray columnsArray = jsonar.getJSONArray(j);
				DataObject newDataObject = _contentType.newDataObject(session);
				updateDataObjectFromJSONArray(newDataObject, requestedFields, columnsArray);
				dataObjects.add(newDataObject);

				String emailAddress2 = (String) newDataObject.getFieldContent(ContactConstants.EMAIL2);
				if (emailAddress2 != null && !"".equals(emailAddress2)) {
					DataObject objCopy = newDataObject.createCopy(true);
					objCopy.setFieldContent(ContactConstants.EMAIL1, null);
					objCopy.setFieldContent(ContactConstants.EMAIL3, null);
					dataObjects.add(objCopy);
				}

				String emailAddress3 = (String) newDataObject.getFieldContent(ContactConstants.EMAIL3);
				if (emailAddress3 != null && !"".equals(emailAddress3)) {
					DataObject objCopy = newDataObject.createCopy(true);
					objCopy.setFieldContent(ContactConstants.EMAIL1, null);
					objCopy.setFieldContent(ContactConstants.EMAIL2, null);
					dataObjects.add(objCopy);
				}
				//remove additional email addresses from the original object
				newDataObject.setFieldContent(ContactConstants.EMAIL2, null);
				newDataObject.setFieldContent(ContactConstants.EMAIL3, null);

			} catch (JSONException je) {
				_journal.error(session + " Can not read the result data for resolving recipients", je);
			} catch (InternalUSMException e) {
				_journal.error(session + " Internal error resolving recipients", e);
			}
		}
		return dataObjects.toArray(new DataObject[dataObjects.size()]);

	}

	public byte[] getPictureData(DataObject dataObject, String pictureFormat, int maxSize)
			throws AuthenticationFailedException, OXCommunicationException, PictureConversionException {
		Image image = (Image) dataObject.getFieldContent(ImageDataType.IMAGE1);
		if (image == null)
			return null;
		String url = image.getUrl();
		if (Toolkit.isNullOrEmpty(url))
			return null;
		try {
			url = URLDecoder.decode(url, "UTF-8");
		} catch (UnsupportedEncodingException e1) {
			return null;
		}
		//		String imageContentType = (String) dataObject.getFieldContent("image1_content_type");
		OXResource sourceData = _ajaxAccess.getResource(dataObject.getSession(), url, null);
		int sourceSize = sourceData.getData().length;
		if (sourceSize == 0)
			return null;
		if (sourceData.getContentType().equalsIgnoreCase("image/" + pictureFormat) && sourceSize <= maxSize)
			return sourceData.getData();
		// Conversion required because picture is in the wrong format or too big
		// The scale factor is just an estimate, since we have to scale to some target byte size in the output
		// The picture is either too big or in the wrong format, in either case we assume that if we scale to
		// about 90% of the limit divided by its current size (if the picture is bigger than that), our result
		// will be within the limit. We have a fall back algorithm, that halves the size up to 2 times if our
		// estimate should be too big. After that we return null (assuming that the provided limit is too small
		// for the image format).
		float sizeLimit = maxSize * 0.9f;
		float scaleFactor = (sourceSize > sizeLimit) ? sizeLimit / sourceSize : 1.0f;
		BufferedImage sourceImage;
		try {
			sourceImage = ImageIO.read(new ByteArrayInputStream(sourceData.getData()));
		} catch (IOException e) {
			throw new OXCommunicationException(USMContentTypesContactsErrorCodes.IMAGE_ERROR_NUMBER1,
					"Picture received from OX server could not be read into a BufferedImage", e);
		}
		int type = sourceImage.getType();
		if (type == 0)
			type = BufferedImage.TYPE_INT_RGB;
		for (int i = 0; i < 3; i++) {
			int scaledWidth = Math.round(sourceImage.getWidth() * scaleFactor);
			int scaledHeight = Math.round(sourceImage.getHeight() * scaleFactor);
			BufferedImage scaledImage = sourceImage;
			if (scaleFactor < 1.0f) {
				scaledImage = new BufferedImage(scaledWidth, scaledHeight, type);
				Graphics2D g = scaledImage.createGraphics();
				g.drawImage(sourceImage, 0, 0, scaledWidth, scaledHeight, null);
				g.dispose();
			}
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			try {
				ImageIO.write(scaledImage, pictureFormat, os);
			} catch (IOException e) {
				throw new PictureConversionException(USMContentTypesContactsErrorCodes.IMAGE_ERROR_NUMBER2,
						"Can't write picture in type " + pictureFormat, e);
			}
			int size = os.size();
			if (size == 0)
				return null;
			if (size <= maxSize)
				return os.toByteArray();
			scaleFactor *= 0.5f; // estimate was too big, half the size and try again
		}
		return null; // one quarter of the estimate is still too big, maybe the limit is too small, return "no image" instead
	}

	@Override
	protected void appendExtraColumns(StringBuilder columns, int i) {
		if (i == _image1FieldIndex)
			columns.append(EXTRA_IMAGE_COLUMNS);
	}

	@Override
	protected void updateDataObjectFromJSONArray(DataObject destination, BitSet requestedFields, JSONArray array)
			throws InternalUSMException {
		int len = checkAndGetRequestedFieldLength(requestedFields);
		int j = 0;
		ContentTypeField[] fields = destination.getContentType().getFields();
		for (int i = 0; i < len; i++) {
			if (requestedFields.get(i)) {
				updateDataObjectFieldFromJSONArray(array, j, fields[i], destination);
				if (fields[i].getFieldName().equals(ImageDataType.IMAGE1))
					j += 4;
				else
					j++;
			}
		}
	}

	private boolean containsImage(Image i) {
		return (i != null) && i.getContentType() != null && i.getContentType().length() > 0 && i.getData() != null;
	}

	@Override
	public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
		Image i = (Image) object.getFieldContent(ImageDataType.IMAGE1);
		if (containsImage(i)) {
			if (_journal.isDebugEnabled())
				_journal.debug(object.getSession() + " Updating contact with picture");
			checkTypeToWrite(object, _contentType.getID());
			HashMap<String, String> parameters = new HashMap<String, String>();
			parameters.put(UtilConstants.ID, object.getID());
			parameters.put(UtilConstants.FOLDER, object.getOriginalParentFolderID());
			parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
			JSONObject requestBody = createRequestBody(object, true, false);

			JSONResult result = _ajaxAccess.storeResource(getOXAjaxAccessPath(), UtilConstants.ACTION_UPDATE, object
					.getSession(), parameters, requestBody, i.getData(), i.getContentType());
			checkResult(result, JSONResultType.JSONObject);
			// XXX Shouldn't we use the timestamp returned from the OX server ?
			object.setTimestamp(timestamp);
		} else
			super.writeUpdatedDataObject(object, timestamp);
	}

	@Override
	public void writeNewDataObject(DataObject object) throws USMException {
		Image i = (Image) object.getFieldContent(ImageDataType.IMAGE1);
		checkTypeToWrite(object, _contentType.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		JSONObject requestBody = createRequestBody(object, false, true);
		JSONResult result;

		if (containsImage(i)) {
			if (_journal.isDebugEnabled())
				_journal.debug(object.getSession() + " New contact with picture");

			result = _ajaxAccess.storeResource(getOXAjaxAccessPath(), UtilConstants.ACTION_NEW, object.getSession(),
					parameters, requestBody, i.getData(), i.getContentType());
		} else {
			if (_journal.isDebugEnabled())
				_journal.debug(object.getSession() + " New " + _contentType.getID());

			if (requestBody.has(ImageDataType.IMAGE1) && requestBody.isNull(ImageDataType.IMAGE1))
				requestBody.remove(ImageDataType.IMAGE1);

			result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_NEW, object.getSession(),
					parameters, requestBody);
		}

		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		readOptionalTimestamp(o, object);
		try {
			object.setID(o.getJSONObject(UtilConstants.RESULT_DATA).getString(UtilConstants.RESULT_ID));
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER1, object,
					"Missing result ID on new " + _contentType.getID(), e);
		}
	}

	@Override
	protected void addFieldToRequestBody(DataObject object, boolean isOnlyModifiedFields,
			boolean isIdshouldNotBePresent, JSONObject requestBody, ContentTypeField field, int fieldIndex)
			throws InternalUSMException {
		if (!ContactConstants.NUMBER_OF_ATTACHMENTS.equals(field.getFieldName()))
			super.addFieldToRequestBody(object, isOnlyModifiedFields, isIdshouldNotBePresent, requestBody, field,
					fieldIndex);
	}

	public DataObject[] getAllUsers(Session session, BitSet requestedFields) throws AuthenticationFailedException,
			OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Read all users ");
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		JSONResult result = _ajaxAccess.doGet(UtilConstants.USERS_PATH, UtilConstants.ACTION_ALL, session, parameters);
		return getListOfResults(session, requestedFields, result);
	}

	private DataObject[] getListOfResults(Session session, BitSet requestedFields, JSONResult result)
			throws OXCommunicationException, InternalUSMException {
		JSONArray jsonar = null;
		long timestamp = 0L;
		checkResult(result, JSONResultType.JSONObject);
		JSONObject jsonObject = result.getJSONObject();
		try {
			jsonar = jsonObject.getJSONArray(UtilConstants.RESULT_DATA);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER7, null,
					"getListOfResults: data array not present", e);
		}
		if (jsonar.length() == 0)
			return new DataObject[0];
		try {
			if (jsonObject.has(UtilConstants.RESULT_TIMESTAMP))
				timestamp = jsonObject.getLong(UtilConstants.RESULT_TIMESTAMP);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.INVALID_TIMESTAMP_NUMBER3, null,
					"getListOfResults: invalid timestamp", e);
		}
		DataObject[] dataObjects = new DataObject[jsonar.length()];
		for (int j = 0; j < jsonar.length(); j++) {
			// get array which contains the information specified by the corresponding identifiers in the columns parameter
			try {
				JSONArray columnsArray = jsonar.getJSONArray(j);
				DataObject newDataObject = createDataObject(columnsArray, session, requestedFields, 0);
				dataObjects[j] = newDataObject;
				newDataObject.setTimestamp(timestamp);
			} catch (JSONException je) {
				throw generateException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER6, null,
						"getListOfResults: error while reading index " + j + " of users ", je);
			}
		}
		return dataObjects;
	}

	private DataObject createDataObject(JSONArray columnsArray, Session session, BitSet requestedFields, int idIndex)
			throws InternalUSMException, OXCommunicationException {
		String objId;
		try {
			objId = columnsArray.getString(idIndex);
		} catch (JSONException e) {
			throw generateException(USMContentTypesUtilErrorCodes.ID_NOT_PRESENT_NUMBER6, null,
					"read users: no id for new object", e);
		}
		DataObject dataObj = _contentType.newDataObject(session);
		dataObj.setID(objId);
		updateDataObjectFromJSONArray(dataObj, requestedFields, columnsArray);
		return dataObj;
	}

	public void getUser(DataObject object, BitSet requestedFields) throws AuthenticationFailedException,
			OXCommunicationException, UnsupportedContentOperationException, InternalUSMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Read user with id " + object.getID());
		checkTypeToWrite(object, _contentType.getID());
		Map<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		JSONResult result = _ajaxAccess.doGet(UtilConstants.USERS_PATH, UtilConstants.ACTION_GET, object.getSession(),
				parameters);
		checkResult(result, JSONResultType.JSONObject);
		updateDataObjectFromJSONResult(object, requestedFields, result);
	}
}
