/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.contenttypes.calendar.impl;

import java.util.*;

import org.apache.commons.logging.Log;
import org.json.*;

import com.openexchange.usm.api.contenttypes.*;
import com.openexchange.usm.api.datatypes.DataTypes;
import com.openexchange.usm.api.exceptions.*;
import com.openexchange.usm.api.session.*;
import com.openexchange.usm.contenttypes.util.*;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.ox_json.*;

/**
 * Transfer Handler for the module "Calendar"
 * @author ldo
 *
 */
public class AppointmentContentTypeTransferHandler extends AbstractTransferHandler {

	
	public AppointmentContentTypeTransferHandler(AppointmentContentType appointmentContentType, Log log,
			OXJSONAccess ajaxAccess) {
		super(appointmentContentType, log, ajaxAccess);
	}

	@Override
	public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.FOLDER, folder.getID());
		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());
		Session session = folder.getSession();
		parameters.put(UtilConstants.START, String.valueOf(session.getStartDate()));
		parameters.put(UtilConstants.END, String.valueOf(session.getEndDate()));

		DataObject[] result = performActionAndCheckRights(UtilConstants.ACTION_ALL, folder, requestedFields, parameters);

		resetRecurrenceTypeBasedFields(result);
		return result;
	}

	public DataObject[] getAllAppointments(Session session, BitSet requestedFields) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Getting new appointments for date range " + session.getStartDate() + " - "
					+ session.getEndDate());
		HashMap<String, String> parameters = new HashMap<String, String>();

		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.START, String.valueOf(session.getStartDate()));
		parameters.put(UtilConstants.END, String.valueOf(session.getEndDate()));
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());

		JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_ALL, session, parameters);
		JSONArray jArray = extractArrayResult(jsonResult);
		List<DataObject> resultList = new ArrayList<DataObject>();
		for (int i = 0; i < jArray.length(); i++) {
			JSONArray columnsArray;
			DataObject destination = null;
			try {
				columnsArray = jArray.getJSONArray(i);
				destination = _contentType.newDataObject(session);
				updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
				resultList.add(destination);
			} catch (JSONException e) {
				throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1, destination,
						"get new appointmets:data not present", e);
			}
		}
		DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
		resetRecurrenceTypeBasedFields(result);
		return result;
	}
	

	public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject)
			throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Getting all appointments as recurrences" );
		HashMap<String, String> parameters = new HashMap<String, String>();

		parameters.put(UtilConstants.COLUMNS, createColumnsParameter(requestedFields));
		parameters.put(UtilConstants.START, String.valueOf(session.getStartDate()));
		parameters.put(UtilConstants.END, String.valueOf(session.getEndDate()));
		parameters.put(UtilConstants.RECURRENCE_MASTER, Boolean.FALSE.toString());

		JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_ALL, session, parameters);
		JSONArray jArray = extractArrayResult(jsonResult);
		List<DataObject> resultList = new ArrayList<DataObject>();
		for (int i = 0; i < jArray.length(); i++) {
			JSONArray columnsArray;
			DataObject destination = null;
			try {
				columnsArray = jArray.getJSONArray(i);
				destination = _contentType.newDataObject(session);
				updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
				readOptionalTimestamp(jsonResult.getJSONObject(), destination);
				if (seriesObject!= null) {
					if (destination.getID().equals(seriesObject.getID())
							|| seriesObject.getID().equals(
									String.valueOf(destination.getFieldContent(CalendarConstants.RECURRENCE_ID))))
						resultList.add(destination);
				} else {
					resultList.add(destination);
				}
			} catch (JSONException e) {
				throw generateException(USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1, destination,
						"get new appointmets:data not present", e);
			}
		}
		DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
		resetRecurrenceTypeBasedFields(result);
		return result;
	}
	

	@Override
	public void writeDeletedDataObject(DataObject object) throws USMException {

		String recurrencePosition = String.valueOf(object
				.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION));
		if (recurrencePosition == null || recurrencePosition.equals("0") || recurrencePosition.equals("null"))
			writeExtendedDeleteDataObject(object, UtilConstants.FOLDER, object.getParentFolderID());
		else
			writeExtendedDeleteDataObject(object, CommonCalendarTasksFieldNames.RECURRENCE_POSITION,
					recurrencePosition, UtilConstants.FOLDER, object.getParentFolderID());
	}

	@Override
	public void writeNewDataObject(DataObject object) throws USMException {
		//don't call the standard write utility method because conflicts array can be returned from server
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " New " + DefaultContentTypes.CALENDAR_ID);
		checkTypeToWrite(object, DefaultContentTypes.CALENDAR_ID);
		HashMap<String, String> parameters = new HashMap<String, String>();
		JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, false);
		//		printTimeValues("New", requestBody);
		JSONResult result = _ajaxAccess.doPut(CalendarConstants.CALENDAR_PATH, UtilConstants.ACTION_NEW, object
				.getSession(), parameters, requestBody);
		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		readOptionalTimestamp(o, object);
		try {
			JSONObject resultObject = o.getJSONObject(UtilConstants.RESULT_DATA);
			if (resultObject.has(UtilConstants.RESULT_ID)) {
				object.setID(resultObject.getString(UtilConstants.RESULT_ID));
			} else if (resultObject.has(CalendarConstants.CONFLICTS)) {
				JSONArray arr = resultObject.getJSONArray(CalendarConstants.CONFLICTS);
				throw new ConflictingChangeException(USMContentTypesCalendarErrorCodes.CONFLICTS,
						"Appointment could not be created because of conflicts", arr.getJSONObject(0));
			} else {
				throw new OXCommunicationException(USMContentTypesCalendarErrorCodes.OTHER_ERROR,
						"Appointment could not be created because of ox communication error", resultObject.toString());
			}
		} catch (JSONException e) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER1,
					"Invalid Result on new " + DefaultContentTypes.CALENDAR_ID, e);
		}
		writeDeletedExceptions(object);
	}

	

	private void writeDeletedExceptions(DataObject seriesObject) throws USMException {
		Object[] deleteExceptions = (Object[]) seriesObject.getFieldContent(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
		if (deleteExceptions == null || deleteExceptions.length == 0)
				//|| !seriesObject.isFieldModified(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS))
			return;
		DataObject[] allOcurrences = getAllRecurrencesAsAppointments(seriesObject.getSession(), seriesObject.getSession()
				.getFieldFilter(seriesObject.getContentType()), seriesObject);
		for (int i = 0; i < deleteExceptions.length; i++) {
			for (int j = 0; j < allOcurrences.length; j++) {
				if (allOcurrences[j] != null) {
					Object occurenceStartTime = allOcurrences[j].getFieldContent(CommonCalendarTasksFieldNames.START_DATE);
					if (occurenceStartTime != null && occurenceStartTime.equals(deleteExceptions[i])) {
						deleteSingleOccurenceInSeries(seriesObject, allOcurrences[j]);
					}
				}
			}
		}
	}

	private void deleteSingleOccurenceInSeries(DataObject seriesObject, DataObject ocurrence)
			throws USMException {
		Object oldRecPos = ocurrence
				.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
		readDataObject(ocurrence, seriesObject.getSession().getFieldFilter(seriesObject.getContentType()));
		ocurrence.setFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION, oldRecPos);
		writeDeletedDataObject(ocurrence);
	}

	private JSONObject createRequestBodyWithIgnoreConflicts(DataObject object, boolean isUpdate)
			throws InternalUSMException {
		JSONObject requestBody = new JSONObject();
		ContentTypeField[] fields = object.getContentType().getFields();
		boolean isException = isSeriesException(object);
		boolean isNewException = isException && isNewException(object);
		for (int i = 0; i < fields.length; i++) {
			boolean isNoSpecialField = isNoSpecialSeriesField(fields[i], !isNewException);
			if (isException) {
				if (isNoSpecialField)
					addFieldToRequestBody(object, isUpdate, !isUpdate, requestBody, fields[i], i);
			} else {
				addFieldToRequestBody(object, isUpdate && isNoSpecialField, !isUpdate, requestBody, fields[i], i);
			}
		}
		requestBody.remove(CalendarConstants.RECURRENCE_ID);
		// XXX Workaround for Bug #14952 (start)
		if (requestBody.optBoolean(CalendarConstants.FULL_TIME) && !requestBody.has(CommonCalendarTasksFieldNames.START_DATE)) {
			for (int i = 0; i < fields.length; i++) {
				if (fields[i].getFieldName().equals(CommonCalendarTasksFieldNames.START_DATE)) {
					addFieldToRequestBody(object, false, true, requestBody, fields[i], i);
					break;
				}
			}
		}
		// XXX Workaround for Bug #14952 (end)
		if (!isException) {
			try {
				requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
				requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
				requestBody.remove(CommonCalendarTasksFieldNames.CHANGE_EXCEPTIONS);
				requestBody.remove(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
				requestBody.put(CalendarConstants.IGNORE_CONFLICTS, Boolean.TRUE);
				Number recType = (Number) object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
				if (recType != null && recType.intValue() > 0) {
					if (object.isFieldModified(CommonCalendarTasksFieldNames.UNTIL)) {
						//this is a special case for changing the until of the series. (iPhone - for all future appointments) 
						//All relevant recurrence fields have to be sent 
						requestBody.put(CommonCalendarTasksFieldNames.RECURRENCE_TYPE, recType);
						requestBody.put(CommonCalendarTasksFieldNames.INTERVAL, object
								.getFieldContent(CommonCalendarTasksFieldNames.INTERVAL));
						requestBody.put(CommonCalendarTasksFieldNames.UNTIL, object
								.getFieldContent(CommonCalendarTasksFieldNames.UNTIL));
						Object days = object.getFieldContent(CommonCalendarTasksFieldNames.DAYS);
						if (days instanceof Number && ((Number) days).intValue() != 127)
							requestBody.put(CommonCalendarTasksFieldNames.DAYS, days);
						else
							requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
					}
				} else {
					requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
					requestBody.remove(CommonCalendarTasksFieldNames.INTERVAL);
					requestBody.remove(CommonCalendarTasksFieldNames.UNTIL);
					requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
				}
			} catch (JSONException e) {
				throwInternalUSMException(USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER1, "writing", object,
						CalendarConstants.IGNORE_CONFLICTS, DataTypes.BOOLEAN, e);
			}
		}
		return requestBody;
	}

	private boolean isNoSpecialSeriesField(ContentTypeField field, boolean isUpdate) {
		int id = field.getFieldID();
		return id < (isUpdate ? 206 : 209) || (id > 216 && id != 222);
	}

	private boolean isSeriesException(DataObject object) {
		Object o = object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
		return (o instanceof Number) && ((Number) o).intValue() != 0;
	}
	
	
	

	private boolean isNewException(DataObject object) {
		Object o = object.getFieldContent(CalendarConstants.RECURRENCE_ID);
		return String.valueOf(o).equals(object.getID());
	}

	@Override
	public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(object.getSession() + " Updating " + _contentType.getID());
		checkTypeToWrite(object, _contentType.getID());
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.ID, object.getID());
		parameters.put(UtilConstants.FOLDER, object.getOriginalParentFolderID());
		parameters.put(UtilConstants.TIMESTAMP, String.valueOf(timestamp));
		JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, true);

		//		printTimeValues("Change", requestBody);
		JSONResult result = _ajaxAccess.doPut(getOXAjaxAccessPath(), UtilConstants.ACTION_UPDATE, object.getSession(),
				parameters, requestBody);

		checkResult(result, JSONResultType.JSONObject);
		// XXX Shouldn't we use the timestamp returned from the OX server ?
		object.setTimestamp(timestamp);
		writeDeletedExceptions(object);
	}

	@Override
	public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp)
			throws USMException {
		Session session = folder.getSession();
		// XXX Why do we use ACTION_ALL here ? Shouldn't we do only an update ?
		DataObject[] result = readStandardUpdatedFolderContent(folder, timestamp, requestedFields,
				UtilConstants.ACTION_ALL, UtilConstants.FOLDER, null, null, String.valueOf(session.getStartDate()),
				String.valueOf(session.getEndDate()));
		resetRecurrenceTypeBasedFields(result);
		return result;
	}

	@Override
	protected String getOXAjaxAccessPath() {
		return CalendarConstants.CALENDAR_PATH;
	}
	
	public String resolveUID(Session session, String uid) throws USMException {
		if (_journal.isDebugEnabled())
			_journal.debug(session + " Resolving uid " + uid);
		HashMap<String, String> parameters = new HashMap<String, String>();
		parameters.put(UtilConstants.UID, uid);

		JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), UtilConstants.ACTION_RESOLVEUID, session,
				parameters);

		checkResult(result, JSONResultType.JSONObject);
		JSONObject o = result.getJSONObject();
		try {
			JSONObject resultObject = o.getJSONObject(UtilConstants.RESULT_DATA);
			if (resultObject.has(UtilConstants.ID)) {
				String id = resultObject.getString(UtilConstants.ID);
				return id;
			}
		} catch (JSONException e1) {
			throw new OXCommunicationException(USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER7, "Invalid id",
					e1);
		}
		return null;
	}
	
}
