/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.subscribe;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.groupware.container.FolderObject;
import com.openexchange.groupware.contexts.Context;


/**
 * {@link SubscriptionSourceCollector}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 *
 */
public class SubscriptionSourceCollector implements SubscriptionSourceDiscoveryService {

    private Map<String, SubscribeService> services = new HashMap<String, SubscribeService>();
    private Map<String, SortedSet<SubscribeService>> shelvedServices = new HashMap<String, SortedSet<SubscribeService>>();
    
    public SubscriptionSource getSource(String identifier) {
        if(!services.containsKey(identifier)) {
            return null;
        }
        return services.get(identifier).getSubscriptionSource();
    }

    public List<SubscriptionSource> getSources(int folderModule) {
        List<SubscriptionSource> sources = new LinkedList<SubscriptionSource>();
        for(SubscribeService subscriber : services.values()) {
            if(folderModule == -1 || subscriber.handles(folderModule)) {
                sources.add(subscriber.getSubscriptionSource());
            }
        }
        return sources;
    }
    
    public List<SubscriptionSource> getSources() {
        return getSources(-1);
    }

    public boolean knowsSource(String identifier) {
        return services.containsKey(identifier);
    }

    public void addSubscribeService(SubscribeService service) {
        SubscribeService oldService = services.get(service.getSubscriptionSource().getId());
        if(oldService != null) {
            if(oldService.getSubscriptionSource().getPriority() < service.getSubscriptionSource().getPriority()) {
                shelfService(oldService);
                services.put(service.getSubscriptionSource().getId(), service);
            } else {
                shelfService(service);
            }
        } else {
            services.put(service.getSubscriptionSource().getId(), service);
        }
        
    }

    // FIXME: This is not unique anymore
    public void removeSubscribeService(String identifier) {
        services.remove(identifier);        
        resurrectFromShelf(identifier);
    }
    
    public SubscriptionSource getSource(Context context, int subscriptionId) throws AbstractOXException {
        for(SubscribeService source : services.values()) {
            if(source.knows(context, subscriptionId)) {
                return source.getSubscriptionSource();
            }
        }
        return null;
    }
    

    private void shelfService(SubscribeService service) {
        String identifier = service.getSubscriptionSource().getId();
        SortedSet<SubscribeService> set = shelvedServices.get(identifier);
        if(set == null) {
            set = new TreeSet<SubscribeService>(new Comparator<SubscribeService>(){

                public int compare(SubscribeService o1, SubscribeService o2) {
                    return o1.getSubscriptionSource().getPriority() - o2.getSubscriptionSource().getPriority();
                }
                
            });
            shelvedServices.put(identifier, set);
        }
        set.add(service);
    }

    private void resurrectFromShelf(String identifier) {
        SortedSet<SubscribeService> set = shelvedServices.get(identifier);
        if(set != null) {
            services.put(identifier, set.first());
            set.remove(set.first());
        }
    }


}
