/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.subscribe;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import com.openexchange.groupware.AbstractOXException;
import com.openexchange.groupware.contexts.Context;


/**
 * {@link CompositeSubscriptionSourceDiscoveryService}
 *
 * @author <a href="mailto:francisco.laguna@open-xchange.com">Francisco Laguna</a>
 *
 */
public class CompositeSubscriptionSourceDiscoveryService implements SubscriptionSourceDiscoveryService {

    private List<SubscriptionSourceDiscoveryService> services = new ArrayList<SubscriptionSourceDiscoveryService>();
    
    /* (non-Javadoc)
     * @see com.openexchange.subscribe.SubscriptionSourceDiscoveryService#getSource(java.lang.String)
     */
    public SubscriptionSource getSource(String identifier) {
        SubscriptionSource current = null;
        for(SubscriptionSourceDiscoveryService subDiscoverer : services) {
            if(subDiscoverer.knowsSource(identifier)) {
                SubscriptionSource source = subDiscoverer.getSource(identifier);
                if(current == null || current.getPriority() < source.getPriority()) {
                    current = source;
                }
            }
        }
        return current;
    }

    public List<SubscriptionSource> getSources(int folderModule) {
        Map<String, SubscriptionSource> allSources = new HashMap<String, SubscriptionSource>();
        for(SubscriptionSourceDiscoveryService subDiscoverer : services) {
            List<SubscriptionSource> sources = subDiscoverer.getSources(folderModule);
            for (SubscriptionSource subscriptionSource : sources) {
                SubscriptionSource previousSource = allSources.get(subscriptionSource.getId());
                if(previousSource == null || previousSource.getPriority() < subscriptionSource.getPriority()) {
                    allSources.put(subscriptionSource.getId(), subscriptionSource);
                }
            }
        }
        List<SubscriptionSource> sources = new ArrayList<SubscriptionSource>(allSources.values());
        Collections.sort(sources, new Comparator<SubscriptionSource>() {

            public int compare(SubscriptionSource o1, SubscriptionSource o2) {
                if(o1.getDisplayName() != null && o2.getDisplayName() != null) {
                    return o1.getDisplayName().compareTo(o2.getDisplayName());
                } else {
                    return o1.getDisplayName() == null ? -1 : 1;
                }
            }
            
        });
        return sources;
    }
    
    public List<SubscriptionSource> getSources() {
        return getSources(-1);
    }

    public boolean knowsSource(String identifier) {
        for(SubscriptionSourceDiscoveryService subDiscoverer : services) {
            if(subDiscoverer.knowsSource(identifier)) {
                return true;
            }
        }
        return false;
    }
    
    public void addSubscriptionSourceDiscoveryService(SubscriptionSourceDiscoveryService service) {
        services.add(service);
    }
    
    public void removeSubscriptionSourceDiscoveryService(SubscriptionSourceDiscoveryService service) {
        services.remove(service);
    }

    public SubscriptionSource getSource(Context context, int subscriptionId) throws AbstractOXException {
        for(SubscriptionSourceDiscoveryService subDiscoverer : services) {
            SubscriptionSource source = subDiscoverer.getSource(context, subscriptionId);
            if(source != null) {
                return source;
            }
        }
        return null;
    }

    public void clear() {
        services.clear();
    }

}
