/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.groupware.update.tasks;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.database.DBPoolingException;
import com.openexchange.databaseold.Database;
import com.openexchange.groupware.tasks.Mapping;
import com.openexchange.groupware.tasks.SQL;
import com.openexchange.groupware.tasks.StorageType;
import com.openexchange.groupware.tasks.Task;
import com.openexchange.groupware.tasks.TaskException;
import com.openexchange.groupware.update.Schema;
import com.openexchange.groupware.update.UpdateTask;
import com.openexchange.tools.sql.DBUtils;

/**
 * This class implements a database update task the doubles the VARCHAR for the
 * title of tasks.
 * @author <a href="mailto:marcus@open-xchange.org">Marcus Klein</a>
 */
public final class EnlargeTaskTitle implements UpdateTask {

    /**
     * Logger.
     */
    private static final Log LOG = LogFactory.getLog(EnlargeTaskTitle.class);

    /**
     * Default constructor.
     */
    public EnlargeTaskTitle() {
        super();
    }

    /**
     * {@inheritDoc}
     */
    public int addedWithVersion() {
        return 18;
    }

    /**
     * {@inheritDoc}
     */
    public int getPriority() {
        return UpdateTaskPriority.NORMAL.priority;
    }

    /**
     * {@inheritDoc}
     * @throws TaskException 
     */
    public void perform(final Schema schema, final int contextId) throws TaskException {
        LOG.info("Performing update task EnlargeTaskTitle.");
        Connection con = null;
        try {
            con = Database.get(contextId, true);
        } catch (final DBPoolingException e) {
            throw new TaskException(TaskException.Code.NO_CONNECTION, e);
        }
        try {
            for (final StorageType type : StorageType.TYPES_AD) {
                alterTitle(con, SQL.TASK_TABLES.get(type));
            }
        } catch (final SQLException e) {
            throw new TaskException(TaskException.Code.SQL_ERROR, e);
        } finally {
            Database.back(contextId, true, con);
        }
    }

    private void alterTitle(final Connection con, final String table)
        throws SQLException {
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            stmt.execute("ALTER TABLE " + table + " MODIFY " + title
                + " VARCHAR(256)");
            LOG.info("Altered table " + table + " changed " + title
                + " to VARCHAR(256).");
        } finally {
            DBUtils.closeSQLStuff(null, stmt);
        }
    }

    private final String title = Mapping.getMapping(Task.TITLE)
        .getDBColumnName();
}
