/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.messaging.generic.internet;

import java.io.IOException;
import java.io.InputStream;
import javax.mail.Part;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import com.openexchange.messaging.BinaryContent;
import com.openexchange.messaging.MessagingException;
import com.openexchange.messaging.MessagingExceptionCodes;

/**
 * {@link MimeBinaryContent} - A MIME {@link BinaryContent binary content}.
 * 
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @since Open-Xchange v6.16
 */
public class MimeBinaryContent implements BinaryContent {

    /**
     * The MIME part.
     */
    protected final Part part;

    /**
     * Initializes a new {@link MimeBinaryContent}.
     * 
     * @param part The MIME part
     */
    public MimeBinaryContent(final Part part) {
        super();
        this.part = part;
    }

    public InputStream getData() throws MessagingException {
        return getBodyPartInputStream();
    }

    /**
     * Gets MIME part's input stream in a safe manner: If {@link Part#getInputStream()} fails, the {@link MimeBodyPart#getRawInputStream()}
     * is invoked.
     * 
     * @return The either decoded or raw input stream
     * @throws MessagingException If neither input stream nor raw input stream can be returned
     */
    private InputStream getBodyPartInputStream() throws MessagingException {
        try {
            return part.getInputStream();
        } catch (final IOException e) {
            try {
                if (part instanceof MimeBodyPart) {
                    return ((MimeBodyPart) part).getRawInputStream();
                }
                throw e;
            } catch (final javax.mail.MessagingException me) {
                me.setNextException(e);
                throw MessagingExceptionCodes.MESSAGING_ERROR.create(e, e.getMessage());
            } catch (final IOException ioe) {
                throw MessagingExceptionCodes.IO_ERROR.create(ioe, ioe.getMessage());
            }
        } catch (final javax.mail.MessagingException e) {
            try {
                if (part instanceof MimeMessage) {
                    return ((MimeMessage) part).getRawInputStream();
                }
                throw e;
            } catch (final javax.mail.MessagingException me) {
                me.setNextException(e);
                throw MessagingExceptionCodes.MESSAGING_ERROR.create(e, e.getMessage());
            }
        }
    }

}
