/**
 * 
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 * 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * 
 * Copyright (C) 2004-2010 Open-Xchange, Inc.
 * Mail: info@open-xchange.com 
 * 
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 * 
 */

/*jslint white: true, browser: true, devel: true, evil: true, forin: true, undef: true, eqeqeq: true, immed: true */

/*global ox, jQuery, window, console, newWindow, urlify, activefolder : true, activemodule : true,
corewindow, json, AjaxRoot, session, OXMailMapping, _, triggerEvent, 
register, unregister, currentCalendarView, configGetKey, oMainFolderTree, config,
mail_printSingleMail, activeYear, activeMonth, activeDay, calendar_print,
getDayInSameWeek, calendar_getAllFoldersAttribute, getWWStartTime, getWWEndTime, 
showNode, changeView, setActiveFolder, mail_accounts, escapeRegExp, newAlert,
configuration_askforSave, isEmpty, configSetKey, configContainsKey, logout_location,
formatError, newConfirm, AlertPopup */

if (window.ox === undefined) {
    // required to run in "lightweight" windows
    /**
     * @name ox
     * @namespace
     */
    window.ox = {};
}

/**
 * @name ox.browser
 * @namespace
 */
ox.browser = (function () {
    // adopted from prototype.js
    var ua = navigator.userAgent;
    var isOpera = Object.prototype.toString.call(window.opera) === "[object Opera]";
    return {
        /** @lends ox.browser */
        /** is IE? */
        IE:     !!window.attachEvent && !isOpera,
        /** is Opera? */
        Opera:  isOpera,
        /** is WebKit? */
        WebKit: ua.indexOf('AppleWebKit/') > -1,
        /** is Gecko/Firefox? */
        Gecko:  ua.indexOf('Gecko') > -1 && ua.indexOf('KHTML') === -1
    };
}());

/**
 * @name ox.test
 * @namespace
 */
ox.test = (function () {

    var $ = jQuery;
    
    return {
        /** @lends ox.test */
        
        /**
         * Execute JavaScript code via eval() in current window (helps SELENIUM tests).
         * Otherwise, internal tests, e.g. using "object.constructor", will fail due to permission problems.
         * @param {string} code JavaScript code
         * @returns {Object} Result of eval()
         * @example
         * // API call inside SELENIUM unit test
         * ox.test.evalScript("ox.api.calendar.compose({ data: { title: 'Test' } });");
         */
        evalScript: function (code) {
            return eval(code);
        },
        
        /**
         * Trigger an event in a nested window. First parameter is the window's GUID, second the event name.
         * Any further parameter is passed to the corresponding event handlers.
         * @param {string} guid Window GUID. "OX.#" refers to last window.
         * @param {string} name Event name
         * @example
         * ox.test.trigger("OX.1", "OX_Cancel_Object");
         */
        trigger: function (guid, name) {
            // exists?
            if (ox.api.window.exists(guid)) {
                // get arguments
                var args = $.makeArray(arguments);
                // remove guid
                args.shift();
                // get window & trigger event
                var w = ox.api.window.get(guid).window;
                w.triggerEvent.apply(w, args);
            }
        },
        
        /**
         * Trigger save event in a window (convenience function)
         * @param {string} guid Window GUID. "OX.#" refers to last window.
         * @example
         * // save object
         * ox.test.triggerSave("OX.1");
         */
        triggerSave: function (guid) {
            this.trigger(guid, ox.api.event.common.SaveObject);
        },
        
        /**
         * Trigger cancel event in a window (convenience function)
         * @param {string} guid Window GUID. "OX.#" refers to last window.
         * @example
         * // cancel object (might trigger dialog "Do you really want to cancel...?")
         * ox.test.triggerCancel("OX.1");
         */
        triggerCancel: function (guid) {
            this.trigger(guid, ox.api.event.common.CancelObject);
        }
    };
}());

/**
 * @name ox.util
 * @namespace
 */
ox.util = {
    
    /**
      * Get a new empty array (constructed by current window; useful in IE for cross-window issues)
      * @returns {Array} New empty array
      * @example
      * var tmp = ox.util.getArray(); // simply returns []
      */
    getArray: function () {
        return [];
    },
    
    /**
     * Serialize object (key/value pairs) to fit into URLs (e.g. a=1&b=2&c=HelloWorld)
     * @param {Object} obj Object to serialize
     * @param {string} [delimiter] Delimiter
     * @returns {string} Serialized object
     * @example
     * ox.util.serialize({ a: 1, b: 2, c: "text" });
     */
    serialize: function (obj, delimiter) {
        var tmp = [], e = encodeURIComponent;
        if (typeof obj === "object") {
            for (var id in (obj || {})) {
                if (obj[id] !== undefined) {
                    tmp.push(e(id) + "=" + e(obj[id]));
                }
            }
        }
        return tmp.join(delimiter === undefined ? "&" : delimiter);
    },
    
    /**
     * Deserialize object (key/value pairs)
     * @param {string} str String to deserialize
     * @param {string} [delimiter] Delimiter
     * @returns {Object} Deserialized object
     * @function
     * @name ox.util.deserialize
     * @example
     * ox.util.deserialize("a=1&b=2&c=text");
     */
    deserialize: function (str, delimiter) {
        var pairs = (str || "").split(delimiter === undefined ? "&" : delimiter);
        var i = 0, $l = pairs.length, pair, obj = {}, d = decodeURIComponent;
        for (; i < $l; i++) {
            pair = pairs[i];
            var keyValue = pair.split(/\=/), key = keyValue[0], value = keyValue[1];
            if (key !== "" || value !== undefined) {
                obj[d(key)] = d(value);
            }
        }
        return obj;
    },
    
    /**
     * This function simply writes its parameters to console.
     * Useful to debug callbacks, e.g. event handlers.
     * @example
     * ox.api.calendar.httpGet({ params: { id: 158302 }, success: ox.util.inspect });
     */
    inspect: function () {
        var args = jQuery.makeArray(arguments); // $ is not jQuery here
        args.unshift("Inspect");
        console.log.apply(console, args);
    },
    
    /**
     * Create absolute URL
     * @param {string} url (relative/absolute) URL
     * @returns {string} Absolute URL
     * @example
     * ox.util.getAbsoluteURL("index.html"); // returns protocol://host/path/index.html
     * ox.util.getAbsoluteURL("/index.html"); // returns protocol://host/index.html
     * ox.util.getAbsoluteURL("http://localhost/foo/index.html"); // returns http://localhost/foo/index.html
     */
    getAbsoluteURL: function (url) {
        // change?
        if (url.search(/^http/) !== -1) {
            return url;
        } else {
            var l = window.location;
            if (url.substr(0, 1) === "/") {
                // add protocol & host
                return l.protocol + "//" + l.host + url;
            } else {
                // strip file & inject version
                return l.href.replace(/\/[^\/]+$/, "/") + ox.api.window.core.oxProductInfo.build + "/" + url;
            }
        }
    },
    
    /**
     * Check whether an object is an array
     * @params {Object} obj The object to test
     * @returns {boolean} True/False
     */
    isArray: function (obj) {
        // we cannot use 'elem.constructor === array' under SELENIUM (permission denied error).
        // so, we simply rely on the existence of splice, unshift, and push.
        // We even cannot test if they are functions (does not work in IE).
        // Same reason: Object.prototype.toString.call(elem) == "[Object Array]" does not work.
        return obj.splice !== undefined && obj.unshift !== undefined && obj.push !== undefined;
    },
    
    /**
     * Simple test to check whether an object is an instance of Window or a DOM node
     * @params {Object} obj The object to test
     * @returns {boolean} True/False
     */
    isDOMObject: function (obj) {
        // not null, not undefined, but DOM node or window 
        return obj && (obj.nodeType || obj.setInterval);
    }
};

/** @namespace */
ox.api = (function () {
    
    // ref
    var $ = jQuery;

    // api
    var that;
    
    // debug
    var debugEnabled = false;
    var debug = !debugEnabled ? $.noop : function () {
        console.log.apply(this, arguments);
    };
    
    // clone
    var clone = function clone(elem) {
        if (typeof elem !== "object") {
            return elem;
        } else {
            var isArray = ox.util.isArray;
            var isDOMObject = ox.util.isDOMObject;
            var subclone = function (elem) {
                if (!elem) {
                    return null;
                } else {
                    var tmp = isArray(elem) ? [] : {}, prop;
                    for (var i in elem) {
                        prop = elem[i];
                        tmp[i] = typeof prop === "object" && !isDOMObject(prop) ? subclone(prop) : prop;
                    }
                    return tmp;
                }
            };
            return subclone(elem);
        }
    };
    
    // get hash & query
    var queryData = ox.util.deserialize(document.location.search.substr(1), /&/);
    var hashData = ox.util.deserialize(document.location.hash.substr(1), /&/);
    
    // default window options
    var defaultWindowOptions = {
        resizable: "yes",
        menubar: "no",
        toolbar: "no",
        status: "no",
        location: "yes",
        scrollbars: "no"
    };
    
    /**
     * Default window factory (can be overridden).
     * A window factory is a function that produces a new window (e.g. via window.open) or
     * an iframe. The function receives an options objects. After creating the new window,
     * the function calls registerWindow(guid, window) and calls the callback function options.success
     * with the same two parameters: 1) the new window object, and 2) the GUID of that window.
     * @public
     * @param {Object} opt Factory options
     * @param {string} opt.url Window URL
     * @param {string} opt.guid Window GUID
     * @param {Object} opt.options Window options
     */
    var windowFactory = function (opt) {
        if (opt.guid !== undefined) {
            // open
            var win = newWindow(ox.util.getAbsoluteURL(opt.url), that.window.serializeOptions(opt.options), opt.id);
            /// register window
            that.window.register(opt.guid, win);
            // set guid
            win.guid = opt.guid;
            // focus
            win.focus();
            // continuation
            opt.success(win, opt.guid);
        }
        else {
            console.error("windowFactory: guid is missing!");
        }
    };
    
    /**
     * Default window closer (can be overridden)
     * @public
     */
    var windowCloser = function (obj) {
        obj.window.close();
    };
    
    /**
     * Default window positioner
     * @name ox.api-windowPositioner
     * @public
     * @returns {Object} Bounds
     * @example
     * function () {
     *   return { x: 0, y: 0, width: 400, height: 300 };
     * }
     */
    var windowPositioner = function () {
        // width & height
        var width = screen.availWidth <= 1024 ? 800 : 1000;
        var height = screen.availHeight <= 800 ? 700 : Math.max(700, $(document).height());
        // return bounds
        return {
            x: Math.round((screen.width - width) / 2),
            y: Math.round((screen.height - height) / 2) - 25,
            width: width,
            height: height
        };
    };
    
    /**
     * Internal window preprocessor
     * @name ox.api-constructWindow
     * @function
     * @public
     * @param {string} urlPrefix URL path
     * @param {Object} o Options
     * @param {Object} o.data Data passed to the new window
     * @param {Object} o.params Params added to the url as hash (#anchor)
     * @param {function ()} o.success Called when the new window is created
     * @param {function ()} o.closer Custom closer function (called when closing the window)
     * @param {Object} o.bounds Window bounds {@link ox.api-windowPositioner}
     * @param {Object} o.options Window options {@link ox.api.setDefaultWindowOptions}
     * @returns {string} New window GUID
     */
    var constructWindow = function (urlPrefix, o) {
        // set data
        var guid = o.params.guid = that.window.setData(o.data);
        // build url
        var url = urlPrefix + "#" + ox.util.serialize(o.params);
        // open new window
        var opt = {
            guid: guid,
            url: url,
            options: $.extend(o.bounds, o.options),
            success: o.success, 
            closer: o.closer,
            id: o.id
        };
        debug("constructWindow", opt);
        windowFactory(opt);
        return guid;
    };
    
    // process options
    var processOptions = function (d, o) {
        // general defaults
        var defaults = {
            params: {},
            options: {},
            data: {},
            bounds: windowPositioner(),
            success: $.noop,
            closer: $.noop
        };
        // merge general defaults, local default, and specific options
        return $.extend(true, defaults, d || {}, o || {});
    };
    
    // process select options
    var processSelectOptions = function (module, o) {
        var defaultFolder = module === "mail" ? configGetKey("mail.folder.inbox") : configGetKey("folder")[module];
        var opt = $.extend({
            module: module,
            folder: defaultFolder,
            id: undefined,
            success: $.noop
        }, o || {});
        // copy
        opt.folder_id = opt.folder;
        return opt;
    };
    
    var processSelect = function (module, options) {
        // options
        var o = processSelectOptions(module, options);
        // change view (always; all users have mail)
        changeView("mail");
        // trigger event
        triggerEvent("OX_Direct_Linking", module, o);
        // call back
        o.success(o);
    };
    
    var processHTTPOptions = function (module, defaultParams, options, defaultOptions) {
        // options
        var o = $.extend(defaultOptions || {}, options || { params: {} });
        // params
        o.params = $.extend(defaultParams || {}, o.params || {});
        // set module
        o.module = module;
        // done
        return o;
    };
    
    // helper for setModal
    var modal = false;
    
    that = {
        /** @lends ox.api */

        /**
         * @name ox.api.event
         * @namespace
         */
        event: {
        
            /**
             * Register for global event
             * @param {string} name Event name
             * @param {function ()} fn Event handler
             * @example
             * ox.api.event.register(ox.api.event.common.ViewChanged, ox.util.inspect);
             */
            register: function (name, fn) {
                register(name, fn);
            },

            /**
             * Unregister for global event
             * @param {string} name Event name
             * @param {function ()} fn Event handler
             * @example
             * ox.api.event.unregister(ox.api.event.common.ViewChanged, ox.util.inspect);
             */
            unregister: function (name, fn) {
                unregister(name, fn);
            },
            
            /**
             * Trigger an event. First parameter is the event name.
             * Any further parameter is passed to the corresponding event handlers.
             * @param {string} name Event name
             */
            trigger: function (name) {
                triggerEvent.apply(window, arguments);
            },
            
            /**
             * A list of common events used throughout the UI
             * @name ox.api.event.common
             * @namespace
             */
            common: {
                /** @type string */
                ViewChanged: "OX_View_Changed",
                /** @type string */
                Refresh: "OX_Refresh",
                /** @type string */
                ConfigurationLoadedComplete: "OX_Configuration_Loaded_Complete",
                /** @type string */
                ConfigurationChanged: "OX_Configuration_Changed",
                /** @type string */
                LanguageChanged: "LanguageChanged",
                /** @type string */
                Logout: "Logout",
                /** @type string */
                SaveObject: "OX_SAVE_OBJECT",
                /** @type string */
                CancelObject: "OX_Cancel_Object",
                /** @type string */
                NewUnreadMail: "OX_New_Unread_Mail",
                /** @type string */
                Ready: "Ready"
            }
        },

        /**
         * @name ox.api.config
         * @namespace
         */
        config: (function () {
            
            return {
                /** @lends ox.api.config */
                
                /**
                 * Get configuration value
                 * @params {string} path Configuration path
                 * @params {Object} [defaultValue] Default value to be returned if the path does not exist 
                 * @returns {Object} Value
                 * @example
                 * // get team view zoom level
                 * ox.api.config.get("gui.calendar.teamview.zoom");
                 * // use default value if the configuration does not contain the given path
                 * ox.api.config.get("gui.calendar.teamview.zoom", 100);
                 */
                get: function (path, defaultValue) {
                    if (!path) { // undefined, null, ""
                        return config;
                    } else {
                        if (defaultValue === undefined) {
                            return configGetKey(path);
                        } else {
                            return this.contains(path) ? configGetKey(path) : defaultValue;
                        }
                    }
                },
                
                /**
                 * Set configuration value
                 * @params {string} path Configuration path
                 * @params {Object} value Value
                 * @example
                 * // set "save configuration after logout" to "true"
                 * ox.api.config.set("gui.global.save", 1);
                 */
                set: function (path, value) {
                    if (path) {
                        configSetKey(path, value);
                    }
                },
             
                /**
                 * Check if configuration contains a specific path
                 * @params {string} path Configuration path
                 * @retuns {boolean} True/False
                 * @example
                 * ox.api.config.contains("gui.global.save");
                 */
                contains: function (path) {
                    return configContainsKey(path);
                },
                
                /**
                 * Save configuration
                 * @params {Object} [options] Options
                 * @params {Object} [options.force] Force save
                 * @params {Object} [options.silent] If true no message will be shown
                 * @params {function (data)} [options.success] Success handler
                 * @params {function (error)} [options.error] Error handler
                 * @example
                 * // save configuration
                 * ox.api.config.save({
                 *   success: function () {
                 *      console.log("Configuration saved!");
                 *   }
                 * });
                 */
                save: function (options) {
                    // defaults
                    var opt = $.extend({
                        force: true,
                        silent: true,
                        success: $.noop,
                        error: $.noop
                    }, options || {});
                    // this is done via an event
                    ox.api.event.trigger(
                        "OX_Save_Configuration", opt.force, opt.silent, 
                        opt.success, opt.error
                    );
                }
            };
        }()),

        /**
         * @name ox.api.ui
         * @namespace
         */
        ui: (function () {
            
            // helper for setModule("mail")
            var setMail = function (cont) {
                // callback (step #4)
                var foundFolder = function (id) {
                    setActiveFolder(id, null, true);
                    triggerEvent("OX_Switch_Module", "mail", "newfolder");
                    cont();
                };
                // continuation (step #3)
                var processFolders = function () {
                    // cache is not cold
                    var cache = oMainFolderTree.cache.fast_access;
                    // get separator
                    var separator = configGetKey("modules.mail.defaultseparator") || "/";
                    // loop over available mail accounts
                    var i, acc;
                    for (i in mail_accounts.data.data) {
                        // get account
                        acc = mail_accounts.data.data[i];
                        // check
                        if (acc && acc.id < Infinity && (cache["default" + acc.id] || cache["default" + acc.id + separator + "INBOX"])) {
                            // found
                            foundFolder("default" + acc.id + separator + "INBOX");
                            return;
                        }
                    }
                    // if we come around here, we haven't found anything yet, so...
                    var rex = new RegExp("^default\\d+(" + escapeRegExp(separator) + "INBOX)?$"), match;
                    for (i in cache) {
                        if ((match = rex.exec(i))) {
                            foundFolder(match[1] ? i : i + separator + "INBOX");
                            return;
                        }
                    }
                    // still not found...
                    newAlert(
                        _("The E-Mail module is not available"), 
                        _("Unable to establish a connection to the E-Mail server. Possible reasons: the mail server is (temporarily) down or there are network connection problems. To prevent further errors the module has been disabled. Please contact your administrator."), /**i18n**/
                        null
                    );
                };
                // get folder (step #2)
                var getFolder = function () {
                    if (configGetKey("modules.folder.tree") === 1) {
                        processFolders();
                    } else {
                        // cache magic
                        oMainFolderTree.cache.do_load_subfolders("1", oMainFolderTree.cache.find_folder("1"), {}, processFolders);
                    }
                };
                
                // cache lookup (step #1)
                var lookup = function () {
                    // in cache?
                    if (isEmpty(oMainFolderTree.cache.fast_access)) {
                        // no, try again alter
                        setTimeout(lookup, 250);
                    } else {
                        getFolder();
                    }
                };
                // start with folder look up
                lookup();
            };
            
            // helper for public setModule
            var setModule = function (name, cont) {
                // continuation
                cont = cont || $.noop;
                // set global variable
                activemodule = name;
                // which module?
                switch (name) {
                case "mail":
                    setMail(cont);
                    break;
                case "configuration": 
                    activefolder = configGetKey("folder.configuration");
                    triggerEvent("OX_Switch_Module", name, "newfolder");
                    cont();
                    break;
                case "portal":
                    activefolder = undefined;
                    triggerEvent("OX_Switch_Module", name, "newfolder");
                    cont();
                    break;
                default:
                    setActiveFolder(configGetKey("folder")[name], null, true);
                    triggerEvent("OX_Switch_Module", name, "newfolder");
                    cont();
                    break;
                }
            };
            
            return {
                /** @lends ox.api.ui */
                
                /**
                 * Set active module
                 * @params {string} name Module name, e.g. mail, calendar etc.
                 * @params {function ()} cont Continuation
                 * @example
                 * ox.api.ui.setModule("calendar"); // switch to calendar
                 */
                setModule: function (name, cont) {
                    // configuration?
                    if (activemodule === "configuration") {
                        // ask first
                        configuration_askforSave(function () {
                            setModule(name, cont);
                        });
                    } else {
                        // set now
                        setModule(name, cont);
                    }
                },
                
                /**
                 * Get active module
                 * @returns {string} Active module
                 * @example
                 * // get current module, e.g. mail, calendar, etc.
                 * ox.api.ui.getModule();
                 */
                getModule: function () {
                    // quite simple
                    return activemodule;
                },
                
                /**
                 * Logout
                 * @params {Object} [options] Logout options
                 * @params {string} [options.location] Location to redirect to after logout
                 * @params {boolean} [options.force] If true no checks are done before, e.g. save configuration
                 * @example
                 * // logout
                 * ox.api.ui.logout({
                 *   location: "http://www.google.de", // landing page after logout
                 *   force: true // don't try to save configuration before logout
                 * });
                 */
                logout: function (options) {
                    // defaults
                    var opt = $.extend({
                        location: logout_location.format(),
                        force: false
                    }, options || {});
                    // api call
                    var processLogout = function () {
                        ox.api.http.GET({
                            module: "login",
                            params: {
                                action: "logout"
                            },
                            // on success or error
                            complete: function (data) {
                                // remove unload handlers
                                window.onbeforeunload = null;
                                $(window).unbind("beforeunload");
                                // redirect
                                window.setTimeout(function () { 
                                    window.location.replace(opt.location); 
                                }, 0);
                            }
                        });
                    };
                    // force logout?
                    if (opt.force === true) {
                        // go!
                        processLogout();
                    } else {
                        // check things first
                        var save = ox.api.config.get("gui.global.save", 1);
                        // save configuration before every logout?
                        if (save === 0) {
                            // no!
                            processLogout();
                        } else {
                            // yes!
                            // replace with 1 (might contain deprecated values)
                            ox.api.config.set("gui.global.save", 1);
                            // save configuration
                            ox.api.config.save({
                                success: function (data) {
                                    // logout
                                    processLogout();
                                },
                                error: function (error) {
                                    // could not save configuration?
                                    if (error !== null && error.code === "SES-0203") {
                                        // session expired, so...
                                        processLogout();
                                    } else {
                                        var msg = error ? "\n\n" + formatError(error) : "";
                                        newConfirm(
                                            _("Server Error:"),
                                            _("Your configuration settings could not be saved. All your changes will be lost. Do you really want to continue?") + msg, /*i18n*/ 
                                            AlertPopup.YESNO, null, null, 
                                            processLogout, // yes 
                                            $.noop // no
                                        );
                                    }
                                }
                            });
                        }
                    }
                }
            };
        }()),
        
        /**
         * @name ox.api.window
         * @namespace
         */
        window: (function () {
            
            // globally unique identifier
            var GUID = 1;
            
            var newGUID = function () {
                return "OX." + (GUID++);
            };
            
            var getGUID = function (obj) {
                if (typeof obj === "string") {
                    return obj === "OX.#" ? "OX." + (GUID - 1) : obj;
                } else {
                    return obj ? obj.guid : undefined;
                }
            };
            
            // window registry
            var windows = {};
            // window data
            var windowData = {};
            // is nested window?
            var isNested = function () {
                // already in core window? (we rely on this simple variable not to mess up SELENIUM tests)
                return window.oxCoreWindow === true ? false : true;
            };
            
            return {
                /** @lends ox.api.window */
                
                /**
                 * Returns true if current window is a nested window.
                 * @name ox.api.window.isNested
                 * @type Bool
                 */
                isNested: isNested(),
                
                /**
                 * Reference to topmost core window
                 * @name ox.api.window.core
                 * @type {Window}
                 */
                core: isNested() ? (window.opener || window.parent).ox.api.window.core : window,
                
                /**
                 * Sets a new window factory consisting of an opening and a closing function. See {@link ox.api-windowFactory}
                 * @param opener {function ()} The opening function
                 * @param closer {function ()} The closing function
                 * @example
                 * ox.api.setWindowFactory(
                 *   // opener
                 *   function (opt) {
                 *     // open
                 *     var win = window.open(opt, "", ox.api.serializeWindowOptions(opt.options));
                 *     // register window
                 *     ox.api.registerWindow(opt.guid, win);
                 *     // set guid
                 *     win.guid = opt.guid;
                 *     // continuation
                 *     opt.success(win, opt.guid);
                 *   },
                 *   // closer
                 *   function (obj) {
                 *     // call native function "close"
                 *     obj.window.close();
                 *   }
                 * );
                 */
                setFactory: function (opener, closer) {
                    if (opener !== undefined && closer !== undefined) {
                        windowFactory = opener;
                        windowCloser = closer;
                    } else {
                        console.error("setFactory(opener, closer): One factory is missing");
                    }
                },
                
                /**
                 * @description Set a window positioner
                 * @param {function ()} fn A function that returns the window bounds
                 * @see ox.api-windowPositioner
                 */
                setPositioner: function (fn) {
                    windowPositioner = fn;
                },
                
                /**
                 * Define new window defaults
                 * @param {Object} options Default options
                 * @example
                 * ox.api.setDefaultWindowOptions({
                 *   resizable: "yes",
                 *   menubar: "no",
                 *   toolbar: "no",
                 *   status: "no",
                 *   location: "yes",
                 *   scrollbars: "no"
                 * });
                 */
                setDefaultOptions: function (options) {
                    defaultWindowOptions = options;
                },
                
                /**
                 * Serialize window options
                 * @param {Object} o Window options (e.g. width, height, scrollbars etc.)
                 * @returns {string} Serialized string
                 */
                serializeOptions: function (o) {
                    // merge (take care not to extend the defaults permanently)
                    var opt = $.extend({}, defaultWindowOptions, o || {});
                    // copy bounds
                    opt.width = o.width;
                    opt.height = o.height;
                    opt.left = o.x;
                    opt.top = o.y;
                    // serialize
                    return ox.util.serialize(opt, ",");
                },
                
                /**
                 * Register a new window.
                 * @param {string} guid The GUID of the window
                 * @param {Window} window The new window object
                 * @param {Object} options Further information to be stored (extends internal registry object)
                 * @returns {Object} The internal registry object
                 */
                register: function (guid, window, options) {
                    windows[guid] = $.extend({ window: window }, options);
                    return windows[guid];
                },
                
                /**
                 * Check if a window is registered under the given GUID
                 * @param {string} guid Window GUID. "OX.#" refers to last window.
                 * @returns {Boolean}
                 */
                exists: function (guid) {
                    return windows[getGUID(guid)] !== undefined;
                },
                
                /**
                 * Get window from internal window registry
                 * @param {string} guid Window GUID. "OX.#" refers to last window.
                 * @returns {Object} Internal registry object
                 * @example
                 * ox.api.window.get("OX.1");
                 */
                get: function (guid) {
                    guid = getGUID(guid);
                    return windows[guid] !== undefined ? windows[guid] : null;
                },
                
                /**
                 * Get all windows from internal window registry
                 * @returns {Object} All open windows
                 * @example
                 * var windows = ox.api.window.getAll();
                 */
                getAll: function () {
                    return windows;
                },
                
                /**
                 * Passes data to a window
                 * @param {Object} data Data
                 * @param {string} [guid] Existing GUID (to update data)
                 * @returns {string} New/Existing GUID
                 * @example
                 * var guid = ox.api.window.setData({ hello: "world" });
                 */
                setData: function (data, guid) {
                    if (data !== undefined) {
                        guid = getGUID(guid);
                        // exists?
                        if (guid === undefined || windowData[guid] === undefined) {
                            // get new GUID
                            guid = newGUID();
                        }
                        debug("setData", guid, data);
                        windowData[guid] = { data: data || {} };
                    }
                    return guid;
                },
                
                /**
                 * Get window-specific data
                 * @param {string|Window} GUID
                 * @returns {Object} Data
                 * @example
                 * var data = ox.api.window.getData("OX.1");
                 * console.log(data);
                 */
                getData: function (obj) {
                    // get guid
                    var guid = getGUID(obj);
                    if (guid !== undefined) {
                        // get data
                        if (isNested()) {
                            return this.core.ox.api.window.getData(guid);
                        } else {
                            var d = windowData[guid];
                            debug("getData", guid, d.data, "identifiedBy", obj);
                            return clone(d.data);
                        }
                    } else {
                        console.error("getData: guid is missing in given window!");
                    }
                },
                
                /**
                 * Open a new window using the current window factory
                 * @param {string} path URL Path
                 * @param {Object} [options] Options
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.window.open("test.html", { options: { locationbar: true, top: 100, left: 100, width: 400, height: 300 } });
                 */
                open: function (path, options) {
                    return constructWindow(path, processOptions({}, options));
                },
                
                /**
                 * Close a window
                 * @param {Window|string} obj window or GUID
                 * @example
                 * ox.api.window.close("OX.1");
                 * ox.api.window.close("OX.#"); // close last window
                 */
                close: function (obj) {
                    // get guid
                    var guid = getGUID(obj);
                    // defined?
                    if (windows[guid] !== undefined) {
                        windowCloser(windows[guid]);
                        delete windows[guid];
                        delete windowData[guid];
                    }
                },
                
                /**
                 * Select (i.e. focus) a window
                 * @param {Window|string} obj window or GUID
                 * @example
                 * ox.api.window.select("OX.1");
                 * ox.api.window.select("OX.#"); // select last window
                 */
                select: function (obj) {
                    // get guid
                    var guid = getGUID(obj);
                    // defined?
                    if (windows[guid] !== undefined) {
                        windows[guid].window.focus();
                    }
                },
                
                /**
                 * (Re-)select last window
                 * @example
                 * ox.api.window.selectLast();
                 * // same as
                 * ox.api.window.select("OX.#");
                 */
                selectLast: function () {
                    this.select("OX.#");
                },
                
                /**
                 * Returns last GUID
                 * @returns {string} GUID
                 * @example
                 * var guid = ox.api.window.getLastGUID();
                 */
                getLastGUID: function () {
                    return "OX." + (GUID - 1);
                },
                
                /**
                 * Returns next GUID
                 * @returns {string} GUID
                 */
                getNextGUID: function () {
                    return "OX." + (GUID);
                }
            };
        }()),
        
        /**
         * @param {string} [name] Name of the hash parameter
         * @returns {Object} Value or all values
         */
        getHash: function (name) {
            return name === undefined ? hashData : hashData[name];
        },

        /**
         * @param name {string} [Name] of the query parameter
         * @returns {Object} Value or all values
         */
        getParam: function (name) {
            return name === undefined ? queryData : queryData[name];
        },
        
        /**
         * Switch to modal mode
         * @param {Boolean} flag On/Off
         */
        setModal: function (flag) {
            // copy nodes from allnodes hash
            showNode("modal-dialog-decorator");
            showNode("modal-dialog");
            // show/hide?
            if (flag) {
                $("#modal-dialog-decorator").show();
                $("#modal-dialog").show();
            } else {
                $("#modal-dialog-decorator").hide();
                $("#modal-dialog").hide();
            }
            // remember
            modal = !!flag;
        },
        
        isModal: function () {
            return modal;
        },
        
        /**
         * @name ox.api.mail
         * @namespace
         */
        mail: (function () {
        
            var open = function (options) {
                // options
                var opt = processOptions({
                    params: {
                        folder: activefolder === undefined ? "default0/INBOX": activefolder,
                        id: undefined,
                        account: undefined,
                        noimg: !corewindow.loadblockedimages,
                        action: ""
                    },
                    message: undefined,
                    closer: function (opt) {
                        that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                    }
                }, options);
                var p = opt.params;
                // message object?
                if (opt.message !== undefined) {
                    p.folder = opt.message.folder_id;
                    p.id = opt.message.id;
                }
                // fix account
                if (p.account === undefined) {
                    p.account = corewindow.getMailAccountIdByFolder(p.folder);
                }
                if ((p.id + "").match(/^default/)) {
                    p.account = corewindow.getMailAccountIdByFolder(p.id);
                }
                // construct
                return constructWindow("newMail.html", opt);
            };
            
            return {
                /** @lends ox.api.mail */
                
                /**
                 * Select an email (changes view & folder)
                 * @param {Object} options Options
                 * @param {string} options.id Id
                 * @param {string} [options.folder] Folder
                 * @param {function ()} [options.success] Callback
                 * @example
                 * ox.api.mail.select({ id: "4451" });
                 * ox.api.mail.select({ id: "4451", folder: "default0/INBOX" });
                 */
                select: function (options) {
                    processSelect("mail", options);
                },
                
                /**
                 * Generic open function
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 */
                open: function (options) {
                    return open(options);
                },

                /**
                 * Open compose dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog 
                 * ox.api.mail.compose();
                 * 
                 * // open dialog with subject set
                 * ox.api.mail.compose({ data: { subject: "Hello World" } });
                 * 
                 * // open dialog with subject, recipients, and mail text set
                 * ox.api.mail.compose({
                 *   data: {
                 *     to: [["Max Mustermann", "max.mustermann@example.org"], ["John Doe", "john.doe@exmaple.org"]],
                 *     subject: "Meeting Minutes",
                 *     priority: 2,
                 *     mailtext: "Hi Max and John,&lt;br/>&lt;br/>enclosed you'll find the meeting minutes from this morning.&lt;br/>&lt;br/>Cheers,&lt;br/>Lila", 
                 *     disp_notification_to: true
                 *   }
                 * });
                 */
                compose: function (options) {
                    return open($.extend(true, options || {}, { params: { action: "" }}));
                },

                /**
                 * Open "forward" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.forward({ params: { id: 4415 }});
                 */
                forward: function (options) {
                    return open($.extend(true, options || {}, { params: { action: "forward" }}));
                },

                /**
                 * Open "reply" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.reply({ params: { id: 4415 }});
                 */
                reply: function (options) {
                    return open($.extend(true, options || {}, { params: { action: "reply" }}));
                },

                /**
                 * Open "replyAll" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.replyAll({ params: { id: 4415 }});
                 */
                replyAll: function (options) {
                    return open($.extend(true, options || {}, { params: { action: "replyAll" }}));
                },

                /**
                 * Open "edit draft" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.draft({ params: { id: 4415 }});
                 */
                draft: function (options) {
                    return open($.extend(true, options || {}, { params: { action: "draft" }}));
                },

                /**
                 * Open detail view
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.view({params: { id: 4415 }});
                 * ox.api.mail.view({params: { id: 4415, folder: "default0/INBOX" }});
                 */
                view: function (options) {
                    // options
                    var opt = processOptions({
                        params: {
                            id: undefined,
                            folder: "default0/INBOX",
                            sid: undefined
                        },
                        closer: function (opt) {
                            that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                        }
                    }, options);
                    // set type
                    opt.params.type = "isNestedMessage";
                    // construct
                    return constructWindow("detailMail.html", opt);
                },
                
                /**
                 * Open source view
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.mail.showSource({ params: { id: 4415 }});
                 */
                showSource: function (options) {
                    // options
                    var opt = processOptions({
                        params: {
                            triggerevent: "OX_Popup_Cb",
                            folder: "default0/INBOX"
                        },
                        mail: undefined,
                        options: { scrollbars: "yes" },
                        closer: function (opt) {
                            that.window.close(that.window.get(opt.guid).window);
                        }
                    }, options);
                    // handler
                    var handler = function (win) {
                        // get mail source
                        json.get(AjaxRoot + "/mail?" + 
                            ox.util.serialize({ action: "get", session: session, folder: opt.params.folder, id: opt.params.id, src: "true" }),
                            null,
                            function (response) {
                            if (opt.mail) {
                                OXMailMapping.editObjectsInternal([opt.mail], null, true, [{ flag: 32, bool: true }]);
                            }
                            var text = _("Unable to display E-Mail source."); /*i18n*/
                            if (response.data) {
                                text = response.data.replace(/\r/g, ""); // IE does not like CRs here
                            }
                            // show source
                            var doc = win.document, area;
                            $("body", doc).css({
                                margin: "0px",
                                padding: "10px",
                                backgroundColor: "lightyellow",
                                fontFamily: "Arial, Helvetica, sans-serif",
                                fontSize: "9pt",
                                borderTop: "5px solid #fc0"
                            }).append(
                                $("<div/>", doc).css({ 
                                    position: "absolute",
                                    top: "5px", 
                                    left: "0px", 
                                    right: "10px", 
                                    height: "25px", 
                                    lineHeight: "25px",
                                    textAlign: "right"
                                }).
                                append(
                                    $("<a/>", doc).attr({ href: "#" }).
                                    text(_("Close")).
                                    bind("click", function (e) {
                                        ox.api.window.close(win);
                                    })
                                )
                            ).append(
                                $("<div/>", doc).css({
                                    position: "absolute",
                                    top: "30px",
                                    left: "0px",
                                    right: "0px",
                                    bottom: "0px",
                                    padding: "10px"
                                }).append(
                                    area = $("<textarea/>", doc).css({
                                        width: "100%",
                                        height: "100%",
                                        backgroundColor: "lightyellow",
                                        border: "0px none",
                                        whiteSpace: "pre"
                                    }).val(text)
                                )
                            );
                            // focus textarea
                            area.focus();
                            // unregister
                            unregister('OX_Popup_Cb', handler);
                        });
                    };
                    // talk via event
                    register('OX_Popup_Cb', handler);
                    // construct
                    return constructWindow("popup_trigger_cb.html", opt);
                },
                
                /**
                 * Open print view
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @example
                 * ox.api.mail.print({ id: 4415 });
                 */
                print: function (options) {
                    // options
                    var opt = processOptions({
                        id: undefined,
                        folder: "default0/INBOX",
                        closer: function (opt) {
                            ox.api.window.close(opt.guid);
                        }
                    }, options);
                    // get mail
                    this.httpGet({
                        params: {
                            id: opt.id,
                            folder: opt.folder,
                            view: "noimg"
                        },
                        success: function (data) {
                            // copy property
                            data.attachments_html_noimage = data.attachments;
                            // print (global function in js/mail.js)
                            mail_printSingleMail(data, opt.closer);
                        }
                    });
                },
                
                /**
                 * Get mail (HTTP API)
                 * @param {Object} [options] Options
                 */
                httpGet: function (options) {
                    // options
                    var opt = options || { params: {} };
                    // prepare options
                    ox.api.http.GET({
                        module: "mail",
                        params: {
                            action: "get",
                            id: opt.params.id,
                            folder: opt.params.folder || "default0/INBOX",
                            view: undefined
                        },
                        success: opt.success || $.noop,
                        error: opt.error || $.noop,
                        appendColumns: false
                    });
                }
            };
        }()),
        
        /**
         * @namespace
         * @name ox.api.calendar
         */
        calendar: (function () {
            
            var fix = function (o) {
                var p = o.params;
                p.folder_id = p.folder_id !== undefined ? o.folder_id : p.folder;
                delete p.folder;
                // recurrence position set?
                if (p.recurrence_position !== undefined && p.recurrence_position > 0) {
                    p.singleappointment = "yes";
                }
                return o;
            };
            
            var open = function (options) {
                // options
                var opt = processOptions({
                    params: {
                        modul: "new",
                        folder: config.folder.calendar,
                        view: currentCalendarView,
                        folderOwner: undefined,
                        start_date: new Date().getTime(),
                        end_date: "",
                        full_time: false,
                        id: undefined
                    },
                    closer: function (opt) {
                        that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                    }
                }, options);
                // construct
                return constructWindow("newAppointment.html", fix(opt));
            };
            
            return {
                /** @lends ox.api.calendar */
                
                /**
                 * Select an appointment
                 * @param {Object} options Options
                 * @param {string} options.id Id
                 * @param {string} [options.folder] Folder
                 * @param {function ()} [options.success] Callback
                 * @example
                 * ox.api.calendar.select({ id: "158402" });
                 */
                select: function (options) {
                    processSelect("calendar", options);
                },
                
                /**
                 * Open compose dialog. Fields are described <a href="http://oxpedia.org/index.php?title=HTTP_API#DetailedTaskAndAppointmentData">here</a>.
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @param {Object} [options.data] Data. Prefilled fields.
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog
                 * ox.api.calendar.compose();
                 * 
                 * // open dialog with fields set
                 * ox.api.calendar.compose({
                 *   data: {
                 *     title: "Meeting",
                 *     location: "Berlin",
                 *     start_date: 1280919600000, // UTC
                 *     end_date: 1280937600000, // UTC
                 *     private_flag: false,
                 *     shown_as: 3,
                 *     full_time: false,
                 *     categories: "Meeting, Business",
                 *     note: "Please be prepared for the meeting.",
                 *     alarm: 300 // = 300 seconds = 5 minutes
                 *   }
                 * });
                 *
                 * // how to deal with timestamps
                 * ox.api.calendar.compose({
                 *   data: {
                 *     title: "Do not use local time!",
                 *     start_date: Date.UTC(2010, 0, 1, 12, 0, 0), // shows 12:00 PM
                 *     end_date: Date.UTC(2010, 0, 1, 14, 0, 0), // shows 02:00 PM (14:00)
                 *   }
                 * });
                 * 
                 * // open dialog with exemplary success handler
                 * ox.api.calendar.compose({
                 *   data: {
                 *     title: "Using the success handler",
                 *   },
                 *   success: function (win) {
                 *     // focus core window
                 *     window.focus()
                 *     // say hello
                 *     alert("Hello");
                 *     // focus new window via select
                 *     ox.api.window.select(ox.api.window.getLastGUID()); // e.g. "OX.1"
                 *   }
                 * });
                 */
                compose: function (options) {
                    // get next guid
                    var guid = ox.api.window.getNextGUID();
                    // get options
                    var opt = $.extend(true, options || {}, { params: { modul: "new" }});
                    // continuation
                    var cont = function (folder) {
                        // for private or shared folders add owner to participant list
                        if (folder.oxfolder.data.type === 1 || folder.oxfolder.data.type === 3) {
                            opt.params.folderOwner = folder.oxfolder.data.created_by;
                        } else {
                            opt.params.folderOwner = configGetKey('identifier');
                        }
                        // go!
                        open(opt);
                    };
                    // get folder data (default is active folder)
                    var id = opt.params.folder !== undefined ? opt.params.folder : (activefolder ? activefolder : config.folder.calendar);
                    // first try
                    oMainFolderTree.cache.get_folder(id, function (folder) {
                        // is calendar?
                        if (folder.oxfolder.data.module === "calendar") {
                            // ok
                            cont(folder);
                        } else {
                            // fail?
                            if (opt.params.folder !== undefined) {
                                // yes
                                triggerEvent("OX_New_Error", 4, _("Can not create appointment in a folder other than appointment folder")); /*i18n*/
                            } else {
                                // try default calendar folder
                                oMainFolderTree.cache.get_folder(config.folder.calendar, cont);
                            }
                        }
                    });
                    // return potential GUID
                    return guid;
                },
                
                /**
                 * Open edit dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.calendar.edit({ params: { id: 158302 }}); // edit appointment
                 * ox.api.calendar.edit({ params: { id: 158374, recurrence_position: 0 }}); // edit series
                 * ox.api.calendar.edit({ params: { id: 158374, recurrence_position: 1 }}); // edit exception (1st appointment of series)
                 * ox.api.calendar.edit({ params: { id: 158374, recurrence_position: 5 }}); // edit exception (5th appointment of series)
                 */
                edit: function (options) {
                    return open($.extend(true, options || {}, { params: { modul: "edit" }}));
                },

                /**
                 * Open print dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.calendar.print(); // prints day view (today)
                 * ox.api.calendar.print({ view: "month" }); // prints month view (current month)
                 * ox.api.calendar.print({ view: "week", year: 2010, month: 1 }); // prints month view (January 2010)
                 */
                print: function (options) {
                    
                    // options
                    var opt = processOptions({
                        view: "day",
                        day: activeDay,
                        month: activeMonth + 1,
                        year: activeYear,
                        options: { scrollbars: "yes" },
                        closer: function (opt) {
                            ox.api.window.close(opt.guid);
                        }
                    }, options);
                    // get start date
                    var date = new Date(Date.UTC(opt.year, opt.month - 1, opt.day)), start, days;
                    var self = this;
                    // printer
                    var print = function (template, start, end, numdays) {
                        // handler
                        var success = opt.success;
                        opt.success = function (win, guid) {
                            // get via HTTP API
                            self.httpPrint({
                                params: {
                                    start: start,
                                    end: end,
                                    template: template,
                                    folder: calendar_getAllFoldersAttribute() ? undefined : activefolder,
                                    work_week_day_amount: numdays ? numdays : undefined
                                },
                                success: function (response) {
                                    // still open?
                                    if (win && !window.closed) {
                                        // insert response
                                        win.document.open();
                                        // remove window.print from onload
                                        response = response.replace(/window\.print\(\)/, "");
                                        // firefox likes this
                                        if (ox.browser.Gecko) {
                                            response = response.replace(/<\/html>/, '<script type="text/javascript">window.print();</script>');
                                        }
                                        win.document.write(response);
                                        win.document.close();
                                        // for IE/Safari
                                        if (ox.browser.IE || ox.browser.WebKit) {
                                            setTimeout(function () {
                                                win.print();
                                            }, 0);
                                        }
                                        // continue
                                        success(win, guid);
                                    }
                                }
                            });
                        };
                        // open dummy window
                        return constructWindow("newInfoItemHidden.html", opt);
                    };
                    // process view
                    switch (opt.view) {
                    case "day":
                        start = date.getTime();
                        return print("cp_dayview_table.tmpl", start, start + 864e5, null);
                    case "workweek":
                        start = getDayInSameWeek(date, configGetKey("gui.calendar.workweek.startday")) * 864e5;
                        days = configGetKey("gui.calendar.workweek.countdays");
                        return print("cp_weekview_table.tmpl", start, start + days * 864e5, days);
                    case "week":
                        start = getDayInSameWeek(date, 1) * 864e5; // 1 = Monday
                        return print("cp_weekview_table.tmpl", start, start + 6048e5, null);
                    case "month":
                        return print("cp_monthview_table.tmpl", Date.UTC(opt.year, opt.month - 1, 1), Date.UTC(opt.year, opt.month, 1), null);
                    }
                },
                
                /**
                 * Get calendar object (HTTP API)
                 * @param {Object} options Request options
                 * @param {Object} options.params Request parameters
                 * @param {function (response)} options.success Success handler
                 * @param {function (response)} options.error Error handler
                 * @example
                 * ox.api.calendar.httpGet({ params: { id: 158302 } });
                 * ox.api.calendar.httpGet({ params: { id: 158302 }, success: ox.util.inspect });
                 */
                httpGet: function (options) {
                    ox.api.http.GET(processHTTPOptions("calendar", {
                        action: "get",
                        folder: config.folder.calendar,
                        id: undefined
                    }, options));
                },
                
                /**
                 * Get all calendar objects in a folder (HTTP API)
                 * @param {Object} options Request options#
                 * @example
                 * ox.api.calendar.httpGetAll();
                 * ox.api.calendar.httpGetAll({ folder: 4384 });
                 */
                httpGetAll: function (options) {
                    var now = (new Date()).getTime();
                    ox.api.http.GET(processHTTPOptions("calendar", {
                        action: "all",
                        folder: config.folder.calendar,
                        start: now,
                        end: now + 1000 * 3600 * 24 * 7
                    }, options));
                },
                
                /**
                 * Print calendar via template (HTTP API)
                 */
                httpPrint: function (options) {
                    var now = (new Date()).getTime();
                    ox.api.http.GET(processHTTPOptions("printCalendar", {
                        start: now,
                        end: now + 864e5, // day
                        template: "cp_dayview_table.tmpl", // day
                        folder: undefined, // = all folders
                        work_day_start_time: getWWStartTime(),
                        work_day_end_time: getWWEndTime(),
                        work_week_day_amount: undefined
                    }, options, { 
                        dataType: "text",
                        appendColumns: false
                    }));
                }
            };
            
        }()),
        
        /**
         * @namespace
         * @name ox.api.contact
         */
        contact: (function () {
            
            var open = function (file, options) {
                // options
                var opt = processOptions({
                    params: {
                        modul: "new",
                        currentView: undefined,
                        folder: config.folder.contacts, // = personal folder. 6 = global address book
                        session: session
                    },
                    closer: function (opt) {
                        that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                    }
                }, options);
                // construct
                return constructWindow(file, opt);
            };
            
            return {
                /** @lends ox.api.contact */
                
                /**
                 * Select a contact
                 * @param {Object} options Options
                 * @param {string} options.id Id
                 * @param {string} [options.folder] Folder
                 * @param {function ()} [options.success] Callback
                 * @example
                 * ox.api.contact.select({ id: "93437" });
                 */
                select: function (options) {
                    processSelect("contacts", options);
                },
                
                /**
                 * Open "compose new contact" dialog. . Fields are described <a href="http://oxpedia.org/index.php?title=HTTP_API#DetailedContactData">here</a>.
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog
                 * ox.api.contact.compose();
                 * 
                 * // open dialog with predefined fields
                 * ox.api.contact.compose({
                 *   data: {
                 *     title: "Mr.", first_name: "Max", last_name: "Mustermann",
                 *     company: "Mustermann Inc.", street_business: "Musterstr. 11",
                 *     postal_code_business: "12345", city_business: "Berlin", state_business: "Berlin",
                 *     country_business: "DE", telephone_business1: "+49 1234 5678",
                 *     fax_business: "+49 1234 56990", email1: "max.mustermann@example.org",
                 *     url: "http://www.mustermann.inc", number_of_employees: "6", department: "IT",
                 *     note: "Business Contact, Mustermann Inc.", birthday: 147913200000
                 *   }
                 * });
                 */
                compose: function (options) {
                    return open("newContact.html", $.extend(true, options || {}, { params: { modul: "new" }}));
                },

                /**
                 * Open "compose distribution list" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog
                 * ox.api.contact.composeDistributionList();
                 * 
                 * // open dialog with prefilled fields
                 * ox.api.contact.composeDistributionList({
                 *   data: {
                 *     title: "My List", 
                 *     distribution_list: [
                 *       { display_name: "Max Mustermann", mail: "max.muster@example.org"}, 
                 *       { display_name: "John Doe", mail: "john.doe@example.org" }
                 *     ]
                 *   }
                 * });
                 */
                composeDistributionList: function (options) {
                    return open("newDistributionList.html", $.extend(true, options || {}, { params: { modul: "new" }}));
                },
                
                /**
                 * Open "edit contact" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.contact.edit({ params: { id: 73303 }});
                 * ox.api.contact.edit({ params: { id: 73303, folder: 4385 }}); // private contact in private contact folder (default)
                 * ox.api.contact.edit({ params: { id: 29, folder: 6 }}); // 6 = global address book
                 */
                edit: function (options) {
                    return open("newContact.html", $.extend(true, options || {}, { params: { modul: "edit" }}));
                },
                
                /**
                 * Open "edit distribution list" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.contact.editDistributionList({ params: { id: 115787 }});
                 * ox.api.contact.editDistributionList({ params: { id: 115787, folder: 4385 }});
                 */                
                editDistributionList: function (options) {
                    return open("newDistributionList.html", $.extend(true, options || {}, { params: { modul: "edit" }}));
                },

                /**
                 * Open dialog to create new contact based on existing contact
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.contact.duplicate({ params: { id: 29, folder: 6 }}); // 6 = global address book
                 */                
                duplicate: function (options) {
                    return open("newContact.html", $.extend(true, options || {}, { params: { modul: "duplicate" }}));
                },

                /**
                 * Open dialog to create new distribution list based on existing list
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.contact.duplicateDistributionList({ params: { id: 115787 }});
                 */
                duplicateDistributionList: function (options) {
                    return open("newDistributionList.html", $.extend(true, options || {}, { params: { modul: "duplicate" }}));
                }
            };
            
        }()),
        
        /**
         * @namespace
         * @name ox.api.task
         */
        task: (function () {
            
            var open = function (options) {
                // options
                var opt = processOptions({
                    params: {
                        modul: undefined,
                        folder: configGetKey("folder.tasks"),
                        session: session
                    },
                    closer: function (opt) {
                        that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                    }
                }, options);
                // construct
                return constructWindow("newTask.html", opt);
            };
            
            return {
                /** @lends ox.api.task */
                
                /**
                 * Select a task
                 * @param {Object} options Options
                 * @param {string} options.id Id
                 * @param {string} [options.folder] Folder
                 * @param {function ()} [options.success] Callback
                 * @example
                 * ox.api.task.select({ id: "3810" });
                 */
                select: function (options) {
                    processSelect("tasks", options);
                },
                
                /**
                 * Open "compose new task" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog
                 * ox.api.task.compose();
                 * 
                 * // open dialog with predefined fields
                 * ox.api.task.compose({
                 *   data: {
                 *     title: "Setup new Hardware",
                 *     start_date: 1280919600000,
                 *     end_date: 1280937600000,
                 *     priority: 3,
                 *     note: "We need to setup the new hardware. Be prepared.",
                 *     alarm: 1280937600000
                 *   }
                 * });
                 */
                compose: function (options) {
                    return open($.extend(true, options || {}, { params: { modul: undefined }}));
                },
                
                /**
                 * Open "edit task" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.task.edit({ params: { id: 3616 }});
                 */
                edit: function (options) {
                    return open($.extend(true, options || {}, { params: { modul: "edit" }}));
                },
                
                /**
                 * Open dialog to create new task based on existing task
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.task.duplicate({ params: { id: 3616 }});
                 */
                duplicate: function (options) {
                    return open($.extend(true, options || {}, { params: { modul: "duplicate" }}));
                }
            };
        }()),
        
        /**
         * @namespace
         * @name ox.api.infostore
         */
        infostore: (function () {
            
            var fix = function (options) {
                options.params.edit_id = options.params.id;
                delete options.params.id;
                return options;
            };
            
            var open = function (options) {
                // options
                var opt = processOptions({
                    params: {
                        id: undefined,
                        folder: configGetKey("folder.infostore")
                    },
                    closer: function (opt) {
                        that.window.get(opt.guid).window.triggerEvent("OX_Cancel_Object");
                    }
                }, options);
                // construct
                return constructWindow("newInfoItemMain.html", fix(opt));
            };
            
            return {
                /** @lends ox.api.infostore */
                
                /**
                 * Select an infostore item
                 * @param {Object} options Options
                 * @param {string} options.id Id
                 * @param {string} [options.folder] Folder
                 * @param {function ()} [options.success] Callback
                 * @example
                 * ox.api.infostore.select({ id: "16560" });
                 */
                select: function (options) {
                    processSelect("infostore", options);
                },
                
                /**
                 * Open compose dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * // open simple dialog
                 * ox.api.infostore.compose();
                 * 
                 * // prefill some fields
                 * ox.api.infostore.compose({
                 *   data: {
                 *     title: "Exemplary document",
                 *     description: "Latest documentation", 
                 *     categories: ["Important", "Internal", "Business"]
                 *   }
                 * });
                 */
                compose: function (options) {
                    return open($.extend(true, options || {}, { params: { id: -1 }}));
                },
                
                /**
                 * Open "compose from attachment" dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.infostore.composeFromAttachment({
                 *   params: {
                 *     attachmail: 1,
                 *     filename: "test.doc",
                 *     mailid: 4711,
                 *     mailfolder: "default0/INBOX"
                 *   },
                 *   success: function (win) {
                 *     // do something on success
                 *   }
                 * });
                 */
                composeFromAttachment: function (options) {
                    return open($.extend(true, options || {}, { params: { id: -1 }}));
                },
                
                /**
                 * Open edit dialog
                 * @param {Object} [options] Options {@link ox.api-constructWindow}
                 * @returns {string} New window GUID
                 * @example
                 * ox.api.infostore.edit({ params: { id: 16560 }});
                 */
                edit: function (options) {
                    return open(options);
                },
                
                /**
                 * Download file
                 * @param {string} id Infostore item id
                 * @param {string} [filename] Filename (helps getting a useful filename in IE)
                 * @example
                 * ox.api.infostore.download(16449);
                 */
                download: function (id, filename) {
                    // continuation
                    var cont = function (id, filename) {
                        // url parts
                        var parts = [AjaxRoot, "infostore", filename];
                        // create url
                        var url = parts.join("/") + "?" + ox.util.serialize({
                            action: "document",
                            id: id,
                            content_type: "application/octet-stream",
                            session: session
                        });
                        // create hidden frame
                        var iframe = $("<iframe/>", { src: url }).css({
                            display: "none",
                            width: "0px",
                            height: "0px"
                        }).one("load", function (e) {
                            // remove frame immediately
                            setTimeout(function () {
                                iframe.remove();
                            }, 10);
                        }).appendTo(document.body);
                    };
                    // missing filename? (just in IE)
                    if (ox.browser.IE && filename === undefined) {
                        // get object via HTTP API
                        ox.api.http.GET({
                            module: "infostore",
                            params: {
                                action: "get",
                                id: 16449,
                                columns: "702"
                            },
                            success: function (response) {
                                cont(id, response.filename);
                            }
                        });
                    } else {
                        cont(id, filename);
                    }
                }
            };
        }()),
        
        /**
         * @namespace
         * @name ox.api.http
         */
        http: (function () {
            
            // default columns for each module
            var idMapping = {
                "common" : {
                    "1" : "id",
                    "2" : "created_by",
                    "3" : "modified_by",
                    "4" : "creation_date",
                    "5" : "last_modified",
                    "100" : "categories",
                    "101" : "private_flag",
                    "102" : "color_label",
                    "104" : "number_of_attachments",
                    "20" : "folder_id"
                },
                "mail" : {
                    "102" : "color_label",
                    "600" : "id",
                    "601" : "folder_id",
                    "602" : "attachment",
                    "603" : "from",
                    "604" : "to",
                    "605" : "cc",
                    "606" : "bcc",
                    "607" : "subject",
                    "608" : "size",
                    "609" : "sent_date",
                    "610" : "received_date",
                    "611" : "flags",
                    "612" : "level",
                    "613" : "disp_notification_to",
                    "614" : "priority",
                    "615" : "msgref",
                    "651" : "flag_seen",
                    "652" : "account_name"
                },
                "contacts" : {
                    "500" : "display_name",
                    "501" : "first_name",
                    "502" : "last_name",
                    "503" : "second_name",
                    "504" : "suffix",
                    "505" : "title",
                    "506" : "street_home",
                    "507" : "postal_code_home",
                    "508" : "city_home",
                    "509" : "state_home",
                    "510" : "country_home",
                    "511" : "birthday",
                    "512" : "marital_status",
                    "513" : "number_of_children",
                    "514" : "profession",
                    "515" : "nickname",
                    "516" : "spouse_name",
                    "517" : "anniversary",
                    "518" : "note",
                    "519" : "department",
                    "520" : "position",
                    "521" : "employee_type",
                    "522" : "room_number",
                    "523" : "street_business",
                    "525" : "postal_code_business",
                    "526" : "city_business",
                    "527" : "state_business",
                    "528" : "country_business",
                    "529" : "number_of_employees",
                    "530" : "sales_volume",
                    "531" : "tax_id",
                    "532" : "commercial_register",
                    "533" : "branches",
                    "534" : "business_category",
                    "535" : "info",
                    "536" : "manager_name",
                    "537" : "assistant_name",
                    "538" : "street_other",
                    "539" : "city_other",
                    "540" : "postal_code_other",
                    "541" : "country_other",
                    "542" : "telephone_business1",
                    "543" : "telephone_business2",
                    "544" : "fax_business",
                    "545" : "telephone_callback",
                    "546" : "telephone_car",
                    "547" : "telephone_company",
                    "548" : "telephone_home1",
                    "549" : "telephone_home2",
                    "550" : "fax_home",
                    "551" : "cellular_telephone1",
                    "552" : "cellular_telephone2",
                    "553" : "telephone_other",
                    "554" : "fax_other",
                    "555" : "email1",
                    "556" : "email2",
                    "557" : "email3",
                    "558" : "url",
                    "559" : "telephone_isdn",
                    "560" : "telephone_pager",
                    "561" : "telephone_primary",
                    "562" : "telephone_radio",
                    "563" : "telephone_telex",
                    "564" : "telephone_ttytdd",
                    "565" : "instant_messenger1",
                    "566" : "instant_messenger2",
                    "567" : "telephone_ip",
                    "568" : "telephone_assistant",
                    "569" : "company",
                    "570" : "image1",
                    "571" : "userfield01",
                    "572" : "userfield02",
                    "573" : "userfield03",
                    "574" : "userfield04",
                    "575" : "userfield05",
                    "576" : "userfield06",
                    "577" : "userfield07",
                    "578" : "userfield08",
                    "579" : "userfield09",
                    "580" : "userfield10",
                    "581" : "userfield11",
                    "582" : "userfield12",
                    "583" : "userfield13",
                    "584" : "userfield14",
                    "585" : "userfield15",
                    "586" : "userfield16",
                    "587" : "userfield17",
                    "588" : "userfield18",
                    "589" : "userfield19",
                    "590" : "userfield20",
                    "592" : "distribution_list",
                    "594" : "number_of_distribution_list",
                    "596" : "contains_image1",
                    "597" : "image_last_modified",
                    "598" : "state_other",
                    "599" : "file_as",
                    "104" : "number_of_attachments",
                    "601" : "image1_content_type",
                    "602" : "mark_as_distributionlist",
                    "605" : "default_address",
                    "524" : "internal_userid",
                    "606" : "image1_url"
                },
                "calendar" : {
                    "200" : "title",
                    "201" : "start_date",
                    "202" : "end_date",
                    "203" : "note",
                    "204" : "alarm",
                    "207" : "recurrence_position",
                    "208" : "recurrence_date_position",
                    "209" : "recurrence_type",
                    "212" : "days",
                    "213" : "days_in_month",
                    "214" : "month",
                    "215" : "interval",
                    "216" : "until",
                    "220" : "participants",
                    "221" : "users",
                    "400" : "location",
                    "401" : "full_time",
                    "402" : "shown_as"
                },
                "infostore" : {
                    "700" : "title",
                    "701" : "url",
                    "702" : "filename",
                    "703" : "file_mimetype",
                    "704" : "file_size",
                    "705" : "version",
                    "706" : "description",
                    "707" : "locked_until",
                    "708" : "file_md5sum",
                    "709" : "version_comment",
                    "710" : "current_version",
                    "711" : "number_of_versions"
                },
                "task" : {
                    "200" : "title",
                    "201" : "start_date",
                    "202" : "end_date",
                    "203" : "note",
                    "204" : "alarm",
                    "209" : "recurrence_type",
                    "212" : "days",
                    "213" : "days_in_month",
                    "214" : "month",
                    "215" : "internal",
                    "216" : "until",
                    "220" : "participants",
                    "221" : "users",
                    "300" : "status",
                    "301" : "percent_completed",
                    "302" : "actual_costs",
                    "303" : "actual_duration",
                    "305" : "billing_information",
                    "307" : "target_costs",
                    "308" : "target_duration",
                    "309" : "priority",
                    "312" : "currency",
                    "313" : "trip_meter",
                    "314" : "companies",
                    "315" : "date_completed"
                },
                "folders" : {
                    "1" : "id",
                    "2" : "created_by",
                    "3" : "modified_by",
                    "4" : "creation_date",
                    "5" : "last_modified",
                    "6" : "last_modified_utc",
                    "20" : "folder_id",
                    "300" : "title",
                    "301" : "module",
                    "302" : "type",
                    "304" : "subfolders",
                    "305" : "own_rights",
                    "306" : "permissions",
                    "307" : "summary",
                    "308" : "standard_folder",
                    "309" : "total",
                    "310" : "new",
                    "311" : "unread",
                    "312" : "deleted",
                    "313" : "capabilities",
                    "314" : "subscribed",
                    "315" : "subscr_subflds",
                    "316" : "standard_folder_type"
                },
                "user" : {
                    "610" : "aliases",
                    "611" : "timezone",
                    "612" : "locale",
                    "613" : "groups",
                    "614" : "contact_id",
                    "615" : "login_info"
                }
            };
            
            // extend with commons (not all modules use common columns, e.g. folders)
            $.extend(idMapping.contacts, idMapping.common);
            $.extend(idMapping.calendar, idMapping.common);
            $.extend(idMapping.infostore, idMapping.common); 
            delete idMapping.infostore["101"]; // not "common" here (exception)
            delete idMapping.infostore["104"];
            $.extend(idMapping.task, idMapping.common);
            $.extend(idMapping.user, idMapping.contacts, idMapping.common);
            
            // get all columns of a module
            var getAllColumns = function (module) {
                // get ids
                var ids = idMapping[module];
                // flatten this array
                var tmp = [];
                for (var column in ids) {
                    tmp.push(column);
                }
                tmp.sort(function (a, b) {
                    return a - b;
                });
                return tmp;
            };
            
            // transform arrays to objects
            var makeObject = function (data, module, columns) {
                // get ids
                var ids = idMapping[module];
                var obj = {}, i = 0, $l = data.length;
                // loop through data
                for (; i < $l; i++) {
                    // get id
                    var id = ids[columns[i]];
                    // extend object
                    obj[id] = data[i];
                }
                return obj;
            };
            
            var processOptions = function (options, type) {
                // defaults
                var o = $.extend({
                    module: "",
                    params: {},
                    data: {},
                    dataType: "json",
                    success: $.noop,
                    error: $.noop,
                    complete: $.noop,
                    appendColumns: type === "GET" ? false : true
                }, options || {});
                // prepend root
                o.url = AjaxRoot + "/" + o.module;
                // add session
                o.params.session = session;
                // add columns
                if (o.appendColumns === true && o.params.columns === undefined) {
                    o.params.columns = getAllColumns(o.module).join(",");
                }
                // data & body
                if (type === "GET" || type === "POST") {
                    // GET & POST
                    o.data = o.params;
                } else {
                    // PUT & DELETE
                    o.url += "?" + ox.util.serialize(o.params);
                    o.data = JSON.serialize(o.data);
                    o.processData = false;
                    o.contentType = "text/javascript";
                }
                // done
                return o;
            };
            
            var processData = function (data, module, columns) {
                // not array or no columns given?
                if (!$.isArray(data) || !columns) {
                    // typically from "action=get" (already sanitized)
                    return data;
                } else {
                    // POST/PUT - sanitize data
                    var i = 0, $l = data.length, sanitized = [];
                    var columnList = columns.split(",");
                    for (; i < $l; i++) {
                        sanitized.push(makeObject(data[i], module, columnList));
                    }
                    return sanitized;
                }
            };
            
            var ajax = function (options, type) {
                // process options
                var o = processOptions(options, type);
                // ajax
                $.ajax({
                    // type (GET, POST, PUT, ...)
                    type: type,
                    // url
                    url: o.url,
                    // data
                    data: o.data,
                    dataType: o.dataType,
                    contentType: o.contentType || "application/x-www-form-urlencoded",
                    // handler
                    success: function (response) {
                        // process response
                        if (response && response.error !== undefined) {
                            // server error
                            o.error(response);
                        } else {
                            // success
                            if (o.dataType === "json") {
                                // response? (logout e.g. hasn't any)
                                if (response) {
                                    var data = processData(response.data, o.module, o.params.columns);
                                    o.success(data);
                                } else {
                                    o.success({});
                                }
                            } else {
                                // e.g. plain text
                                o.success(response || "");
                            }
                        }
                    },
                    error: function (xhr) {
                        o.error({}, xhr);
                    },
                    complete: function (xhr) {
                        o.complete({}, xhr);
                    }
                });
            };
            
            return {
                /** @lends ox.api.http */
                
                /**
                 * Send a GET request
                 * @param {Object} options Request options
                 * @param {string} options.module Module, e.g. folder, mail, calendar etc.
                 * @param {Object} options.params URL parameters
                 * @example
                 * ox.api.http.GET({ module: "mail", params: { action: "all", folder: "default0/INBOX" }});
                 */
                GET: function (options) {
                    ajax(options, "GET");
                },
                
                /**
                 * Send a POST request
                 * @param {Object} options Request options
                 * @param {string} options.module Module, e.g. folder, mail, calendar etc.
                 * @param {Object} options.params URL parameters
                 */
                POST: function (options) {
                    ajax(options, "POST");
                },

                /**
                 * Send a PUT request
                 * @param {Object} options Request options
                 * @param {string} options.module Module, e.g. folder, mail, calendar etc.
                 * @param {Object} options.params URL parameters
                 */
                PUT: function (options) {
                    ajax(options, "PUT");
                },
                
                /**
                 * Send a DELETE request
                 * @param {Object} options Request options
                 * @param {string} options.module Module, e.g. folder, mail, calendar etc.
                 * @param {Object} options.params URL parameters
                 */
                DELETE: function (options) {
                    ajax(options, "DELETE");
                }
            };
        }())
    };
    
    // local functions (no bubbling)
    var local = (function () {
        var localFn = {};
        return {
            add: function (fn) {
                localFn[fn] = true;
                return this;
            },
            has: function (fn) {
                return localFn[fn] !== undefined;
            }
        };
    }());
    
    local.add(that.window.getData).
        add(that.getHash).add(that.getParam).add(that.setModal).add(that.http);

    // wrap functions?
    if (that.window.isNested) {

        // wrap!
        var wrap = function (fn, scope, getArray) {
            return function () {
                // get arguments
                var tmp = $.makeArray(arguments), i = 0, $l = tmp.length;
                var args = getArray($l);
                // process arguments
                for (; i < $l; i++) {
                    // clone object except for DOM objects
                    if (!ox.util.isDOMObject(tmp[i])) {
                        args[i] = clone(tmp[i]);
                    }
                }
                // call
                return fn.apply(scope, args);
            };
        };
        var recurse = function (here, there, getArray) {
            // is object?
            if ($.isPlainObject(here) && there !== undefined) {
                // loop over properties
                for (var id in here) {
                    // cannot check there[id] via jQuery's isFunction()!
                    if (!local.has(here[id]) && $.isFunction(here[id]) && there[id] !== undefined) {
                        // replace function with wrapper
                        here[id] = wrap(there[id], there, getArray);
                    } else {
                        recurse(here[id], there[id], getArray);
                    }
                }
            }
        };
        var ref = that.window.core.ox;
        recurse(that, ref.api, ref.util.getArray);
    } 
    
    // done. return
    return that;
    
}());

// API test

/**
 * Exemplary replacement of window factory
 * @name ox.api.window.morph
 * @function
 * @example
 * ox.api.window.morph();
 */
ox.api.window.morph = function () {
    var $ = jQuery;
    var count = 0;
    var open = [];
    var addToList = function (guid) {
        var i = 0, $l = open.length;
        for (; i < $l; i++) {
            ox.api.window.get(open[i]).container.animate({ marginLeft: "-=100px" }, 0);
        }
        open.push(guid);
    };
    var removeFromList = function () {
        open.pop();
        var i = 0, $l = open.length;
        for (; i < $l; i++) {
            ox.api.window.get(open[i]).container.animate({ marginLeft: "+=100px" }, 0);
        }
    };
    ox.api.window.setFactory(
        function (opt) {
            // inc
            count += 2;
            // locals
            var iframe, lightbox, container;
            // create iframe
            iframe = $("<iframe/>", { src: urlify(opt.url), frameborder: "no" }).css({
                position: "absolute",
                border: "0px none",
                width: "100%",
                height: "100%"
            }).one("load", function (e) { // bind might fail, if success handler work with document.write (endless loop in IE)
                // set guid
                var win = this.contentWindow;
                win.guid = opt.guid;
                // register new window object
                ox.api.window.register(opt.guid, win, { lightbox: lightbox, container: container });
                // remove spinner
                lightbox.removeClass("busy-spinner-black");
                // show container
                container.css("visibility", "visible");
                // add to list
                addToList(opt.guid);
                // continue
                opt.success(win, opt.guid);
            });
            // create lightbox
            lightbox = $("<div/>").addClass("busy-spinner-black").css({
                position: "absolute", 
                top: "0px", 
                right: "0px",
                bottom: "0px",
                left: "0px",
                zIndex: 19 + count,
                backgroundColor: "black", 
                opacity: 0.4
            }).bind("click", function (e) {
                // close
                opt.closer(opt);
            }).appendTo(document.body);
            // create container
            var w = opt.options.width;
            container = $("<div/>").css({
                position: "absolute", 
                top: "0px", 
                bottom: "0px", 
                zIndex: 20 + count,
                left: "50%", 
                marginLeft: (-Math.floor(w / 2)) + "px", 
                width: w + "px",
                border: "1px solid #555", 
                borderWidth: "0px 1px 0px 1px", 
                backgroundColor: "white", 
                MozBoxShadow: "0px 5px 25px 5px #000",
                webkitBoxShadow: "0px 5px 25px 5px #000",
                visibility: "hidden"
            }).append(iframe).appendTo(document.body);
        },
        function (obj) {
            // close
            obj.container.remove();
            obj.container = null;
            obj.lightbox.remove();
            obj.lightbox = null;
            // dec
            count -= 2;
            // remove
            removeFromList();
        }
    );
};