package org.json;

/*
 Copyright (c) 2002 JSON.org

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all
 copies or substantial portions of the Software.

 The Software shall be used for Good, not Evil.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 SOFTWARE.
 */

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * A JSONObject is an unordered collection of name/value pairs. Its external
 * form is a string wrapped in curly braces with colons between the names and
 * values, and commas between the values and names. The internal form is an
 * object having <code>get</code> and <code>opt</code> methods for accessing
 * the values by name, and <code>put</code> methods for adding or replacing
 * values by name. The values can be any of these types: <code>Boolean</code>,
 * <code>JSONArray</code>, <code>JSONObject</code>, <code>Number</code>,
 * <code>String</code>, or the <code>JSONObject.NULL</code> object. A
 * JSONObject constructor can be used to convert an external form JSON text into
 * an internal form whose values can be retrieved with the <code>get</code>
 * and <code>opt</code> methods, or to convert values into a JSON text using
 * the <code>put</code> and <code>toString</code> methods. A
 * <code>get</code> method returns a value if one can be found, and throws an
 * exception if one cannot be found. An <code>opt</code> method returns a
 * default value instead of throwing an exception, and so is useful for
 * obtaining optional values.
 * <p>
 * The generic <code>get()</code> and <code>opt()</code> methods return an
 * object, which you can cast or query for type. There are also typed
 * <code>get</code> and <code>opt</code> methods that do type checking and
 * type coersion for you.
 * <p>
 * The <code>put</code> methods adds values to an object. For example,
 * 
 * <pre>
 * myString = new JSONObject().put(&quot;JSON&quot;, &quot;Hello, World!&quot;).toString();
 * </pre>
 * 
 * produces the string <code>{"JSON": "Hello, World"}</code>.
 * <p>
 * The texts produced by the <code>toString</code> methods strictly conform to
 * the JSON sysntax rules. The constructors are more forgiving in the texts they
 * will accept:
 * <ul>
 * <li>An extra <code>,</code>&nbsp;<small>(comma)</small> may appear just
 * before the closing brace.</li>
 * <li>Strings may be quoted with <code>'</code>&nbsp;<small>(single quote)</small>.</li>
 * <li>Strings do not need to be quoted at all if they do not begin with a
 * quote or single quote, and if they do not contain leading or trailing spaces,
 * and if they do not contain any of these characters:
 * <code>{ } [ ] / \ : , = ; #</code> and if they do not look like numbers and
 * if they are not the reserved words <code>true</code>, <code>false</code>,
 * or <code>null</code>.</li>
 * <li>Keys can be followed by <code>=</code> or <code>=></code> as well as
 * by <code>:</code>.</li>
 * <li>Values can be followed by <code>;</code> <small>(semicolon)</small>
 * as well as by <code>,</code> <small>(comma)</small>.</li>
 * <li>Numbers may have the <code>0-</code> <small>(octal)</small> or
 * <code>0x-</code> <small>(hex)</small> prefix.</li>
 * <li>Comments written in the slashshlash, slashstar, and hash conventions
 * will be ignored.</li>
 * </ul>
 * 
 * @author JSON.org
 * @version 2
 */
public class JSONObject implements JSONValue {

	private static final String STR_TRUE = "true";

	private static final String STR_FALSE = "false";

	private static final String STR_NULL = "null";

	/**
	 * JSONObject.NULL is equivalent to the value that JavaScript calls null,
	 * whilst Java's null is equivalent to the value that JavaScript calls
	 * undefined.
	 */
	private static final class Null implements Cloneable {

	    public Null() {
	        super();
	    }

		/**
		 * There is only intended to be a single instance of the NULL object, so
		 * the clone method returns itself.
		 * 
		 * @return NULL.
		 */
		@Override
		protected Object clone() {
			return this;
		}

		/**
		 * A Null object is equal to the null value and to itself.
		 * 
		 * @param object
		 *            An object to test for nullness.
		 * @return true if the object parameter is the JSONObject.NULL object or
		 *         null.
		 */
		@Override
		public boolean equals(final Object object) {
			return object == null || object == this;
		}

		/**
		 * Get the "null" string value.
		 * 
		 * @return The string "null".
		 */
		@Override
		public String toString() {
			return STR_NULL;
		}
	}

	/**
	 * The hash map where the JSONObject's properties are kept.
	 */
	private final Map<String, Object> myHashMap;

	/**
	 * It is sometimes more convenient and less ambiguous to have a
	 * <code>NULL</code> object than to use Java's <code>null</code> value.
	 * <code>JSONObject.NULL.equals(null)</code> returns <code>true</code>.
	 * <code>JSONObject.NULL.toString()</code> returns <code>"null"</code>.
	 */
	public static final Object NULL = new Null();

	/**
	 * Construct an empty JSONObject.
	 */
	public JSONObject() {
		this.myHashMap = new HashMap<String, Object>();
	}

	/**
	 * Construct a JSONObject from a subset of another JSONObject. An array of
	 * strings is used to identify the keys that should be copied. Missing keys
	 * are ignored.
	 * 
	 * @param jo
	 *            A JSONObject.
	 * @param sa
	 *            An array of strings.
	 * @exception JSONException
	 *                If a value is a non-finite number.
	 */
	public JSONObject(final JSONObject jo, final String[] sa) throws JSONException {
		this();
		for (int i = 0; i < sa.length; i += 1) {
			putOpt(sa[i], jo.opt(sa[i]));
		}
	}

	/**
	 * Construct a JSONObject from a JSONTokener.
	 * 
	 * @param x
	 *            A JSONTokener object containing the source string.
	 * @throws JSONException
	 *             If there is a syntax error in the source string.
	 */
	public JSONObject(final JSONTokener x) throws JSONException {
		this();
		parseJSONTokener(x);
	}
	
	/**
	 * Parses a JSONTokener and fills its values into this JSONObject
	 * 
	 * @param tokener
	 *            A JSONTokener object containing the source string
	 * @throws JSONException
	 */
	private final void parseJSONTokener(final JSONTokener tokener) throws JSONException {
		char c;
		String key;
		if (tokener.nextClean() != '{') {
			throw tokener.syntaxError("A JSONObject text must begin with '{'");
		}
		for (;;) {
			c = tokener.nextClean();
			switch (c) {
			case 0:
				throw tokener.syntaxError("A JSONObject text must end with '}'");
			case '}':
				return;
			default:
				tokener.back();
				key = tokener.nextValue().toString();
			}
			/*
			 * The key is followed by ':'. We will also tolerate '=' or '=>'.
			 */
			c = tokener.nextClean();
			if (c == '=') {
				if (tokener.next() != '>') {
					tokener.back();
				}
			} else if (c != ':') {
				throw tokener.syntaxError("Expected a ':' after a key");
			}
			put(key, tokener.nextValue());
			/*
			 * Pairs are separated by ','. We will also tolerate ';'.
			 */
			switch (tokener.nextClean()) {
			case ';':
			case ',':
				if (tokener.nextClean() == '}') {
					return;
				}
				tokener.back();
				break;
			case '}':
				return;
			default:
				throw tokener.syntaxError("Expected a ',' or '}'");
			}
		}
	}

	/**
	 * Construct a JSONObject from a Map.
	 * 
	 * @param map
	 *            A map object that can be used to initialize the contents of
	 *            the JSONObject.
	 */
	public JSONObject(final Map<String, ? extends Object> map) {
		this.myHashMap = (map == null) ? new HashMap<String, Object>() : new HashMap<String, Object>(map);
	}

	/**
	 * Construct a JSONObject from an Object, using reflection to find the
	 * public members. The resulting JSONObject's keys will be the strings from
	 * the names array, and the values will be the field values associated with
	 * those keys in the object. If a key is not found or not visible, then it
	 * will not be copied into the new JSONObject.
	 * 
	 * @param object
	 *            An object that has fields that should be used to make a
	 *            JSONObject.
	 * @param names
	 *            An array of strings, the names of the fields to be used from
	 *            the object.
	 */
	public JSONObject(final Object object, final String names[]) {
		this();
		final Class<?> c = object.getClass();
		for (int i = 0; i < names.length; i += 1) {
			try {
				final String name = names[i];
				final Field field = c.getField(name);
				final Object value = field.get(object);
				this.put(name, value);
			} catch (final Exception e) {
				/* forget about it */
			}
		}
	}

	/**
	 * Construct a JSONObject from a string. This is the most commonly used
	 * JSONObject constructor.
	 * 
	 * @param string
	 *            A string beginning with <code>{</code>&nbsp;<small>(left
	 *            brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *            brace)</small>.
	 * @exception JSONException
	 *                If there is a syntax error in the source string.
	 */
	public JSONObject(final String string) throws JSONException {
		this(new JSONTokener(string));
	}
	
	/**
	 * Fills JSONObject with the given source string. This method is dedicated
	 * for <b>re-using</b> a JSONObject in combination with the <code>reset()</code>
	 * method, since it gives the same possiblity as the common-used
	 * <code>JSONObject(String string)</code> constructor to create a
	 * JSONObject from a string.
	 * 
	 * @param string
	 *            A string beginning with <code>{</code>&nbsp;<small>(left
	 *            brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *            brace)</small>.
	 * @return this.
	 * @exception JSONException
	 *                If there is a syntax error in the source string.
	 * @see #reset()
	 */
	public JSONObject parseJSONString(final String string) throws JSONException {
		parseJSONTokener(new JSONTokener(string));
		return this;
	}

	/**
	 * Accumulate values under a key. It is similar to the put method except
	 * that if there is already an object stored under the key then a JSONArray
	 * is stored under the key to hold all of the accumulated values. If there
	 * is already a JSONArray, then the new value is appended to it. In
	 * contrast, the put method replaces the previous value.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            An object to be accumulated under the key.
	 * @return this.
	 * @throws JSONException
	 *             If the value is an invalid number or if the key is null.
	 */
	public JSONObject accumulate(final String key, final Object value) throws JSONException {
		testValidity(value);
		final Object o = opt(key);
		if (o == null) {
			put(key, value instanceof JSONArray ? new JSONArray().put(value) : value);
		} else if (o instanceof JSONArray) {
			((JSONArray) o).put(value);
		} else {
			put(key, new JSONArray().put(o).put(value));
		}
		return this;
	}

	private static final String STR_ERR01 = "JSONObject[";

	/**
	 * Append values to the array under a key. If the key does not exist in the
	 * JSONObject, then the key is put in the JSONObject with its value being a
	 * JSONArray containing the value parameter. If the key was already
	 * associated with a JSONArray, then the value parameter is appended to it.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            An object to be accumulated under the key.
	 * @return this.
	 * @throws JSONException
	 *             If the key is null or if the current value associated with
	 *             the key is not a JSONArray.
	 */
	public JSONObject append(final String key, final Object value) throws JSONException {
		testValidity(value);
		final Object o = opt(key);
		if (o == null) {
			put(key, new JSONArray().put(value));
		} else if (o instanceof JSONArray) {
			put(key, ((JSONArray) o).put(value));
		} else {
			throw new JSONException(STR_ERR01 + key + "] is not a JSONArray.");
		}
		return this;
	}

	/**
	 * Resets this JSONObject for re-use
	 */
	public void reset() {
		myHashMap.clear();
	}

	/**
	 * Produce a string from a double. The string "null" will be returned if the
	 * number is not finite.
	 * 
	 * @param d
	 *            A double.
	 * @return A String.
	 */
	static public final String doubleToString(final double d) {
		if (Double.isInfinite(d) || Double.isNaN(d)) {
			return STR_NULL;
		}

		// Shave off trailing zeros and decimal point, if possible.

		String s = Double.toString(d);
		if (s.indexOf('.') > 0 && s.indexOf('e') < 0 && s.indexOf('E') < 0) {
			while (s.endsWith("0")) {
				s = s.substring(0, s.length() - 1);
			}
			if (s.endsWith(".")) {
				s = s.substring(0, s.length() - 1);
			}
		}
		return s;
	}

	/**
	 * Get the value object associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return The object associated with the key.
	 * @throws JSONException
	 *             if the key is not found.
	 */
	public Object get(final String key) throws JSONException {
		final Object o = opt(key);
		if (o == null) {
			throw new JSONException(STR_ERR01 + quote(key) + "] not found.");
		}
		return o;
	}

	/**
	 * Get the boolean value associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return The truth.
	 * @throws JSONException
	 *             if the value is not a Boolean or the String "true" or
	 *             "false".
	 */
	public boolean getBoolean(final String key) throws JSONException {
		final Object o = get(key);
		if (o.equals(Boolean.FALSE) || (o instanceof String && ((String) o).equalsIgnoreCase(STR_FALSE))) {
			return false;
		} else if (o.equals(Boolean.TRUE) || (o instanceof String && ((String) o).equalsIgnoreCase(STR_TRUE))) {
			return true;
		}
		throw new JSONException(STR_ERR01 + quote(key) + "] is not a Boolean.");
	}

	/**
	 * Get the double value associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return The numeric value.
	 * @throws JSONException
	 *             if the key is not found or if the value is not a Number
	 *             object and cannot be converted to a number.
	 */
	public double getDouble(final String key) throws JSONException {
		final Object o = get(key);
		try {
			return o instanceof Number ? ((Number) o).doubleValue() : Double.parseDouble((String) o);
		} catch (final Exception e) {
			throw new JSONException(STR_ERR01 + quote(key) + "] is not a number.");
		}
	}

	/**
	 * Get the int value associated with a key. If the number value is too large
	 * for an int, it will be clipped.
	 * 
	 * @param key
	 *            A key string.
	 * @return The integer value.
	 * @throws JSONException
	 *             if the key is not found or if the value cannot be converted
	 *             to an integer.
	 */
	public int getInt(final String key) throws JSONException {
		final Object o = get(key);
		return o instanceof Number ? ((Number) o).intValue() : (int) getDouble(key);
	}

	/**
	 * Get the JSONArray value associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return A JSONArray which is the value.
	 * @throws JSONException
	 *             if the key is not found or if the value is not a JSONArray.
	 */
	public JSONArray getJSONArray(final String key) throws JSONException {
		final Object o = get(key);
		if (o instanceof JSONArray) {
			return (JSONArray) o;
		}
		throw new JSONException(STR_ERR01 + quote(key) + "] is not a JSONArray.");
	}

	/**
	 * Get the JSONObject value associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return A JSONObject which is the value.
	 * @throws JSONException
	 *             if the key is not found or if the value is not a JSONObject.
	 */
	public JSONObject getJSONObject(final String key) throws JSONException {
		final Object o = get(key);
		if (o instanceof JSONObject) {
			return (JSONObject) o;
		}
		throw new JSONException(STR_ERR01 + quote(key) + "] is not a JSONObject.");
	}

	/**
	 * Get the long value associated with a key. If the number value is too long
	 * for a long, it will be clipped.
	 * 
	 * @param key
	 *            A key string.
	 * @return The long value.
	 * @throws JSONException
	 *             if the key is not found or if the value cannot be converted
	 *             to a long.
	 */
	public long getLong(final String key) throws JSONException {
		final Object o = get(key);
		return o instanceof Number ? ((Number) o).longValue() : (long) getDouble(key);
	}

	/**
	 * Get the string associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return A string which is the value.
	 * @throws JSONException
	 *             if the key is not found.
	 */
	public String getString(final String key) throws JSONException {
		return get(key).toString();
	}

	/**
	 * Determine if the JSONObject contains a specific key.
	 * 
	 * @param key
	 *            A key string.
	 * @return true if the key exists in the JSONObject.
	 */
	public boolean has(final String key) {
		return this.myHashMap.containsKey(key);
	}

	/**
	 * Determine if the JSONObject contains a specific key AND if the value
	 * associated with the key is not null.
	 * 
	 * @param key
	 *            A key string.
	 * @return true if the JSONObject contains a specific key AND if the value
	 *         associated with the key is not null.
	 */
	public boolean hasAndNotNull(final String key) {
		return has(key) && !isNull(key);
	}

	/**
	 * Determine if the value associated with the key is null or if there is no
	 * value.
	 * 
	 * @param key
	 *            A key string.
	 * @return true if there is no value associated with the key or if the value
	 *         is the JSONObject.NULL object.
	 */
	public boolean isNull(final String key) {
		return JSONObject.NULL.equals(opt(key));
	}

	/**
	 * Get an enumeration of the keys of the JSONObject.
	 * 
	 * @return An iterator of the keys.
	 */
	public Iterator<String> keys() {
		return this.myHashMap.keySet().iterator();
	}

	/**
	 * Get the key set of the JSONObject.
	 * 
	 * @return A key set
	 */
	public Set<String> keySet() {
		return this.myHashMap.keySet();
	}
	
	/**
	 * Get the entry set of the JSONObject
	 * 
	 * @return A entry set
	 */
	public Set<Map.Entry<String, Object>> entrySet() {
		return this.myHashMap.entrySet();
	}

	/**
	 * Get the number of keys stored in the JSONObject.
	 * 
	 * @return The number of keys in the JSONObject.
	 */
	public int length() {
		return this.myHashMap.size();
	}

	/**
	 * Produce a JSONArray containing the names of the elements of this
	 * JSONObject.
	 * 
	 * @return A JSONArray containing the key strings, or null if the JSONObject
	 *         is empty.
	 */
	public JSONArray names() {
		final JSONArray ja = new JSONArray();
		final Set<String> keys = keySet();
		for (final String name : keys) {
			ja.put(name);
		}
		return ja.length() == 0 ? null : ja;
	}

	/**
	 * Produce a string from a Number.
	 * 
	 * @param n
	 *            A Number
	 * @return A String.
	 * @throws JSONException
	 *             If n is a non-finite number.
	 */
	static public final String numberToString(final Number n) throws JSONException {
		if (n == null) {
			throw new JSONException("Null pointer");
		}
		testValidity(n);

		// Shave off trailing zeros and decimal point, if possible.

		String s = n.toString();
		if (s.indexOf('.') > 0 && s.indexOf('e') < 0 && s.indexOf('E') < 0) {
			while (s.endsWith("0")) {
				s = s.substring(0, s.length() - 1);
			}
			if (s.endsWith(".")) {
				s = s.substring(0, s.length() - 1);
			}
		}
		return s;
	}

	/**
	 * Get an optional value associated with a key.
	 * 
	 * @param key
	 *            A key string.
	 * @return An object which is the value, or null if there is no value.
	 */
	public Object opt(final String key) {
		return key == null ? null : this.myHashMap.get(key);
	}

	/**
	 * Get an optional boolean associated with a key. It returns false if there
	 * is no such key, or if the value is not Boolean.TRUE or the String "true".
	 * 
	 * @param key
	 *            A key string.
	 * @return The truth.
	 */
	public boolean optBoolean(final String key) {
		return optBoolean(key, false);
	}

	/**
	 * Get an optional boolean associated with a key. It returns the
	 * defaultValue if there is no such key, or if it is not a Boolean or the
	 * String "true" or "false" (case insensitive).
	 * 
	 * @param key
	 *            A key string.
	 * @param defaultValue
	 *            The default.
	 * @return The truth.
	 */
	public boolean optBoolean(final String key, final boolean defaultValue) {
		try {
			return getBoolean(key);
		} catch (final Exception e) {
			return defaultValue;
		}
	}

	/**
	 * Put a key/value pair in the JSONObject, where the value will be a
	 * JSONArray which is produced from a Collection.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            A Collection value.
	 * @return this.
	 * @throws JSONException
	 */
	public JSONObject put(final String key, final Collection<? extends Object> value) throws JSONException {
		put(key, new JSONArray(value));
		return this;
	}

	/**
	 * Get an optional double associated with a key, or NaN if there is no such
	 * key or if its value is not a number. If the value is a string, an attempt
	 * will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A string which is the key.
	 * @return An object which is the value.
	 */
	public double optDouble(final String key) {
		return optDouble(key, Double.NaN);
	}

	/**
	 * Get an optional double associated with a key, or the defaultValue if
	 * there is no such key or if its value is not a number. If the value is a
	 * string, an attempt will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A key string.
	 * @param defaultValue
	 *            The default.
	 * @return An object which is the value.
	 */
	public double optDouble(final String key, final double defaultValue) {
		try {
			final Object o = opt(key);
			return o instanceof Number ? ((Number) o).doubleValue() : new Double((String) o).doubleValue();
		} catch (final Exception e) {
			return defaultValue;
		}
	}

	/**
	 * Get an optional int value associated with a key, or zero if there is no
	 * such key or if the value is not a number. If the value is a string, an
	 * attempt will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A key string.
	 * @return An object which is the value.
	 */
	public int optInt(final String key) {
		return optInt(key, 0);
	}

	/**
	 * Get an optional int value associated with a key, or the default if there
	 * is no such key or if the value is not a number. If the value is a string,
	 * an attempt will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A key string.
	 * @param defaultValue
	 *            The default.
	 * @return An object which is the value.
	 */
	public int optInt(final String key, final int defaultValue) {
		try {
			return getInt(key);
		} catch (final Exception e) {
			return defaultValue;
		}
	}

	/**
	 * Get an optional JSONArray associated with a key. It returns null if there
	 * is no such key, or if its value is not a JSONArray.
	 * 
	 * @param key
	 *            A key string.
	 * @return A JSONArray which is the value.
	 */
	public JSONArray optJSONArray(final String key) {
		final Object o = opt(key);
		return o instanceof JSONArray ? (JSONArray) o : null;
	}

	/**
	 * Get an optional JSONObject associated with a key. It returns null if
	 * there is no such key, or if its value is not a JSONObject.
	 * 
	 * @param key
	 *            A key string.
	 * @return A JSONObject which is the value.
	 */
	public JSONObject optJSONObject(final String key) {
		final Object o = opt(key);
		return o instanceof JSONObject ? (JSONObject) o : null;
	}

	/**
	 * Get an optional long value associated with a key, or zero if there is no
	 * such key or if the value is not a number. If the value is a string, an
	 * attempt will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A key string.
	 * @return An object which is the value.
	 */
	public long optLong(final String key) {
		return optLong(key, 0);
	}

	/**
	 * Get an optional long value associated with a key, or the default if there
	 * is no such key or if the value is not a number. If the value is a string,
	 * an attempt will be made to evaluate it as a number.
	 * 
	 * @param key
	 *            A key string.
	 * @param defaultValue
	 *            The default.
	 * @return An object which is the value.
	 */
	public long optLong(final String key, final long defaultValue) {
		try {
			return getLong(key);
		} catch (final Exception e) {
			return defaultValue;
		}
	}

	/**
	 * Get an optional string associated with a key. It returns an empty string
	 * if there is no such key. If the value is not a string and is not null,
	 * then it is coverted to a string.
	 * 
	 * @param key
	 *            A key string.
	 * @return A string which is the value.
	 */
	public String optString(final String key) {
		return optString(key, "");
	}

	/**
	 * Get an optional string associated with a key. It returns the defaultValue
	 * if there is no such key.
	 * 
	 * @param key
	 *            A key string.
	 * @param defaultValue
	 *            The default.
	 * @return A string which is the value.
	 */
	public String optString(final String key, final String defaultValue) {
		final Object o = opt(key);
		return o != null ? o.toString() : defaultValue;
	}

	/**
	 * Put a key/boolean pair in the JSONObject.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            A boolean which is the value.
	 * @return this.
	 * @throws JSONException
	 *             If the key is null.
	 */
	public JSONObject put(final String key, final boolean value) throws JSONException {
		put(key, value ? Boolean.TRUE : Boolean.FALSE);
		return this;
	}

	/**
	 * Put a key/double pair in the JSONObject.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            A double which is the value.
	 * @return this.
	 * @throws JSONException
	 *             If the key is null or if the number is invalid.
	 */
	public JSONObject put(final String key, final double value) throws JSONException {
		put(key, Double.valueOf(value));
		return this;
	}

	/**
	 * Put a key/int pair in the JSONObject.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            An int which is the value.
	 * @return this.
	 * @throws JSONException
	 *             If the key is null.
	 */
	public JSONObject put(final String key, final int value) throws JSONException {
		put(key, Integer.valueOf(value));
		return this;
	}

	/**
	 * Put a key/long pair in the JSONObject.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            A long which is the value.
	 * @return this.
	 * @throws JSONException
	 *             If the key is null.
	 */
	public JSONObject put(final String key, final long value) throws JSONException {
		put(key, Long.valueOf(value));
		return this;
	}

	/**
	 * Put a key/value pair in the JSONObject, where the value will be a
	 * JSONObject which is produced from a Map.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            A Map value.
	 * @return this.
	 * @throws JSONException
	 */
	public JSONObject put(final String key, final Map<String, ? extends Object> value) throws JSONException {
		put(key, new JSONObject(value));
		return this;
	}

	/**
	 * Put a key/value pair in the JSONObject. If the value is null, then the
	 * key will be removed from the JSONObject if it is present.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            An object which is the value. It should be of one of these
	 *            types: Boolean, Double, Integer, JSONArray, JSONObject, Long,
	 *            String, or the JSONObject.NULL object.
	 * @return this.
	 * @throws JSONException
	 *             If the value is non-finite number or if the key is null.
	 */
	public JSONObject put(final String key, final Object value) throws JSONException {
		if (key == null) {
			throw new JSONException("Null key.");
		}
		if (value != null) {
			testValidity(value);
			this.myHashMap.put(key, value);
		} else {
			remove(key);
		}
		return this;
	}

	/**
	 * Put a key/value pair in the JSONObject, but only if the key and the value
	 * are both non-null.
	 * 
	 * @param key
	 *            A key string.
	 * @param value
	 *            An object which is the value. It should be of one of these
	 *            types: Boolean, Double, Integer, JSONArray, JSONObject, Long,
	 *            String, or the JSONObject.NULL object.
	 * @return this.
	 * @throws JSONException
	 *             If the value is a non-finite number.
	 */
	public JSONObject putOpt(final String key, final Object value) throws JSONException {
		if (key != null && value != null) {
			put(key, value);
		}
		return this;
	}

	/**
	 * Produce a string in double quotes with backslash sequences in all the
	 * right places. A backslash will be inserted within </, allowing JSON text
	 * to be delivered in HTML. In JSON text, a string cannot contain a control
	 * character or an unescaped quote or backslash.
	 * 
	 * @param string
	 *            A String
	 * @return A String correctly formatted for insertion in a JSON text.
	 */
	public static final String quote(final String string) {
		if (string == null || string.length() == 0) {
			return "\"\"";
		}

		char b;
		char c = 0;
		int i;
		final int len = string.length();
		final StringBuilder sb = new StringBuilder(len + 4);
		String t;

		sb.append('"');
		for (i = 0; i < len; i += 1) {
			b = c;
			c = string.charAt(i);
			switch (c) {
			case '\\':
			case '"':
				sb.append('\\');
				sb.append(c);
				break;
			case '/':
				if (b == '<') {
					sb.append('\\');
				}
				sb.append(c);
				break;
			case '\b':
				sb.append("\\b");
				break;
			case '\t':
				sb.append("\\t");
				break;
			case '\n':
				sb.append("\\n");
				break;
			case '\f':
				sb.append("\\f");
				break;
			case '\r':
				sb.append("\\r");
				break;
			default:
				if (c < ' ' || (c >= '\u0080' && c < '\u00a0') || (c >= '\u2000' && c < '\u2100')) {
					t = "000" + Integer.toHexString(c);
					sb.append("\\u").append(t.substring(t.length() - 4));
				} else {
					sb.append(c);
				}
			}
		}
		sb.append('"');
		return sb.toString();
	}

	/**
	 * Remove a name and its value, if present.
	 * 
	 * @param key
	 *            The name to be removed.
	 * @return The value that was associated with the name, or null if there was
	 *         no value.
	 */
	public Object remove(final String key) {
		return this.myHashMap.remove(key);
	}

	/**
	 * Throw an exception if the object is an NaN or infinite number.
	 * 
	 * @param o
	 *            The object to test.
	 * @throws JSONException
	 *             If o is a non-finite number.
	 */
	static final void testValidity(final Object o) throws JSONException {
        if (o != null) {
            if (o instanceof Double) {
                final Double d = (Double) o;
                if (d.isInfinite() || d.isNaN()) {
                    throw new JSONException("JSON does not allow non-finite numbers.");
                }
            } else if (o instanceof Float) {
                final Float f = (Float) o;
                if (f.isInfinite() || f.isNaN()) {
                    throw new JSONException("JSON does not allow non-finite numbers.");
                }
            }
        }
    }

	/**
	 * Produce a JSONArray containing the values of the members of this
	 * JSONObject.
	 * 
	 * @param names
	 *            A JSONArray containing a list of key strings. This determines
	 *            the sequence of the values in the result.
	 * @return A JSONArray of values.
	 * @throws JSONException
	 *             If any of the values are non-finite numbers.
	 */
	public JSONArray toJSONArray(final JSONArray names) throws JSONException {
		if (names == null || names.length() == 0) {
			return null;
		}
		final JSONArray ja = new JSONArray();
		for (int i = 0; i < names.length(); i += 1) {
			ja.put(this.opt(names.getString(i)));
		}
		return ja;
	}

	/**
	 * Make a JSON text of this JSONObject. For compactness, no whitespace is
	 * added. If this would not result in a syntactically correct JSON text,
	 * then null will be returned instead.
	 * <p>
	 * Warning: This method assumes that the data structure is acyclical.
	 * 
	 * @return a printable, displayable, portable, transmittable representation
	 *         of the object, beginning with <code>{</code>&nbsp;<small>(left
	 *         brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *         brace)</small>.
	 */
	@Override
	public String toString() {
        try {
            final int n = length();
            if (n <= 0) {
                return "{}";
            }

            final StringBuilder sb = new StringBuilder(n * 16).append('{');
            final Iterator<Map.Entry<String, Object>> iter = myHashMap.entrySet().iterator();
            {
                final Map.Entry<String, Object> first = iter.next();
                sb.append(quote(first.getKey()));
                sb.append(':');
                sb.append(valueToString(first.getValue()));
            }
            for (int i = 1; i < n; i++) {
                final Map.Entry<String, Object> next = iter.next();
                sb.append(',');
                sb.append(quote(next.getKey()));
                sb.append(':');
                sb.append(valueToString(next.getValue()));
            }
            sb.append('}');
            return sb.toString();
        } catch (final Exception e) {
            return null;
        }
    }

	/**
	 * Make a prettyprinted JSON text of this JSONObject.
	 * <p>
	 * Warning: This method assumes that the data structure is acyclical.
	 * 
	 * @param indentFactor
	 *            The number of spaces to add to each level of indentation.
	 * @return a printable, displayable, portable, transmittable representation
	 *         of the object, beginning with <code>{</code>&nbsp;<small>(left
	 *         brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *         brace)</small>.
	 * @throws JSONException
	 *             If the object contains an invalid number.
	 */
	public String toString(final int indentFactor) throws JSONException {
		return toString(indentFactor, 0);
	}

	/**
	 * Make a prettyprinted JSON text of this JSONObject.
	 * <p>
	 * Warning: This method assumes that the data structure is acyclical.
	 * 
	 * @param indentFactor
	 *            The number of spaces to add to each level of indentation.
	 * @param indent
	 *            The indentation of the top level.
	 * @return a printable, displayable, transmittable representation of the
	 *         object, beginning with <code>{</code>&nbsp;<small>(left
	 *         brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *         brace)</small>.
	 * @throws JSONException
	 *             If the object contains an invalid number.
	 */
	public String toString(final int indentFactor, final int indent) throws JSONException {
		final int n = length();
		if (n == 0) {
			return "{}";
		}
		final StringBuilder sb = new StringBuilder().append('{');
		final int newindent = indent + indentFactor;
		if (n == 1) {
			final Iterator<String> keys = keys();
			final String o = keys.next();
			sb.append(quote(o));
			sb.append(": ");
			sb.append(valueToString(this.myHashMap.get(o), indentFactor, indent));
		} else {
			final Set<String> keys = keySet();
			for (final String o : keys) {
				if (sb.length() > 1) {
					sb.append(",\n");
				} else {
					sb.append('\n');
				}
				for (int i = 0; i < newindent; i += 1) {
					sb.append(' ');
				}
				sb.append(quote(o));
				sb.append(": ");
				sb.append(valueToString(this.myHashMap.get(o), indentFactor, newindent));
			}
			if (sb.length() > 1) {
				sb.append('\n');
				for (int i = 0; i < indent; i += 1) {
					sb.append(' ');
				}
			}
		}
		sb.append('}');
		return sb.toString();
	}

	/**
	 * Make a JSON text of an Object value. If the object has an
	 * value.toJSONString() method, then that method will be used to produce the
	 * JSON text. The method is required to produce a strictly conforming text.
	 * If the object does not contain a toJSONString method (which is the most
	 * common case), then a text will be produced by the rules.
	 * <p>
	 * Warning: This method assumes that the data structure is acyclical.
	 * 
	 * @param value
	 *            The value to be serialized.
	 * @return a printable, displayable, transmittable representation of the
	 *         object, beginning with <code>{</code>&nbsp;<small>(left
	 *         brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *         brace)</small>.
	 * @throws JSONException
	 *             If the value is or contains an invalid number.
	 */
	static final String valueToString(final Object value) throws JSONException {
		if (value == null || NULL.equals(value)) {
			return STR_NULL;
		}
		if (value instanceof JSONString) {
			Object o;
			try {
				o = ((JSONString) value).toJSONString();
			} catch (final Exception e) {
				throw new JSONException(e);
			}
			if (o instanceof String) {
				return (String) o;
			}
			throw new JSONException("Bad value from toJSONString: " + o);
		}
		if (value instanceof Number) {
			return numberToString((Number) value);
		}
		if (value instanceof Boolean || value instanceof JSONValue) {
			return value.toString();
		}
		return quote(value.toString());
	}

	/**
	 * Make a prettyprinted JSON text of an object value.
	 * <p>
	 * Warning: This method assumes that the data structure is acyclical.
	 * 
	 * @param value
	 *            The value to be serialized.
	 * @param indentFactor
	 *            The number of spaces to add to each level of indentation.
	 * @param indent
	 *            The indentation of the top level.
	 * @return a printable, displayable, transmittable representation of the
	 *         object, beginning with <code>{</code>&nbsp;<small>(left
	 *         brace)</small> and ending with <code>}</code>&nbsp;<small>(right
	 *         brace)</small>.
	 * @throws JSONException
	 *             If the object contains an invalid number.
	 */
	static final String valueToString(final Object value, final int indentFactor, final int indent)
			throws JSONException {
		if (value == null || NULL.equals(value)) {
			return STR_NULL;
		}
		try {
			if (value instanceof JSONString) {
				final Object o = ((JSONString) value).toJSONString();
				if (o instanceof String) {
					return (String) o;
				}
			}
		} catch (final Exception e) {
			/* forget about it */
		}
		if (value instanceof Number) {
			return numberToString((Number) value);
		}
		if (value instanceof Boolean) {
			return value.toString();
		}
		if (value instanceof JSONValue) {
		    return ((JSONValue) value).toString(indentFactor, indent);
		}
		return quote(value.toString());
	}

    /**
     * {@inheritDoc}
     */
    public Writer write(final Writer writer) throws JSONException {
        try {
            writer.write('{');

            final int len = myHashMap.size();
            if (len > 0) {
                final Iterator<Map.Entry<String, Object>> iter = myHashMap.entrySet().iterator();
                writeValue(writer, iter.next());
                for (int i = 1; i < len; i++) {
                    writer.write(',');
                    writeValue(writer, iter.next());
                }
            }

            writer.write('}');
            return writer;
        } catch (final IOException e) {
            throw new JSONException(e);
        }
    }

    private static void writeValue(final Writer writer, final Map.Entry<String, Object> entry) throws IOException, JSONException {
        writer.write(quote(entry.getKey()));
        writer.write(':');
        final Object v = entry.getValue();
        if (v instanceof JSONValue) {
            ((JSONValue) v).write(writer);
        } else {
            writer.write(valueToString(v));
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean isArray() {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isObject() {
        return true;
    }
}
